# coding: UTF-8
require File.expand_path(File.dirname(__FILE__) + '../../../spec_helper')
require File.expand_path(File.dirname(__FILE__) + '/monster_spec_board')

describe RDGC::Monster::MonsterActionBase, "is base of monster's action logic" do

  before(:all) do
    @board = create_board
  end

  before do
    @monster = Character::Base.new
    @monster.extend(Monster::MonsterActionBase)

    class << @monster

      attr_accessor :view_range

    end

    @player = Character::Player.new
    @floor = Map::Floor.create_from_board(@board)
  end

  it "#movable_to? will check move point is movable tile" do
    @floor.add_article(@monster)
    @monster.movable_to?(nil).should be_false

    @monster.set(4, 4)
    Map::Direction.each.all?{|d| @monster.movable_to?(d)}.should be_true

    @monster.set(2, 2)
    @monster.movable_to?(Map::Direction::LEFT).should be_false
    @monster.movable_to?(Map::Direction::UPPER).should be_false
    @monster.movable_to?(Map::Direction::RIGHT).should be_true
    @monster.movable_to?(Map::Direction::BOTTOM).should be_true

    @monster.set(7, 7)
    @monster.movable_to?(Map::Direction::LEFT).should be_true
    @monster.movable_to?(Map::Direction::UPPER).should be_true
    @monster.movable_to?(Map::Direction::RIGHT).should be_false
    @monster.movable_to?(Map::Direction::BOTTOM).should be_false
  end

  it "#chase_miss_count provides target miss count" do
    @monster.chase_miss_count.should == 0
    @monster.add_miss_count
    @monster.chase_miss_count.should == 1
    9.times{@monster.add_miss_count}
    @monster.chase_miss_count.should == 10
    @monster.clear_miss_count
    @monster.chase_miss_count.should == 0
  end

  it "#random_move will move by random at movable directions, or not move" do
    @floor.add_article(@monster)

    dirs1 = [Map::Direction.all, Map::Direction::SELF].flatten
    random_move_test(4, 4, dirs1)

    dirs2 = [Map::Direction::RIGHT, Map::Direction::BOTTOM, Map::Direction::SELF]
    random_move_test(2, 2, dirs2)

    dirs3 = [Map::Direction::LEFT, Map::Direction::UPPER, Map::Direction::SELF]
    random_move_test(7, 7, dirs3)
  end

  it "#select_target will select interesting article, default target is Player" do
    list = [@monster, Article::Step.create_up_step(5, 5)]
    @monster.select_target(list).should be_nil

    list.unshift(@player)
    @monster.select_target(list).should == @player
  end

  it "#action will return action's result, not nil" do

    class << @monster

      def test_mode=(mode)
        @test_mode = mode
      end

      # override
      def monster_action
        @test_mode
      end

    end

    @monster.action.should == :monster_action
    @monster.test_mode = :hoge
    @monster.action.should == :monster_hoge
  end

  def random_move_test(sx, sy, dirs)
    100.times do
      @monster.set(sx, sy)
      @monster.random_move.should == :moved
      dirs.map{|d| @monster.exist?(sx + d.x, sy + d.y)}.any?.should be_true
    end
  end

  describe "#chase_move will move for chase one target" do

    before do

      class << @player

        def result
          @result
        end

        def crear_result
          @result = nil
        end

        # override
        def reaction_to(a)
          @result = "attack!"
        end

      end

      @floor.add_article(@monster)
      @floor.add_article(@player)
    end

    it "simple chase" do
      mx = 5
      my = 5
      @monster.view_range = 3
      @player.set(6, 4)

      dirs = [Map::Direction::RIGHT, Map::Direction::UPPER]
      100.times do
        @monster.set(mx, my)
        @monster.chase_move.should == :moved
        dirs.map{|d| @monster.exist?(mx + d.x, my + d.y)}.any?.should be_true
        @monster.chase_miss_count.should == 0
      end

      @player.set(5, 4)
      100.times do
        @player.crear_result
        @monster.set(mx, my)
        @monster.chase_move.should == :not_move
        @monster.exist?(5, 5).should be_true
        @monster.chase_miss_count.should == 0
        @player.result.should == "attack!"
      end
    end

    it "if not movable, select movable direction" do
      @floor.movable?(8, 5).should be_true
      @floor.movable?(9, 5).should be_true
      @floor.movable?(9, 6).should be_true
      @floor.movable?(8, 6).should be_false

      mx = 9
      my = 6
      @monster.view_range = 3
      @player.set(8, 5)

      100.times do
        @monster.set(mx, my)
        @monster.chase_move.should == :moved
        @monster.exist?(9, 5).should be_true
        @monster.chase_miss_count.should == 0
      end
    end

    it "if other monstars around, select movable direction" do
      mx = 5
      my = 5
      @monster.view_range = 3
      @player.set(6, 4)

      mons1 = create_monster(5, 4)
      mons2 = create_monster(7, 5)

      100.times do
        @monster.set(mx, my)
        @monster.chase_move.should == :moved
        @monster.exist?(6, 5).should be_true
        @monster.chase_miss_count.should == 0
      end
    end

    it "if no target, random_move" do
      mx = 5
      my = 5
      @monster.view_range = 2
      @player.set(5, 2)

      mons1 = create_monster(6, 4)
      mons2 = create_monster(7, 5)

      dirs = [Map::Direction.all, Map::Direction::SELF].flatten
      100.times do
        @monster.clear_miss_count
        @monster.set(mx, my)
        @monster.chase_move.should == :moved
        dirs.map{|d| @monster.exist?(mx + d.x, my + d.y)}.any?.should be_true
        @monster.chase_miss_count.should == 1
      end
    end

    it "if target unreachable, random_move" do
      @floor.movable?(7, 6).should be_true
      @floor.movable?(8, 6).should be_false
      @floor.movable?(9, 6).should be_true

      mx = 9
      my = 6
      @monster.view_range = 5
      @player.set(7, 6)

      dirs = [Map::Direction.all, Map::Direction::SELF].flatten.reject{|d| d == Map::Direction::LEFT}
      100.times do
        @monster.clear_miss_count
        @monster.set(mx, my)
        @monster.chase_move.should == :moved
        dirs.map{|d| @monster.exist?(mx + d.x, my + d.y)}.any?.should be_true
        @monster.chase_miss_count.should == 1
      end
    end

    def create_monster(mx, my)
      mons = Character::Base.new
      mons.set(mx, my)
      @floor.add_article(mons)
      mons
    end

  end

end
