# -*- coding: UTF-8 -*-
module RDGC
  module Map
    class Board
      include TiledArea

      def self.create(width, height, visible_level = nil)
        d = self.new
        d.left = 0
        d.right = width - 1
        d.top = 0
        d.bottom = height - 1
        d.visible_level = visible_level
        d.make
        d
      end

      def tile(x, y)
        return TileType::OUT unless has_xy?(x, y)

        tile_data[x][y] ||= TileType::WALL
        tile_data[x][y]
      end

      def make
        @blocks = Block.create_blocks(top, bottom, left, right)

        # 部屋作成
        blocks.each{|b| b.create_room}

        # 道作成
        create_road

        # 道がつながらなかった区画の部屋を消す
        room_blocks.reject{|b| b.has_road?}.each{|b| b.remove_room}

        # Mapを埋める
        fill_map

        # visibleセット
        fill_visible

        self
      end

      def fill_map
        fill(roads)
        fill(rooms)
      end

      def movable?(x, y)
        tile(x, y).movable?
      end

      def room?(x, y)
        return unless has_xy?(x, y)
        tile(x, y).room?
      end

      def road?(x, y)
        return unless has_xy?(x, y)
        tile(x, y).road?
      end

      def start_point
        room_blocks.find{|b| b.start?}.room.random_point
      end

      def random_point(include_start = true)
        if include_start
          room_blocks.choice.room.random_point
        else
          room_blocks.reject{|b| b.start?}.choice.room.random_point
        end
      end

      def visible_level=(level)
        @visible_level = level
      end

      def visible_level
        @visible_level ||= :normal
        @visible_level
      end

      def visible?(x, y)
        return false  unless has_xy?(x, y)
        visible_data[x][y]
      end

      def to_visible(x, y, range = 1)
        case visible_level
        when :bright
          return
        when :dark
          to_visible_for_range(x, y, range)
        else
          to_visible_for_floor(x, y, range)
        end
      end

      def room_count
        rooms.size
      end

      private

      def blocks
        @blocks ||= []
        @blocks
      end

      def room_blocks
        return [] if blocks.empty?
        blocks.select{|b| b.has_room?}
      end

      def rooms
        room_blocks.map(&:room)
      end

      def roads
        blocks.map(&:roads).flatten
      end

      def create_road(target = nil)
        # 全部道がつながったら終了
        return if room_blocks.all?{|b| b.road_created?}
        return if room_blocks.all?{|b| b.has_road?}

        # 始点を決める
        target ||= room_blocks.choice.set_start

        top_list = collect_cling_block(target, :top)
        bottom_list = collect_cling_block(target, :bottom)
        left_list = collect_cling_block(target, :left)
        right_list = collect_cling_block(target, :right)
        cling_list = [top_list, bottom_list, left_list, right_list].select{|a| a.length > 0}

        # 行き止まり => 終了
        return if cling_list.length <= 0

        # 4方向で選択可能なblock配列から一つ選ぶ
        direction_list = cling_list.shuffle.shift
        cling_block = direction_list.choice

        # ブロックに道を書き、つなぐRoadをセット
        target.create_road_to(cling_block)

        # 残りの方向もランダムにつなぐ
        cling_list.each do |d_list|
          target.create_road_to(d_list.choice) if bool_rand
        end

        # 作成完了
        target.set_road_created

        # 次を再帰的呼び出し
        create_road(cling_block)
      end

      def collect_cling_block(block, direction)
        yet_block = room_blocks.reject{|b| b.road_created?}

        case direction
        when :top
          yet_block.select{|b| block.cling_to_top?(b)}
        when :bottom
          yet_block.select{|b| block.cling_to_bottom?(b)}
        when :left
          yet_block.select{|b| block.cling_to_left?(b)}
        when :right
          yet_block.select{|b| block.cling_to_right?(b)}
        end
      end

      def fill(list)
        list.each do |o|
          o.each_tile do |x, y, t|
            next unless t
            next if t.wall?
            set_tile(x, y, t)
          end
        end
      end

      def visible_data
        @visible_data ||= Hash.new{|hash, key| hash[key] = {}}
        @visible_data
      end

      def fill_visible
        case visible_level
        when :bright
          set_visible(rooms, true)
          set_visible(roads, true)
        when :dark
          set_visible(rooms, false)
          set_visible(roads, false)
        else
          set_visible(rooms, false)
          set_visible(roads, false)
        end
      end

      def set_visible(list, v)
        list.each do |r|
          r.each do |x, y|
            visible_data[x][y] = v
          end
        end
      end

      def to_visible_for_floor(x, y, range)
        to_visible_for_range(x, y, range)

        if room?(x, y)
          target = rooms.find{|r| r.has_xy?(x, y)}
          set_visible([target], true)
        end
      end

      def to_visible_for_range(sx, sy, range, r=0)
        return unless movable?(sx, sy)
        return if r > range

        # 今のポイントを可視に
        visible_data[sx][sy] = true

        # 再帰的に処理
        Direction.each do |dir|
          nx, ny = dir.apply_to(sx, sy)
          to_visible_for_range(nx, ny, range, r+1)
        end
      end

    end
  end
end
