# -*- coding: UTF-8 -*-
module RDGC
  module Dungeon
    class Dungeon
      include Timer::TimerOwner

      attr_reader :player, :floor, :board, :step_up, :step_down

      def self.create(player, floor, params = nil)
        raise 'player unknown' unless player

        params ||= {}
        params = default_params.merge(params)

        _width = params[:width]
        _width = range_rand(20, 40 + floor)

        _height = params[:height]
        _height = range_rand(20, 40 + floor)

        _upstep = params[:upstep]
        _downstep = params[:downstep]
        _init_monster = params[:init_monster]
        _monster_pop_time = params[:monster_pop_time]
        _monster_pop_timer = params[:monster_pop_timer]

        if _monster_pop_time > 0
          _monster_pop_timer ||= Timer::MonsterPopTimer.create(_monster_pop_time)
        end

        _monster_max_ratio = params[:monster_max_ratio]
        _monster_max = params[:monster_max]

        _visible_level = params[:visible_level]
        _monster_factory = params[:monster_factory]

        obj = self.new
        obj.instance_eval do
          # パラメータセット
          @floor = floor
          @params = params

          _visible_level ||= default_visible_level
          _monster_factory ||= default_monster_factory

          # 基本座標系取得
          @board = create_board(_width, _height, _visible_level)

          # 上り階段＝スタート地点設置
          x, y = @board.start_point
          @step_up = Map::Step.create(x, y) if _upstep

          player.x = x
          player.y = y
          @player = player
          @board.to_visible(x, y, player.view_range)

          # 下り階段設置
          if _downstep
            x, y = @board.random_point(false)
            @step_down = Map::Step.create(x, y)
          end

          # モンスター生成Factory
          @monster_factory = _monster_factory

          # モンスター最大値
          @monster_max_ratio = _monster_max_ratio
          @monster_max = _monster_max

          # モンスター配置
          init_monster if _init_monster

          # popタイマーセット
          set_timer(_monster_pop_timer) if _monster_pop_timer
        end

        obj
      end

      def self.default_params
        {
          :width => nil,
          :height => nil,
          :visible_level => nil,
          :downstep => true,
          :upstep => true,
          :monster_factory => nil,
          :init_monster => true,
          :monster_pop_timer => nil,
          :monster_pop_time => ACT_MAX_COUNT * 7,
          :monster_max_ratio => nil,
          :monster_max => nil
        }
      end

      def params
        @params
      end

      def movable?(x, y)
        board.movable?(x, y)
      end

      def update
        return if player.disable?

        [monsters].each do |list|
          list.delete_if{|a| a.disable?}
        end
      end

      def articles(x, y)
        return [] unless board.has_xy?(x, y)
        actors.flatten.select{|a| a.exist?(x, y)}
      end

      def random_point(include_start = true)
        @board.random_point(include_start)
      end

      def monster_max
        @monster_max_ratio ||= 2.0
        @monster_max ||= (@board.room_count * @monster_max_ratio).ceil
        @monster_max
      end

      def create_monster
        return unless @monster_factory
        return if monsters.size >= monster_max

        x, y = random_point
        10.times do
          # プレイヤー周辺には配置しない
          next if near_player?(x, y)
          break if articles(x, y).empty?
          x, y = random_point
        end
        return if near_player?(x, y)
        return unless articles(x, y).empty?

        m = @monster_factory.create(x, y)
        add_monster(m)
      end

      def near_player?(x, y)
        return false if x < player.x-2
        return false if x > player.x+2
        return false if y < player.y-2
        return false if y > player.y+2
        true
      end

      def monsters
        @monsters ||= []
        @monsters
      end

      def add_monster(m)
        monsters << m
      end

      def player_in_step_up?
        return unless @step_up
        @player.collision?(@step_up)
      end

      def player_in_step_down?
        return unless @step_down
        @player.collision?(@step_down)
      end

      def actors
        [player, monsters].flatten
      end

      def proceed
        # プレイヤーが死亡していれば何もしない
        return :player_dead if player.disable?

        # カウントを進める
        add_count = actors.min_by(&:next_act_count).next_act_count
        if add_count > 0
          actors.each do |a|
            a.update_time(add_count)
            a.update_timers(add_count)
          end
          self.update_timers(add_count)
        end

        # 行動可能なものを分離
        acts, yets = actors.partition(&:act_ok?)

        # プレイヤー優先
        actor = acts.find{|a| a.kind_of?(Actor::Character::Player)}
        if actor
          acts.delete(actor)
        else
          actor = acts.sort_by(&:now_time).shift
        end
        return unless actor

        # アクション実行
        rsl = actor.action(self)

        # 有意な動作をしたら時間を薦める
        if rsl
          # タイマー発動
          actor.effect_timer
          self.effect_timer

          actor.reset_time
          [acts, yets].flatten.each{|a| a.update_time}
          self.update
        end

        # プレイヤーが死亡したか？
        return :player_dead if player.disable?

        # アクション結果返却
        rsl
      end

      private

      def init_monster
        # モンスター配置数
        rc = @board.room_count
        set = range_rand((rc*0.5).ceil, (rc*1.0).ceil)

        # モンスターを作る処理
        set.times{|i| create_monster}
      end

      def create_board(width, height, visible = nil)
        Map::Board.create(width, height, visible)
      end

      def default_monster_factory
      end

      def default_visible_level
        select_rand(:normal => 8, :bright => 1, :dark => 1)
      end

    end
  end
end
