
#ifndef GEOMETRY_H
#define GEOMETRY_H

#include <math.h>
#include <functional>

class AngleDeg {
private:
    double M_degree; // This is Degree, NOT Radian!!

public:
    static const double DEG2RAD;
    static const double RAD2DEG;

    AngleDeg()
        : M_degree(0.0)
      { }
    AngleDeg(const double &deg)
        : M_degree(deg)
      {
          normalize();
      }

    inline
    const
    AngleDeg & operator =(const double &d)
      {
          M_degree = d;
          normalize();
          return *this;
      }
private:
    inline
    void normalize()
      {
          while ( M_degree < -180.0 )
          {
              M_degree += 360.0;
          }
          while ( M_degree > 180.0 )
          {
              M_degree -= 360.0;
          }
      }

public:
    inline
    const
    double & degree() const
      {
          return M_degree;
      }
    inline
    double abs() const
      {
          return fabs(degree());
      }
    inline
    double radian() const
      {
          return degree() * DEG2RAD;
      }

    /*
    operator double() const
      {
          return degree();
      }
    */

    inline
    AngleDeg operator -() const
      {
          return AngleDeg(-M_degree);
      }

    inline
    const
    AngleDeg & operator +=(const AngleDeg &a)
      {
          M_degree += a.degree();
          normalize();
          return *this;
      }
    inline
    const
    AngleDeg & operator +=(const double &d)
      {
          M_degree += d;
          normalize();
          return *this;
      }

    inline
    const
    AngleDeg & operator -=(const AngleDeg &a)
      {
          M_degree -= a.degree();
          normalize();
          return *this;
      }
    inline
    const
    AngleDeg & operator -=(const double &d)
      {
          M_degree -= d;
          normalize();
          return *this;
      }

    inline
    const
    AngleDeg & operator *=(const double &d)
      {
          M_degree *= d;
          normalize();
          return *this;
      }

    inline
    const
    AngleDeg & operator /=(const double &d)
      {
          M_degree /= d;
          normalize();
          return *this;
      }

    //////////////////////////////////////////////////////////////////

    // these operators must be defined as member method

    inline
    bool isLeftOf(const AngleDeg &a) const // consider only (-180, 180)
      {
          //return (*this - a).degree() < 0.0;
          double diff = a.degree() - M_degree;
          return ( ( 0.0 < diff && diff < 180.0 )
                   || diff < -180.0 );
      }
    inline
    bool isLeftEqualOf(const AngleDeg &a) const // consider only [-180, 180]
      {
          //return (*this - a).degree() <= 0.0;
          double diff = a.degree() - M_degree;
          return ( ( 0.0 <= diff && diff < 180.0 )
                   || diff < -180.0 );
      }
    inline
    bool isRightOf(const AngleDeg &a) const  // consider only (-180, 180)
      {
          //return (*this - a).degree() > 0.0;
          double diff = M_degree - a.degree();
          return ( ( 0.0 < diff && diff < 180.0 )
                   || diff < -180.0 );
      }
    inline
    bool isRightEqualOf(const AngleDeg &a) const  // consider only [-180, 180]
      {
          //return (*this - a).degree() >= 0.0;
          double diff = M_degree - a.degree();
          return ( ( 0.0 <= diff && diff < 180.0 )
                   || diff < -180.0 );
      }

    //////////////////////////////////////////////////////////////////

    inline
    double Cos() const
      {
          return cos(degree() * DEG2RAD);
      }
    inline
    double Sin() const
      {
          return sin(degree() * DEG2RAD);
      }
    inline
    double Tan() const
      {
          return tan(degree() * DEG2RAD);
      }

    //------------------------------------------------------------------
    // static utility method concerned with angle operation

    inline
    static
    double normalize_angle(double dir)
      {
          while ( dir < -180.0 )
          {
              dir += 360.0;
          }
          while ( dir > 180.0 )
          {
              dir -= 360.0;
          }
          return dir;
      }

    inline
    static
    double rad2deg(const double &radx)
      {
          return radx * RAD2DEG;
      }

    inline
    static
    double deg2rad(const double &degx)
      {
          return degx * DEG2RAD;
      }

    inline
    static
    double cos_deg(const double &degx)
      {
          return cos(deg2rad(degx));
      }

    inline
    static
    double sin_deg(const double &degx)
      {
          return sin(deg2rad(degx));
      }

    inline
    static
    double tan_deg(const double &degx)
      {
          return tan(deg2rad(degx));
      }

    // returned value type is degree
    inline
    static
    double acos_deg(const double &degx)
      {
          return ( degx >= 1.0
                   ? 0.0
                   : ( degx <= -1.0
                       ? 180.0
                       : rad2deg(acos(degx)) ) );
      }

    inline
    static
    double asin_deg(const double &degx)
      {
          return ( degx >= 1.0
                   ? 90.0
                   : ( degx <= -1.0
                       ? -90.0
                       : rad2deg(asin(degx)) ) );
      }

    inline
    static
    double atan_deg(const double &tangent)
      {
          return rad2deg(atan(tangent));
      }

    inline
    static
    double atan_deg2(const double &y, const double &x)
      {
          return ( ( x == 0.0 && y == 0.0 )
                   ? 0.0
                   : rad2deg(atan2(y, x)) );
      }

};

////////////////////////////////////////////////////////////


class Vector2D {
private:

public:
    double x;
    double y;

    Vector2D()
        : x(0.0), y(0.0)
      { }
    Vector2D(const double &arg_x, const double &arg_y)
        : x(arg_x), y(arg_y)
      { }
    Vector2D(int, const double &radius, const AngleDeg &dir)
        : x(radius * dir.Cos()), y(radius * dir.Sin())
      { }

    inline const
    Vector2D & assign(const double &arg_x, const double &arg_y)
      {
          x = arg_x;
          y = arg_y;
          return *this;
      }

    inline const
    Vector2D & assign(int, const double &radius, const AngleDeg &dir)
      {
          x = radius * dir.Cos();
          y = radius * dir.Sin();
          return *this;
      }

    inline const
    Vector2D & add(const double &arg_x, const double &arg_y)
      {
          x += arg_x;
          y += arg_y;
          return *this;
      }

    inline const
    Vector2D & normalize(const double &len = 1.0)
      {
          double mag = this->r();
          if ( mag == 0 )
          {
              return *this;
          }
          return ( (*this) *= (len / mag) );
      }
    inline
    double r() const // radius of vector
      {
          return sqrt(x*x + y*y);
      }
    inline
    double r2() const // squared radius of vector
      {
          return x*x + y*y;
      }

    inline
    AngleDeg th() const
      {
          return AngleDeg(AngleDeg::atan_deg2(y, x));
      }

    inline
    Vector2D abs() const
      {
          return Vector2D(fabs(this->x),
                          fabs(this->y));
      }

    inline
    double absX() const
      {
          return fabs(this->x);
      }
    inline
    double absY() const
      {
          return fabs(this->y);
      }

    inline
    Vector2D normalizedVector(const double &len = 1.0) const
      {
          double mag = this->r();
          if ( mag == 0.0 )
          {
              return Vector2D(*this);
          }
          return Vector2D(*this) *= (len / mag);
      }

    inline
    Vector2D operator -() const
      {
          return Vector2D(-x, -y);
      }

    inline
    const
    Vector2D & operator +=(const Vector2D &a)
      {
          x += a.x;
          y += a.y;
          return *this;
      }

    inline
    const
    Vector2D & operator -=(const Vector2D &a)
      {
          x -= a.x;
          y -= a.y;
          return *this;
      }
    inline
    const
    Vector2D & operator *=(const Vector2D &a)
      {
          x *= a.x;
          y *= a.y;
          return *this;
      }
    inline
    const
    Vector2D & operator -=(const double &a)
      {
          x -= a;
          y -= a;
          return *this;
      }
    inline
    const
    Vector2D & operator *=(const double &a)
      {
          x *= a;
          y *= a;
          return *this;
      }
    inline
    const
    Vector2D & operator /=(const double &a)
      {
          x /= a;
          y /= a;
          return *this;
      }

    inline
    bool operator !=(const Vector2D &a) const
      {
          return (x != a.x) || (y != a.y);
      }
    inline
    bool operator ==(const Vector2D &a) const
      {
          return fabs(x - a.x) < 0.001 && fabs(y - a.y) < 0.001;
      }


    inline
    double dist(const Vector2D &a) const
      {
          return (Vector2D(*this) -= a).r();
      }
    inline
    double dist2(const Vector2D &a) const
      {
          return (Vector2D(*this) -= a).r();
      }

    inline
    Vector2D rotatedVector(const double &deg) const
      {
          double radius = this->r();
          double rotated_angle = this->th().degree();
          rotated_angle += deg;
          rotated_angle *= AngleDeg::DEG2RAD;
          return Vector2D(radius * cos(rotated_angle),
                          radius * sin(rotated_angle));
      }
    inline
    Vector2D rotatedVector(const AngleDeg &angle) const
      {
          double radius = this->r();
          double rotated_angle = this->th().degree();
          rotated_angle += angle.degree();
          rotated_angle *= AngleDeg::DEG2RAD;
          return Vector2D(radius * cos(rotated_angle),
                          radius * sin(rotated_angle));
      }
    inline const
    Vector2D & rotate(const double &deg)
      {
          double radius = this->r();
          double rotated_angle = this->th().degree();
          rotated_angle += deg;
          rotated_angle *= AngleDeg::DEG2RAD;
          this->x = radius * cos(rotated_angle);
          this->y = radius * sin(rotated_angle);
          return *this;
      }
    inline const
    Vector2D & rotate(const AngleDeg &angle)
      {
          double radius = this->r();
          double rotated_angle = this->th().degree();
          rotated_angle += angle.degree();
          rotated_angle *= AngleDeg::DEG2RAD;
          this->x = radius * cos(rotated_angle);
          this->y = radius * sin(rotated_angle);
          return *this;
      }

    inline
    double innerProduct(const Vector2D &v) const
      {
          return this->x * v.x + this->y * v.y;
      }
    inline
    double outerProduct(const Vector2D &v) const
      {
          /*---------------------*
           * assume virtual 3D environment.
           * calculate Z-coordinate of outer product in right hand orientation.
           * For the time being, Input Vector's Z-coordinate is set to ZERO.
           *---------------------*/
          // Normal 3D outer product algorithm
          //   xn = this->y * v.z - this->z * v.y;
          //   yn = this->z * v.x - this->x * v.z;
          // @ zn = this->x * v.y - this->y * v.x;
          return this->x * v.y - this->y * v.x;
      }

    //////////////////////////////////////////////
    // static utility
    inline
    static
    Vector2D polar2vector(const double &mag, const AngleDeg &theta)
      {
          return Vector2D(mag * theta.Cos(), mag * theta.Sin());
      }
    inline
    static
    double inner_product(const Vector2D &v1, const Vector2D &v2)
      {
          return v1.x * v2.x + v1.y * v2.y;
      }
    inline
    static
    double outer_product(const Vector2D &v1, const Vector2D &v2)
      {
          return v1.x * v2.y - v1.y * v2.x;
      }
};

////////////////////////////////////////////////////////
// arith operator

inline
const
Vector2D operator +(const Vector2D &lhs, const Vector2D rhs)
{
    return Vector2D(lhs) += rhs;
}

inline
const
Vector2D operator -(const Vector2D &lhs, const Vector2D rhs)
{
    return Vector2D(lhs) -= rhs;
}

inline
const
Vector2D operator *(const Vector2D &vec, const double &scalar)
{
    return Vector2D(vec) *= scalar;
}

inline
const
Vector2D operator /(const Vector2D &vec, const double &scalar)
{
    return Vector2D(vec) /= scalar;
}

//////////////////////////////////////////////
// functor for compare

class Vector2DXCmp {
public:
    bool operator()(const Vector2D &lhs,
                    const Vector2D &rhs) const
    {
        return lhs.x < rhs.x;
    }
};

class Vector2DYCmp {
public:
    bool operator()(const Vector2D &lhs,
                    const Vector2D &rhs) const
    {
        return lhs.y < rhs.y;
    }
};


////////////////////////////////////////////////////////////////

class Line2D {
private:
    // aX + bY + c = 0
    double M_a;
    double M_b;
    double M_c;

public:
    // construct directly
    Line2D(const double &x_coef = 0.0,
           const double &y_coef = 0.0,
           const double &constant = 0.0)
        : M_a(x_coef), M_b(y_coef), M_c(constant)
      { }
    // construct from 2 points
    Line2D(const Vector2D &p1, const Vector2D &p2)
      {
          setByTwoPoints(p1, p2);
      }
    // construct from origin point + direction
    Line2D(const Vector2D &origin, const AngleDeg &linedir)
      {
          setByTwoPoints(origin,
                         origin + Vector2D::polar2vector(1.0, linedir));
      }

    inline // const StraightLine2d &
    void setByTwoPoints(const Vector2D &p1, const Vector2D &p2)
      {
          if ( fabs(p2.x - p1.x) < 0.0001 )
          {
              if ( fabs(p2.y - p1.y) < 0.0001 )
              {
                  //std::cerr << "StraightLine2D::SetByTwoPoints:"
                  //    " points can not be the same!\n";
              }
              M_a = 1.0;
              M_b = 0.0;
          }
          else
          {
              M_a = -((p2.y - p1.y) / (p2.x - p1.x));
              M_b = 1.0;
          }
          M_c = -(M_a * p2.x + M_b * p2.y);
      }

    inline
    const
    double & A() const
      {
          return M_a;
      }
    inline
    const
    double & B() const
      {
          return M_b;
      }
    inline
    const
    double & C() const
      {
          return M_c;
      }

    inline
    double slope() const
    {
        if ( fabs(B()) < 0.0001 )
            return 0.0;
        return -A() / B();
    }
    inline
    double constant() const
    {
        if ( fabs(B()) < 0.0001 )
            return 0.0;
        return -C() / B();
    }

    // get X-coordinate correspond to Y-coordinate
    inline
    double getX(const double &y) const
      {
          if ( fabs(M_a) < 0.0001 )
          {
              //std::cerr << "cannot get x\n";
              return 0.0;
          }
          return -(M_b * y + M_c) / M_a;
      }
    // get Y-coordinate correspond to X-coordinate
    inline
    double getY(const double &x) const
      {
          if ( fabs(M_b) < 0.0001 )
          {
              //std::cerr << "cannot get y\n";
              return 0.0;
          }

          return -(M_a * x + M_c) / M_b;
      }

    // distance from this line
    inline
    double dist(const Vector2D &p) const
      {
          return fabs( (M_a * p.x + M_b * p.y + M_c)
                       / sqrt(M_a * M_a + M_b * M_b) );
      }
    inline
    double dist2(const Vector2D &p) const
      {
          double d = M_a * p.x + M_b * p.y + M_c;
          return (d * d) / (M_a * M_a + M_b * M_b);
      }

    bool isSameSlope(const Line2D &line) const;
    bool intersection(const Line2D &l, Vector2D &sol) const;

    // get perpendicular line (SUI-SEN)
    inline
    Line2D perpendicular(const Vector2D &p) const
      {
          return Line2D(B(), -A(), A() * p.y - B() * p.x);
      }
    // get projection point (SHA-EI)
    inline
    Vector2D projection(const Vector2D &p) const
      {
          Vector2D sol;
          intersection(perpendicular(p), sol);
          return sol;
      }

};


#endif

