/*-----------------------------------------------------------------------------
Modul Name : BaseAgent.cpp
Date       : 30.10.97
-------------------------------------------------------------------------------
Author     : Klaus Dorer

Description: contains the operations for a basic agent

Comments   : _
-----------------------------------------------------------------------------*/

#include "stdafx.h"
#include "Agentsock.h"
#include "TBaseAgent.h"

#include <math.h>


#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif


#define MAX_MSG_LENGTH 1024

/*-----------------------------------------------------------------------------
Function: TBaseAgent::TBaseAgent()
Date    : 31.10.97
-------------------------------------------------------------------------------
Parameter: none
Description: constructor
-----------------------------------------------------------------------------*/

TBaseAgent::TBaseAgent()
{
	m_iPositionX = 0;
	m_iPositionY = -30;
	m_iDirection = 0;
    M_connected = false;
}

TBaseAgent::~TBaseAgent()
{
	DisconnectAgent();
}

/*-----------------------------------------------------------------------------
Function: BOOL TBaseAgent::ConnectAgent(LPCTSTR lpszTeam, LPCTSTR lpszServer, UINT nPort)
Date    : 31.10.97
-------------------------------------------------------------------------------
Parameter: name of the team
			  name of the server (URL)
			  port for the socket
Returnval: TRUE, if the agent could be created and connected to a server socket
			  FALSE otherwise
Description: creates the socket of the agent and initializes it
-----------------------------------------------------------------------------*/
BOOL TBaseAgent::ConnectAgent(LPCTSTR lpszTeam, LPCTSTR lpszServer, UINT nPort)
{
	// is the agent already connected (should not happen) 
	if ( M_connected )
    {
		return FALSE;
    }

	// create the datagram socket
	if ( ! Create(nPort, lpszServer) )
	{
		// socket could not be created
		OnErrorCreatingSocket();
		return FALSE;
	}
	else
	{
		M_connected = true;
		m_strTeam = lpszTeam;
		OnCreatedSocket();
		return TRUE;
	}
}

/*-----------------------------------------------------------------------------
Function: void OnCreatedSocket()
Date    : 17.12.97
-------------------------------------------------------------------------------
Parameter: none
Returnval: (void)
Description: notification handler of a succesful socket creation
-----------------------------------------------------------------------------*/
void TBaseAgent::OnCreatedSocket()
{
	CString	message;

	// send the (init team) command
	message.Format("(init %s)", (LPCTSTR) m_strTeam);
	SendMsg(message);
}

/*-----------------------------------------------------------------------------
Function: void TBaseAgent::DisconnectAgent()
Date    : 16.12.97
-------------------------------------------------------------------------------
Parameter: none
Returnval: (void)
Description: empties the agent's socket and deletes it
-----------------------------------------------------------------------------*/
void TBaseAgent::DisconnectAgent()
{
	// is the socket active ?
	if ( M_connected )
	{
		BYTE Buffer[50];

		ShutDown();
		while ( Receive(Buffer,50) > 0 );
		Close();
	}
	M_connected = false;
}

/*-----------------------------------------------------------------------------
Function: void TBaseAgent::SendMsg(CString& strText)
Date    : 31.10.97
-------------------------------------------------------------------------------
Parameter: message to be sent
Returnval: (void)
Description: the message is sent to the socket by SendTo (unconnected socket)
-----------------------------------------------------------------------------*/
void TBaseAgent::SendMsg(CString& strText)
{
	int iLength;
	
	// send the buffer
	iLength = SendTo(strText, strText.GetLength(), GetPort(), GetAddress(), 0);
	if ( iLength == strText.GetLength() )
    {
		OnSentMessage((LPCTSTR) strText);
    }
	else
	{
		// an error occured
		#ifdef _DEBUG
			afxDump << "error writing to socket\n";
		#endif
		OnErrorSendingMessage();
	}
}

/*-----------------------------------------------------------------------------
Function: void TBaseAgent::ReceiveMsg()
Date    : 31.10.97
-------------------------------------------------------------------------------
Parameter: none
Returnval: (void)
Description: a message is received from the socket and processed by the agent
-----------------------------------------------------------------------------*/
void TBaseAgent::OnReceive(int iError)
{
	char		buffer[MAX_MSG_LENGTH];
	int		iLength = 0;
	UINT		iPort = GetPort();
	int		iPosition = 0;

	// first call base classes method
	CAsyncSocket::OnReceive(iError);
	//CAgentSocket::OnReceive(iError);

	// read a message from the socket
	iLength = ReceiveFrom(buffer, MAX_MSG_LENGTH, GetAddress(), iPort, 0);
	// the port may have changed
	SetPort(iPort);

	if (iLength > 0)
		OnReceivedMessage((LPCTSTR) buffer);
	else
	{
		// an error occured
		#ifdef _DEBUG
			afxDump << "error reading socket\n";
		#endif
		OnErrorReceivingMessage();
	}
}

/*-----------------------------------------------------------------------------
Function: void TBaseAgent::Move(int x, int y)
Date    : 31.10.97
-------------------------------------------------------------------------------
Parameter: x,y coordinate (in meters) where to move to before playing
Returnval: (void)
Description: following functions implement the abilities of soccer agents
				 see the soccer manual for more details
-----------------------------------------------------------------------------*/
void TBaseAgent::Move(double x, double y)
{
	CString	message;

	message.Format("(move %.2f %.2f)", x, y);
	SendMsg(message);

	m_iPositionX = x;
	m_iPositionY = y;
}

void TBaseAgent::Turn(double angle)
{
	CString	message;

	// angle may be -180 to 180
	message.Format("(turn %.2f)", angle);
	SendMsg(message);

	m_iDirection += angle;
	if ( m_iDirection > 180 )
    {
		m_iDirection -= 360;
    }
}

void TBaseAgent::Dash(double power)
{
	CString	message;

	// power may be -30 to 100
	message.Format("(dash %.2f)", power);
	SendMsg(message);
}

void TBaseAgent::Kick(double power, double direction)
{
	CString	message;
	
	// power -30 to 100
	// direction -180 to 180
	message.Format("(kick %.2f %.2f)", power, direction);
	SendMsg(message);
}

void TBaseAgent::Say(CString msg)
{
	CString	message;

	// max 255 byte
	message.Format("(say ");
	message += msg;
	message += _T(")");
	SendMsg(message);
}

void TBaseAgent::ChangeView(int width, int quality)
{
	CString	message;

	message =_T("(change_view ");
	// set the view angle string
	if ( width == 180 )
    {
		message += _T("wide ");
    }
    else if ( width == 90 )
    {
		message += _T("normal ");
    }
	else
    {
		message += _T("narrow ");
    }

	// set the view quality string
	if ( quality == 0 )
    {
		message += _T("low)");
    }
	else
    {
		message += _T("high)");
    }

	SendMsg(message);
}

