// -*-c++-*-

/*!
  \file player_config.cpp
  \brief player configuration Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "player_config.h"
#include "logger.h"

#include <rcsc/param/param_map.h>

#include <cassert>

#ifdef _MSC_VER
	#include "win_src/StepDebug_Define.h"
#endif

namespace rcsc {

/*-------------------------------------------------------------------*/
/*!

*/
PlayerConfig::PlayerConfig()
{
    setDefaultParam();
}

/*-------------------------------------------------------------------*/
/*!

*/
PlayerConfig::~PlayerConfig()
{

    //cerr << "delete PlayerConfig" << endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerConfig::setDefaultParam()
{
    // basic setting
    M_team_name = "HELIOS";
    M_version = 8.4;
    M_reconnect_number = 0;
    M_goalie = false;

    M_interval_msec = 10;
    M_server_wait_seconds = 5;

    M_wait_time_thr_synch_view = 79;
    M_wait_time_thr_nosynch_view = 75;

    M_normal_view_time_thr = 15;

    M_rcssserver_host = "localhost";
    M_rcssserver_port = 6000;

    M_clang_min = 7;
    M_clang_max = 8;

    M_use_communication = true;
    M_hear_opponent_audio = false;

    M_use_fullstate = true;

    // accuracy threshold
    M_self_pos_count_thr = 20;
    M_self_vel_count_thr = 10;
    M_self_face_count_thr = 5;

    M_ball_pos_count_thr = 10;
    M_ball_rpos_count_thr = 5;
    M_ball_vel_count_thr = 10;

    M_player_pos_count_thr = 30;
    M_player_vel_count_thr = 5;
    M_player_face_count_thr = 2;

    // formation param
    M_config_dir = "./";
    M_player_number = 0;


    //////////////////////////////////////////////////////
    // debug
    M_debug_connect = false;
    M_debug_server_host = "localhost";
    M_debug_server_port = 6000 + 32;

    M_debug_write = false;

    M_log_dir = "/tmp";
    M_log_ext = ".log";

    M_debug = false;
    M_debug_system = false;
    M_debug_sensor = false;
    M_debug_world = false;
    M_debug_action = false;
    M_debug_intercept = false;
    M_debug_kick = false;
    M_debug_dribble = false;
    M_debug_pass = false;
    M_debug_cross = false;
    M_debug_shoot = false;
    M_debug_clear = false;
    M_debug_team = false;
    M_debug_role = false;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerConfig::createParamMap( ParamMap & param_map )
{
    param_map.add()
        ( "team_name", "t", &M_team_name, "specifies team name string." )
        ( "version", "v", &M_version, "specifies the protocol version to communicate with the rcssserver." )
        ( "reconnect", "r", &M_reconnect_number,
          "recconect to the rcssserver. This option requres the player\'s uniform number for the reconnection." )

        ( "goalie", "g", BoolSwitch( &M_goalie ) )

        ( "interval_msec", "", &M_interval_msec )
        ( "server_wait_seconds", "", &M_server_wait_seconds )

        ( "wait_time_thr_synch_view", "", &M_wait_time_thr_synch_view )
        ( "wait_time_thr_nosynch_view","", &M_wait_time_thr_nosynch_view )

        ( "normal_view_time_thr", "", &M_normal_view_time_thr )

        ( "host", "h", &M_rcssserver_host )
        ( "port", "", &M_rcssserver_port )

        ( "clang_min", "", &M_clang_min )
        ( "clang_max", "", &M_clang_max )

        ( "use_communication", "", &M_use_communication )
        ( "hear_opponent_audio", "", &M_hear_opponent_audio )

        ( "use_fullstate", "", &M_use_fullstate )

        ( "self_pos_count_thr", "", &M_self_pos_count_thr )
        ( "self_vel_count_thr", "", &M_self_vel_count_thr )
        ( "self_face_count_thr", "", &M_self_face_count_thr )
        ( "ball_pos_count_thr", "", &M_ball_pos_count_thr )
        ( "ball_rpos_count_thr", "", &M_ball_rpos_count_thr )
        ( "ball_vel_count_thr", "", &M_ball_vel_count_thr )
        ( "player_pos_count_thr", "", &M_player_pos_count_thr )
        ( "player_vel_count_thr", "", &M_player_vel_count_thr )
        ( "player_face_count_thr", "", &M_player_face_count_thr )

        ( "config_dir", "", &M_config_dir )
        ( "player_number", "n",  &M_player_number )

        ( "debug_connect", "", BoolSwitch( &M_debug_connect ) )
        ( "debug_server_host", "", &M_debug_server_host )
        ( "debug_server_port", "", &M_debug_server_port )

        ( "debug_write", "", BoolSwitch( &M_debug_write ) )

        ( "log_dir", "", &M_log_dir )
        ( "log_ext", "", &M_log_ext )

        ( "debug", "", BoolSwitch( &M_debug ) )
        ( "debug_system", "", BoolSwitch( &M_debug_system ) )
        ( "debug_sensor", "", BoolSwitch( &M_debug_sensor ) )
        ( "debug_world", "", BoolSwitch( &M_debug_world ) )
        ( "debug_action", "", BoolSwitch( &M_debug_action ) )
        ( "debug_intercept", "", BoolSwitch( &M_debug_intercept ) )
        ( "debug_kick", "", BoolSwitch( &M_debug_kick ) )
        ( "debug_dribble", "", BoolSwitch( &M_debug_dribble ) )
        ( "debug_pass", "", BoolSwitch( &M_debug_pass ) )
        ( "debug_cross", "", BoolSwitch( &M_debug_cross ) )
        ( "debug_shoot", "", BoolSwitch( &M_debug_shoot ) )
        ( "debug_clear", "", BoolSwitch( &M_debug_clear ) )
        ( "debug_team", "", BoolSwitch( &M_debug_team ) )
        ( "debug_role", "", BoolSwitch( &M_debug_role ) );
}

bool PlayerConfig::Write2StepDebugFile( FILE* pStepDebugFile ) const
{
		// check pointer
		if( pStepDebugFile == 0 )
			return false;
		if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
			return false;

		long lStart = ftell( pStepDebugFile );
		if( lStart == -1 )
			return false;

		// write head
		STEPDEBUG_HEAD	wm_head;
		{
			wm_head._dwRegCode		= REGCODE_PlayerConfig;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );
		}

		// write data
		{
			fwrite( this, ((long)&M_team_name - (long)this), 1, pStepDebugFile );
		}

		// write string
		{
			// the string is limited to 16 byte when saved in data log file
			char buff[16];

			// M_team_name
			memset( buff, 0, 16 );
			if( M_team_name.size() > 15 )
				memcpy( buff, M_team_name.c_str(), 15 );
			else
				strcpy( buff, M_team_name.c_str() );
			fwrite( buff, 16, 1, pStepDebugFile );

			// M_rcssserver_host
			memset( buff, 0, 16 );
			if( M_rcssserver_host.size() > 15 )
				memcpy( buff, M_rcssserver_host.c_str(), 15 );
			else
				strcpy( buff, M_rcssserver_host.c_str() );
			fwrite( buff, 16, 1, pStepDebugFile );

			// M_config_dir
			memset( buff, 0, 16 );
			if( M_config_dir.size() > 15 )
				memcpy( buff, M_config_dir.c_str(), 15 );
			else
				strcpy( buff, M_config_dir.c_str() );
			fwrite( buff, 16, 1, pStepDebugFile );

			// M_debug_server_host
			memset( buff, 0, 16 );
			if( M_debug_server_host.size() > 15 )
				memcpy( buff, M_debug_server_host.c_str(), 15 );
			else
				strcpy( buff, M_debug_server_host.c_str() );
			fwrite( buff, 16, 1, pStepDebugFile );

			// M_log_dir
			memset( buff, 0, 16 );
			if( M_log_dir.size() > 15 )
				memcpy( buff, M_log_dir.c_str(), 15 );
			else
				strcpy( buff, M_log_dir.c_str() );
			fwrite( buff, 16, 1, pStepDebugFile );

			// M_log_ext
			memset( buff, 0, 16 );
			if( M_log_ext.size() > 15 )
				memcpy( buff, M_log_ext.c_str(), 15 );
			else
				strcpy( buff, M_log_ext.c_str() );
			fwrite( buff, 16, 1, pStepDebugFile );


		}

		// rewrite head with size
		{
			if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
				return false;

			long lEnd		= ftell( pStepDebugFile );
			if( lStart == -1 )
				return false;

			wm_head._dwSize	= lEnd - lStart;
			if( fseek( pStepDebugFile, lStart, SEEK_SET ) != 0 )
				return false;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );

			fseek(pStepDebugFile,0L,SEEK_END);
		}

		return true;
}

bool PlayerConfig::ReadFromDataFile( const char* pData ) const
{
	const char* pCurrent = pData;

	// check head

	STEPDEBUG_HEAD*		pHead = (STEPDEBUG_HEAD*)pCurrent;
	if( pHead->_dwRegCode != REGCODE_PlayerConfig )
	{
		ASSERT(false);
		return false;
	}

	// read data
	{
		memcpy( (void*)this, pHead->_pData, ((long)&M_team_name - (long)this) );
		pCurrent = pCurrent + pHead->size() + ((long)&M_team_name - (long)this);
	}

	// read string
	{
		// the string is limited to 16 byte when saved in data log file

		// M_team_name
		(std::string)M_team_name = pCurrent;
		pCurrent += 16;

		// M_rcssserver_host
		(std::string)M_rcssserver_host = pCurrent;
		pCurrent += 16;

		// M_config_dir
		(std::string)M_config_dir = pCurrent;
		pCurrent += 16;

		// M_debug_server_host
		(std::string)M_debug_server_host = pCurrent;
		pCurrent += 16;

		// M_log_dir
		(std::string)M_log_dir = pCurrent;
		pCurrent += 16;

		// M_log_ext
		(std::string)M_log_ext = pCurrent;
		pCurrent += 16;

	}
	
	return true;
}

}
