// -*-c++-*-

/*!
  \file audio_codec.cpp
  \brief audio message encoder/decoder Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "audio_codec.h"

#include <rcsc/math_util.h>
#include <rcsc/types.h>

#include <iostream>
#include <algorithm>
#include <limits>

namespace rcsc {

const double AudioCodec::ERROR_VALUE = std::numeric_limits< double >::max();

const double AudioCodec::X_NORM_FACTOR = 57.5;
const double AudioCodec::Y_NORM_FACTOR = 39.0;
const double AudioCodec::SPEED_NORM_FACTOR = 3.0;

const int AudioCodec::POSX_BIT_L5 = 8; //!< used by posVelToInt32
const int AudioCodec::POSY_BIT_L5 = 8; //!< used by posVelToInt32
const int AudioCodec::VEL_BIT_L5 = 6; //!< used by posVelToInt32
// related to POSX_BIT_L5, POSY_BIT_L5, VEL_BIT_L5
const boost::int32_t AudioCodec::POSX_MASK_L5 = 0x000000ff;
const boost::int32_t AudioCodec::POSY_MASK_L5 = 0x0000ff00;
const boost::int32_t AudioCodec::VELX_MASK_L5 = 0x003f0000;
const boost::int32_t AudioCodec::VELY_MASK_L5 = 0x0fc00000;

/*
  long bit_8 = 0x000000ff; // base 8 bit mask
  long bit_7 = 0x0000007f; // base 7 bit mask
  long bit_6 = 0x0000003f; // base 6 bit mask
  long bit_5 = 0x0000001f; // base 5 bit mask
  long bit_4 = 0x0000000f; // base 4 bit mask

  printf("%#010x\n", bit_8); // POSX_MASK_L5
  printf("%#010x\n", (bit_8 << POSX_BIT_L5)); // POSY_MASK_L5
  printf("%#010x\n", (bit_6 << (POSX_BIT_L5 + POSY_BIT_L5))); // VELX_MASK_L5
  printf("%#010x\n", (bit_6 << (POSX_BIT_L5 + POSY_BIT_L5 + VEL_BIT_L5))); // VELY_MASK_L5
*/

const double AudioCodec::COORD_STEP_L2 = 0.1;
const double AudioCodec::SPEED_STEP_L1 = 0.1;


//[-0-9a-zA-Z ().+*/?<>_]
const std::string AudioCodec::CHAR_SET =
"0123456789"
"abcdefghijklmnopqrstuvwxyz"
"ABCDEFGHIJKLMNOPQRSTUVWXYZ"
"().+-*/?<>_";
//" ().+-*/?<>_"; remove space

const int AudioCodec::CHAR_SIZE = static_cast< int >( CHAR_SET.length() );


/*-------------------------------------------------------------------*/
/*!

*/
AudioCodec::AudioCodec()
{
    int count = 0;

    // create int <-> char map

    // [0-9]
    for ( int i = 0; i < 10; ++i )
    {
        char ch = static_cast< char >( '0' + i );
        M_char_to_int_map.insert( std::make_pair( ch, count++ ) );
        M_int_to_char_map.push_back( ch );
    }
    // [a-z]
    for ( int i = 0; i < 26; ++i )
    {
        char ch = static_cast< char >( 'a' + i );
        M_char_to_int_map.insert( std::make_pair( ch, count++ ) );
        M_int_to_char_map.push_back( ch );
    }
    // [A-Z]
    for ( int i = 0; i < 26; ++i )
    {
        char ch = static_cast< char >( 'A' + i );
        M_char_to_int_map.insert( std::make_pair( ch, count++ ) );
        M_int_to_char_map.push_back( ch );
    }

    // [ ().+-*/?<>_]
    //M_char_to_int_map.insert( std::make_pair( ' ', count++ ) );
    //M_int_to_char_map.push_back( ' ' );
    M_char_to_int_map.insert( std::make_pair( '(', count++ ) );
    M_int_to_char_map.push_back( '(' );
    M_char_to_int_map.insert( std::make_pair( ')', count++ ) );
    M_int_to_char_map.push_back( ')' );
    M_char_to_int_map.insert( std::make_pair( '.', count++ ) );
    M_int_to_char_map.push_back( '.' );
    M_char_to_int_map.insert( std::make_pair( '+', count++ ) );
    M_int_to_char_map.push_back( '+' );
    M_char_to_int_map.insert( std::make_pair( '-', count++ ) );
    M_int_to_char_map.push_back( '-' );
    M_char_to_int_map.insert( std::make_pair( '*', count++ ) );
    M_int_to_char_map.push_back( '*' );
    M_char_to_int_map.insert( std::make_pair( '/', count++ ) );
    M_int_to_char_map.push_back( '/' );
    M_char_to_int_map.insert( std::make_pair( '?', count++ ) );
    M_int_to_char_map.push_back( '?' );
    M_char_to_int_map.insert( std::make_pair( '<', count++ ) );
    M_int_to_char_map.push_back( '<' );
    M_char_to_int_map.insert( std::make_pair( '>', count++ ) );
    M_int_to_char_map.push_back( '>' );
    M_char_to_int_map.insert( std::make_pair( '_', count++ ) );
    M_int_to_char_map.push_back( '_' );
}

/*-------------------------------------------------------------------*/
/*!

*/
boost::int32_t
AudioCodec::posVelToInt32( const Vector2D & pos,
                           const Vector2D & vel ) const
{
    boost::int32_t result = 0;
    double dval;
    int ival;

    // encode pos x
    //   check range[-X_NORM_FACTOr, X_NORM_FACTOr]
    //   normalize(+X_NORM_FACTOR)(*2.0)[0, 4 * X_NORM_FACTOR]
    //              <---- POSX_BIT_L5 bit
    //     (rounded[step=0.5])
    dval = min_max( -X_NORM_FACTOR, pos.x, X_NORM_FACTOR );
    dval += X_NORM_FACTOR;
    dval /= 0.5; // rounded by step 0.5
    ival = static_cast< int >( rint( dval ) );

    result |= ival;


    // encode pos y
    //   check range[-Y_NORM_FACTOR, Y_NORM_FACTOR]
    //   normalize(+Y_NORM_FACTOR)(*2.0)[0, 4 * Y_NORM_FACTOR]
    //              <--------- POSY_BIT_L5 bit
    //    (rounded[step=0.5])
    dval = min_max( -Y_NORM_FACTOR, pos.y, Y_NORM_FACTOR );
    dval += Y_NORM_FACTOR;
    dval /= 0.5; // rounded by step 0.5
    ival = static_cast< int >( rint( dval ) );

    result |= ( ival << POSX_BIT_L5 );


    // encode vel x
    //   check range[-3, 3]
    //   normalize(+3.0)(*10.0)[0, 20 * SPEED_NORM_FACTOR]
    //          <--------- VEL_BIT_L5 bit
    //    (rounded[step=0.1])
    dval = min_max( -SPEED_NORM_FACTOR, vel.x, SPEED_NORM_FACTOR );
    dval += SPEED_NORM_FACTOR;
    dval /= 0.1; // rounded by step 0.1
    ival = static_cast< int >( rint( dval ) );

    result |= ( ival << ( POSX_BIT_L5 + POSY_BIT_L5 ) );


    // encode vel y
    //   check range[-3, 3]
    //   normalize(+3.0)(*10.0)[0, 20 * SPEED_NORM_FACTOR]
    //           <--------- VEL_BIT_L5 bit
    //    (rounded[step=0.1])
    dval = min_max( -SPEED_NORM_FACTOR, vel.y, SPEED_NORM_FACTOR );
    dval += SPEED_NORM_FACTOR;
    dval /= 0.1;
    ival = static_cast< int >( rint( dval ) );

    result |= ( ival << ( POSX_BIT_L5 + POSY_BIT_L5 + VEL_BIT_L5 ) );


    return result;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
AudioCodec::int32ToPosVel( const boost::int32_t & val,
                           Vector2D * pos,
                           Vector2D * vel ) const
{
    boost::int32_t ival;

    // decode pos x
    //cout << std::dec;
    ival = static_cast< boost::int32_t >( val & POSX_MASK_L5 );
    //cout << "d posx int " << ival << endl;
    pos->x
        = static_cast< double >( ival ) * 0.5
        - X_NORM_FACTOR;

    ival = static_cast< boost::int32_t >( val & POSY_MASK_L5 );
    //cout << "d posy int " << ( ival >> POSX_BIT_L5 ) << endl;
    pos->y
        = static_cast< double >( ival >> POSX_BIT_L5 ) * 0.5
        - Y_NORM_FACTOR;

    ival = static_cast< boost::int32_t >( val & VELX_MASK_L5 );
    //cout << "d velx int " << ( ival >> ( POSX_BIT_L5 + POSY_BIT_L5 ) ) << endl;
    vel->x
        = static_cast< double >( ival >> ( POSX_BIT_L5 + POSY_BIT_L5 ) ) * 0.1
        - SPEED_NORM_FACTOR;

    ival = static_cast< boost::int32_t >( val & VELY_MASK_L5 );
    //cout << "d vely int " << ( ival >> ( POSX_BIT_L5 + POSY_BIT_L5 + VEL_BIT_L5 ) ) << endl;
    vel->y
        = static_cast< double >( ival >> ( POSX_BIT_L5 + POSY_BIT_L5 + VEL_BIT_L5 ) ) * 0.1
        - SPEED_NORM_FACTOR;

}

/*-------------------------------------------------------------------*/
/*!

*/
std::string
AudioCodec::encodePosVelToStr5( const Vector2D & pos,
                                const Vector2D & vel ) const
{
    std::string msg;
    msg.reserve( 5 );

    // convert to long

    const boost::int32_t ival = posVelToInt32( pos, vel );

    //cout << "  original long "<< std::dec << ival << endl;

    std::vector< int > talk_values;
    boost::int32_t divided = ival;

    for ( int i = 0; i < 4; i++ )
    {
        talk_values.push_back( divided % CHAR_SIZE );
        divided /= CHAR_SIZE;
    }
    if ( divided > CHAR_SIZE )
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << " ***ERROR*** AudioCodec::encodeL5."
                  << " over flow. pos=" << pos
                  << " vel=" << vel
                  << std::endl;
        return false;
    }
    talk_values.push_back( divided % CHAR_SIZE );

    // reverse copy
    for ( std::vector< int >::reverse_iterator it = talk_values.rbegin();
          it != talk_values.rend();
          ++it )
    {
        msg += intToCharMap()[*it];
        //cout << " int [" << *it << "] to char [" << *msg.rbegin() << "]" << endl;
    }
    //cout << " encode ----> [" << to << "]" << endl;
    return msg;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
AudioCodec::decodeStr5ToPosVel( const std::string & from,
                                Vector2D * pos,
                                Vector2D * vel ) const
{
    if ( from.length() != 5 )
    {
        return false;
    }

    //cout << "AudioCodec::decode" << endl;
    boost::int32_t read_val = 0;
    bool success = true;
    const char *msg = from.c_str();
    int digit_count = std::strlen( msg ) - 1;
    while ( *msg != '\0' && digit_count >= 0 )
    {
        CharToIntCont::const_iterator it;
        //if ( *msg == '0' )
        //{
        //    cout << "decode ZERO" << endl;
        //}
        it = charToIntMap().find( *msg );
        if ( it == charToIntMap().end() )
        {
            std::cerr << __FILE__ << ": " << __LINE__
                      << " ***ERROR*** AudioCodec::encodeL5."
                      << " Unexpected communication message. ["
                      << from << "]"
                      << std::endl;
            success = false;
            break;
        }
        boost::int32_t add_val
            = it->second
            * static_cast< boost::int32_t >
            ( std::pow( static_cast< double >( CHAR_SIZE ),
                        static_cast< double >( digit_count ) ) );
        //cout << " char [" << *msg
        //     << "] read val " << it->second
        //     << "  add val " << add_val << endl;
        read_val += add_val;
        digit_count -= 1;
        msg++;
    }
    //cout << "  read long " << read_val << endl;
    if ( success )
    {
        int32ToPosVel( read_val, pos, vel );
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::string
AudioCodec::encodeCoordToStr2( const double & xy,
                               const double & norm_factor ) const
{
    // normalize value
    double tmp = min_max( -norm_factor, xy , norm_factor );
    tmp += norm_factor;

    // round
    tmp /= COORD_STEP_L2;

    int ival = static_cast< int >( rint( tmp ) );

    int i1 = ival % CHAR_SIZE;
    ival /= CHAR_SIZE;
    int i2 = ival % CHAR_SIZE;
    //std::cout << " posx -> " << ix1 << " " << ix2 << std::endl;

    if ( ival >= CHAR_SIZE )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** AudioCodec::encodeCoordToStr2."
                  << " range over. value = " << xy
                  << " norm_factor = " << norm_factor
                  << std::endl;
        return std::string();
    }

    std::string msg;
    msg.reserve( 2 );

    try
    {
        msg += intToCharMap().at( i1 );
        msg += intToCharMap().at( i2 );
    }
    catch ( ... )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** AudioCodec::encodeCoordToStr2."
                  << " Exception caught! "
                  << std::endl;
        return std::string();
    }

    return msg;
}

/*-------------------------------------------------------------------*/
/*!

*/
double
AudioCodec::decodeStr2ToCoord( const char ch1,
                               const char ch2,
                               const double & norm_factor ) const
{
    const CharToIntCont::const_iterator end = charToIntMap().end();

    CharToIntCont::const_iterator it = charToIntMap().find( ch1 );
    if ( it == end )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** AudioCodec::decodeStr2ToCoord()."
                  << " Unsupported character [" << ch1 << "]"
                  << std::endl;
        return ERROR_VALUE;
    }
    int i1 = it->second;

    it = charToIntMap().find( ch2 );
    if ( it == end )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** AudioCodec::decodeStr2ToCoord()."
                  << " Unsupported character [" << ch2 << "]"
                  << std::endl;
        return ERROR_VALUE;
    }
    int i2 = it->second;

    return
        (
         static_cast< double >( i1 + i2 * CHAR_SIZE ) * COORD_STEP_L2
         - norm_factor
         );
}

/*-------------------------------------------------------------------*/
/*!

 */
std::string
AudioCodec::encodePosToStr4( const Vector2D & pos ) const
{
    std::string pos_str;
    pos_str.reserve( 4 );

    pos_str += encodeCoordToStr2( pos.x, X_NORM_FACTOR );
    pos_str += encodeCoordToStr2( pos.y, Y_NORM_FACTOR );

    if ( pos_str.length() != 4 )
    {
        std::cerr << "AudioCodec::encodePosToStr4(). "
                  << "Failed to encode " << pos
                  << std::endl;
        pos_str.clear();
    }

    return pos_str;
}

/*-------------------------------------------------------------------*/
/*!

 */
Vector2D
AudioCodec::decodeStr4ToPos( const std::string & msg ) const
{
    if ( msg.length() < 4 )
    {
        std::cerr << "AudioCodec::decodeStr4ToPos(). "
                  << "message length is too short " << msg
                  << std::endl;
        return Vector2D::INVALIDATED;
    }

    Vector2D pos;

    pos.x = decodeStr2ToCoord( msg[0], msg[1], X_NORM_FACTOR );
    if ( pos.x == ERROR_VALUE )
    {
        std::cerr << "AudioCodec::decodeStr4ToPos(). "
                  << "Unexpected x value " << msg
                  << std::endl;
        return Vector2D::INVALIDATED;
    }

    pos.y = decodeStr2ToCoord( msg[2], msg[3], Y_NORM_FACTOR );
    if ( pos.y == ERROR_VALUE )
    {
        std::cerr << "AudioCodec::decodeStr4ToPos(). "
                  << "Unexpected x value " << msg
                  << std::endl;
        return Vector2D::INVALIDATED;
    }

    return pos;
}

/*-------------------------------------------------------------------*/
/*!

 */
std::string
AudioCodec::encodePosToStr7( const Vector2D & pos ) const
{
    double tmp = min_max( -52.5, pos.x, 52.5 );
    tmp += 52.5;
    tmp *= 10.0;
    int ix = static_cast< int >( rint( tmp ) );

    tmp = min_max( -34.0, pos.y, 34.0 );
    tmp += 34.0;
    tmp *= 10.0;
    int iy = static_cast< int >( rint( tmp ) );

    char xymsg[10];
    std::snprintf( xymsg, 10, "%04d%03d", ix, iy );
    if ( std::strlen( xymsg ) != 7 )
    {
        std::cerr << "AudioSensor::encodePosToStr7. "
                  << "XY error(2) "
                  << "( " << pos.x << pos.y << ")"
                  << " -> " << xymsg << std::endl;
        return std::string( "" );
    }

    return std::string( xymsg );
}

/*-------------------------------------------------------------------*/
/*!

 */
Vector2D
AudioCodec::decodeStr7ToPos( const char * msg ) const
{
    if ( std::strlen( msg ) != 7 )
    {
        std::cerr << "AudioSensor::decode length error "
                  << "[" << msg << "]" << std::endl;
        return Vector2D::INVALIDATED;
    }

    int x = 0;
    x += 1000 * (*msg - '0'); ++msg;
    x += 100 * (*msg - '0'); ++msg;
    x += 10 * (*msg - '0'); ++msg;
    x += (*msg - '0'); ++msg;

    int y = 0;
    y += 100 * (*msg - '0'); ++msg;
    y += 10 * (*msg - '0'); ++msg;
    y += (*msg - '0'); ++msg;

    return Vector2D( static_cast< double >( x ) * 0.1 - 52.5,
                     static_cast< double >( y ) * 0.1 - 34.0 );
}

/*-------------------------------------------------------------------*/
/*!

*/
char
AudioCodec::encodeSpeedToChar( const double & val ) const
{
    // normalize value
    double tmp = min_max( -SPEED_NORM_FACTOR, val , SPEED_NORM_FACTOR );
    tmp += SPEED_NORM_FACTOR;

    // round
    tmp /= SPEED_STEP_L1;

    int ival = static_cast< int >( rint( tmp ) );

    try
    {
        return  intToCharMap().at( ival );
    }
    catch ( ... )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** AudioCodec::encodeSpeedL1."
                  << " Exception caught! Failed to encode"
                  << std::endl;
        return '\0';
    }

    return '\0';
}

/*-------------------------------------------------------------------*/
/*!

*/
double
AudioCodec::decodeCharToSpeed( const char ch ) const
{
    CharToIntCont::const_iterator it = charToIntMap().find( ch );
    if ( it == charToIntMap().end() )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** AudioCodec::decodeSpeedL1."
                  << " Unsupported character [" << ch << "]"
                  << std::endl;
        return ERROR_VALUE;
    }

    return
        ( static_cast< double >( it->second ) * SPEED_STEP_L1
          - SPEED_NORM_FACTOR
          );
}

/*-------------------------------------------------------------------*/
/*!

*/
std::string
AudioCodec::encodePosVelToStr6( const Vector2D & pos,
                                const Vector2D & vel ) const
{
    std::string msg;
    msg.reserve( 6 );

    msg += encodeCoordToStr2( pos.x, X_NORM_FACTOR );
    msg += encodeCoordToStr2( pos.y, Y_NORM_FACTOR );
    msg += encodeSpeedToChar( vel.x );
    msg += encodeSpeedToChar( vel.y );

    if (  msg.length() != 6 )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** AudioCodec::encodeL6."
                  << " Invalid value! pos = " << pos << " vel = " << vel
                  << std::endl;
        msg.clear();
    }

    return msg;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
AudioCodec::decodeStr6ToPosVel( const std::string & from,
                                Vector2D * pos,
                                Vector2D * vel ) const
{
    if ( from.length() < 6 )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** AudioCodec::decodeL6."
                  << " Invalid string length! " << from.length()
                  << std::endl;
        return false;
    }

    if ( pos )
    {
        pos->x = decodeStr2ToCoord( from[0], from[1], X_NORM_FACTOR );
        if ( pos->x == ERROR_VALUE )
        {
            return false;
        }

        pos->y = decodeStr2ToCoord( from[2], from[3], Y_NORM_FACTOR );
        if ( pos->y == ERROR_VALUE )
        {
            return false;
        }
    }

    if ( vel )
    {
        vel->x = decodeCharToSpeed( from[4] );
        if ( vel->x == ERROR_VALUE )
        {
            return false;
        }

        vel->y = decodeCharToSpeed( from[5] );
        if ( vel->y == ERROR_VALUE )
        {
            return false;
        }
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
char
AudioCodec::unum2hex( const int unum )
{
    switch ( unum ) {
    case 1: return '1'; break;
    case 2: return '2'; break;
    case 3: return '3'; break;
    case 4: return '4'; break;
    case 5: return '5'; break;
    case 6: return '6'; break;
    case 7: return '7'; break;
    case 8: return '8'; break;
    case 9: return '9'; break;
    case 10: return 'A'; break;
    case 11: return 'B'; break;
    default:
        break;
    }

    return '\0';
}

/*-------------------------------------------------------------------*/
/*!

*/
int
AudioCodec::hex2unum( const char hex )
{
    switch ( hex ) {
    case '1': return 1; break;
    case '2': return 2; break;
    case '3': return 3; break;
    case '4': return 4; break;
    case '5': return 5; break;
    case '6': return 6; break;
    case '7': return 7; break;
    case '8': return 8; break;
    case '9': return 9; break;
    case 'A': return 10; break;
    case 'B': return 11; break;
    default:
        break;
    }

    return Unum_Unknown;
}

}
