// -*-c++-*-

/*!
  \file audio_sensor.h
  \brief audio message analyzer Header File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_PLAYER_AUDIO_SENSOR_H
#define RCSC_PLAYER_AUDIO_SENSOR_H

#include <rcsc/player/audio_codec.h>
#include <rcsc/geom/vector_2d.h>

#include <rcsc/game_time.h>
#include <rcsc/types.h>

#include <string>
#include <vector>

namespace rcsc {




/*-------------------------------------------------------------------*/
/*!
  \struct PlayerMessage;
  \brief player message holder
 */
struct PlayerMessage {

    /*!
      \brief communication message type
    */
    enum Type {
        UNKNOWN              = 0x00000000,
        BALL_INFO            = 0x00000001, // "b "
        PASS_INFO            = 0x00000002, // "p "
        OPPONENT_GOALIE_INFO = 0x00000004, // "g "
        OFFSIDE_LINE_INFO    = 0x00000008, // "o "
        DEFENSE_LINE_INFO    = 0x00000010, // "d "
        WAIT_REQUEST         = 0x00000020, // "w "
        ATTACK_REQUEST       = 0x00000040, // "a "
        INTERCEPT_INFO       = 0x00000080, // "i "
        HEY_PASS             = 0x00000100, // "h "
    };

    struct Info {
        std::string str_;
        Type type_;
    };

    GameTime time_; //!< received time
    SideID side_; //!< sender's side
    int unum_; //!< sender's uniform number
    double dir_; //!< sender's direction
    std::string message_; //!< raw message string

    std::vector< Info > info_;

    PlayerMessage()
        : time_( -1, 0 )
        , side_( NEUTRAL )
        , unum_( Unum_Unknown )
        , dir_( 0.0 )
      {
          message_.reserve( 10 );
      }

};

#if 0

struct BallInfoMessage
    : public PlayerMessage {

    Vector2D ball_pos_;
    Vector2D ball_vel_;

    void update( WorldModel & world )
      {
          world.setHeardBallInfo( ball_pos_,
                                  ball_vel_,
                                  time_ );
      }

};

struct PassInfoMessage
    : public PlayerMessage {

    int receiver_;
    Vector2D pos_;

    void update( WorldModel & world )
      {
          world.setHeardPassInfo( receiver_,
                                  pos_,
                                  time_ );
      }
};

#endif

/*-------------------------------------------------------------------*/
/*!
  \struct FreeformMessage
  \brief freeform message from coach/trainer
 */
struct FreeformMessage {
    GameTime time_; //!< received time
    std::string message_; //!< raw message string. this may be a CLang.

    FreeformMessage()
        : time_( -1, 0 )
      {
          message_.reserve( 8192 );
      }
};


/*-------------------------------------------------------------------*/
/*!
  \class AudioSensor
  \brief processor for players' communication
*/
class AudioSensor {
private:
    //! message encoder/decoder
//    AudioCodec M_codec;

    //! last updated time
    GameTime M_time;

    //! last received aural message from other player
//    PlayerMessage M_player_message;

    //! last received freeform message from coach;
//    FreeformMessage M_freeform_message;

    //! last received aural message from trainer
//    FreeformMessage M_trainer_message;


    // ball

    //! heard ball position
    Vector2D M_ball_pos;
    //! heard ball velocity
    Vector2D M_ball_vel;

    // pass

    //! heard pass receiver uniform number
    int M_receiver_number;
    //! heard pass receive position
    Vector2D M_receive_pos;

    // opponent goalie

    //! uniform number of opponent goalie
    int M_goalie_number;
    //! opponent goalie position
    Vector2D M_goalie_pos;

    // offside line
    double M_offside_line_x;

    // defense line
    double M_defense_line_x;


    // intercepter

    GameTime M_intercept_time;

    int M_intercept_number;

    int M_intercept_cycle;


    // hey pass

    GameTime M_hey_pass_time;

    int M_hey_pass_number;

    Vector2D M_hey_pass_pos;

    // player

    // int M_player_number;
    // Vector2D M_player_pos;
    // Vector2D M_player_vel;


    //! last received aural message from other player
    PlayerMessage M_player_message;

    //! message encoder/decoder
    AudioCodec M_codec;


    //! last received freeform message from coach;
    FreeformMessage M_freeform_message;

    //! last received aural message from trainer
    FreeformMessage M_trainer_message;

public:
    /*!
      \brief init member variables by default value
    */
    AudioSensor();

    /*!
      \brief analyze other player's audio message
      \param msg raw server message
      \param current game time when message is received
    */
    void parsePlayer( const char * msg,
                      const GameTime & current );

    /*!
      \brief analyze message from online coach
      \param msg raw server message
      \param current game time when message is received
     */
    void parseCoach( const char * msg,
                     const GameTime & current );

    /*!
      \brief analyze trainer's audio message
      \param msg raw server message
      \param current game time when message is received
    */
    void parseTrainer( const char * msg,
                       const GameTime & current );

public:

    /*!
      \brief get audio codec
      \return const reference to the AudioCodec instance
    */
    const
    AudioCodec & codec() const
      {
          return M_codec;
      }

    /*!
      \brief encode ball info to the message string to be sent by say command
      \param ball_pos position to be encoded
      \param ball_vel velocity to be encoded
      \param say_buf reference to the variable to store the encoded result
    */
    void encodeBall( const Vector2D & ball_pos,
                     const Vector2D & ball_vel,
                     std::string & say_buf ) const;

    /*!
      \brief encode pass info to the message string to be sent by say command
      \param receiver position to be encoded
      \param receive_pos estimated receive position
      \param say_buf reference to the variable to store the encoded result
    */
    void encodePass( const int receiver,
                     const Vector2D & receive_pos,
                     std::string & say_buf ) const;

    /*!
      \brief encode opponent goalie info to the message string to be sent by say command
      \param goalie_number uniform number of opponent goalie
      \param goalie_pos estimated goalie position
      \param say_buf reference to the variable to store the encoded result
    */
    void encodeGoalie( const int goalie_number,
                       const Vector2D & goalie_pos,
                       std::string & say_buf ) const;

    void encodeOffsideLineX( const double & offside_line_x,
                             std::string & say_buf ) const;

    void encodeDefenseLineX( const double & defense_line_x,
                             std::string & say_buf ) const;

    std::string getWaitMessage() const
      {
          return std::string( "w" );
      }

    std::string getAttackMessage() const
      {
          return std::string( "a" );
      }

    void encodeInterceptInfo( const int unum,
                              const int cycle,
                              std::string & say_buf ) const;

    void encodeHeyPassInfo( const int unum,
                            const Vector2D & pos,
                            std::string & say_buf ) const;


    /*!
      \brief get updated time
      \return const referncd to the game time
    */
    const
    GameTime & time() const
      {
          return M_time;
      }

    /*!
      \brief get the last received player message info
      \return const reference to the message object instance
     */
    const
    PlayerMessage & playerMessage() const
      {
          return M_player_message;
      }

    const
    FreeformMessage & freeformMessage() const
      {
          return M_freeform_message;
      }

    /*!
      \brief get the last received trainer message info
      \return const reference to the message object instance
     */
    const
    FreeformMessage & trainerMessage() const
      {
          return M_trainer_message;
      }

    //////////////////////////////////////
    /*!
      \brief get heard ball position
      \return position value
    */
    const
    Vector2D & ballPos() const
      {
          return M_ball_pos;
      }

    /*!
      \brief get heard ball velocity
      \return velocity value
    */
    const
    Vector2D & ballVel() const
      {
          return M_ball_vel;
      }

    //////////////////////////////////////

    /*!
      \brief get heard receiver number
      \return receiver's uniform number
    */
    int receiverNumber() const
      {
          return M_receiver_number;
      }

    /*!
      \brief get heard receive position
      \return position value
    */
    const
    Vector2D & receivePos() const
      {
          return M_receive_pos;
      }

    //////////////////////////////////////
    /*!
      \brief get heard goalie number
      \return goalie's uniform number
    */
    int goalieNumber() const
      {
          return M_goalie_number;
      }

    /*!
      \brief get heard goalie position
      \return position value
    */
    const
    Vector2D & goaliePos() const
      {
          return M_goalie_pos;
      }

    const
    double & offsideLineX() const
      {
          return M_offside_line_x;
      }

    const
    double & defenseLineX() const
      {
          return M_defense_line_x;
      }

    const
    GameTime & interceptTime() const
      {
          return M_intercept_time;
      }

    int interceptNumber() const
      {
          return M_intercept_number;
      }

    int interceptCycle() const
      {
          return M_intercept_cycle;
      }

    const
    GameTime & heyPassTime() const
      {
          return M_hey_pass_time;
      }

    int heyPassNumber() const
      {
          return M_hey_pass_number;
      }

    const
    Vector2D & heyPassPos() const
      {
          return M_hey_pass_pos;
      }

private:
    /*
      \brief analyze v8 format message from teammate
    */
    void parseTeammateAudioV8();

    /*!
      \brief analyze ball info message from teammate
      \param msg head of message body sent from teammate
      \return size of read byte, 0 means the invalid message
     */
    int parseBallInfo( const char * msg );

    /*!
      \brief analyze pass message from teammate
      \param msg head of message body sent from teammate
      \return size of read byte, 0 means the invalid message
    */
    int parsePassInfo( const char * msg );

    /*!
      \brief analyze opponent goalie message from teammate
      \param msg head of message body sent from teammate
      \return size of read byte, 0 means the invalid message
    */
    int parseGoalieInfo( const char * msg );

    /*!
      \brief analyze offside line info message from teammate
      \param msg head of message body sent from teammate
      \return size of read byte, 0 means the invalid message
    */
    int parseOffsideLineInfo( const char * msg );

    /*!
      \brief analyze defense line info message from teammate
      \param msg head of message body sent from teammate
      \return size of read byte, 0 means the invalid message
    */
    int parseDefenseLineInfo( const char * msg );

    //int parseWaitRequest( const char * msg );

    //int parseAttackRequest( const char * msg );

    /*!
      \brief analyze interception info message from teammate
      \param msg head of message body sent from teammate
      \return size of read byte, 0 means the invalid message
    */
    int parseInterceptInfo( const char * msg );

    /*!
      \brief analyze offside line info message from teammate
      \param msg head of message body sent from teammate
      \return size of read byte, 0 means the invalid message
    */
    int parseHeyPassInfo( const char * msg );

#ifdef _MSC_VER

public:
	/*!
      \brief write current world model to a file
      \param file handle
      \return true - success, false - failed.
	*/
	bool Write2StepDebugFile( FILE* pStepDebugFile ) const;

	/*!
      \brief read current world model 
      \param buffer handle
      \return true - success, false - failed.
	*/
	bool ReadFromDataFile( const char* pData ) const;

private:

	bool Write2StepDebugFile_info( FILE* pStepDebugFile ) const;
	bool ReadFromDataFile_info( const char* pData ) const;

#endif
};

}

#endif
