// -*-c++-*-

/*!
  \file audio_sensor.cpp
  \brief audio message analyzer Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "audio_sensor.h"
#include "logger.h"

#include <rcsc/common/server_param.h>
#include <rcsc/math_util.h>

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>

namespace rcsc {

/*-------------------------------------------------------------------*/
/*!

 */
AudioSensor::AudioSensor()
    : M_time( 0, 0 )
    , M_ball_pos( 0.0, 0.0 )
    , M_ball_vel( 0.0, 0.0 )
    , M_receiver_number( 0 )
    , M_receive_pos( 50.0, 0.0 )
    , M_goalie_number( Unum_Unknown )
    , M_goalie_pos( 50.0, 0.0 )
    , M_offside_line_x( 100000.0 )
    , M_defense_line_x( -100000.0 )
    , M_intercept_time( 0, 0 )
    , M_intercept_number( 0 )
    , M_intercept_cycle( 1000 )
    , M_hey_pass_time( 0, 0 )
    , M_hey_pass_number( 0 )
    , M_hey_pass_pos( 0.0, 0.0 )
{

}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::parsePlayer( const char * msg,
                          const GameTime & current )
{
    /*
      players' communication audio format

      // from self
      v7-: (hear <TIME> self <MSG>)
      v7+: (hear <TIME> self "<MSG>")

      // from other
      v7-: (hear <TIME> <DIR> <MSG>)
      v7:  (hear <TIME> <DIR> "<MSG>")
      v8+: (hear <TIME> <DIR> our <UNUM> "<MSG>")
      (hear <TIME> <DIR> opp "<MSG>")
      (hear <TIME> our)
      (hear <TIME> opp)
    */

    M_time = current;

    long cycle;
    double dir;
    int unum;
    int n_read = 0;

    // v8+ complete message
    if ( std::sscanf( msg, " (hear %ld %lf our %d %n ",
                      &cycle, &dir, &unum, &n_read ) != 3 )
    {
        std::cerr << "***ERROR*** AudioSensor::parse."
                  << " got partial or opponent audio?? ["
                  << msg << "]"
                  << std::endl;
        return;
    }

    msg += n_read;


    while ( *msg != '\0' && *msg == ' ' ) ++msg;
    if ( *msg == '\"' ) ++msg;

    std::size_t len = 0;
    while ( *(msg + len) != '\0'
            && *(msg + len) != '\"' )
    {
        ++len;
    }

    if ( len == 0 )
    {
        std::cerr << "***ERROR*** AudioSensor::parsePlayer."
                  << " Empty message "
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Empty message" );
        return;
    }

    M_player_message.message_.assign( msg, len );

    M_player_message.time_ = current;
    M_player_message.side_ = NEUTRAL;
    M_player_message.unum_ = unum;
    M_player_message.dir_ = dir;

    M_player_message.info_.clear();

    parseTeammateAudioV8();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::parseCoach( const char * msg,
                         const GameTime & current )
{
    // (hear <time> online_coach_{left,right} <msg>) : v7-
    // (hear <time> online_coach_{left,right} (freeform "<msg>")) : v7+
    // (hear <time> online_coach_{left,right} <clang>)   : v7+

    M_time = current;

    // skip "(hear "
    while ( *msg != ' ' && *msg != '\0' ) ++msg;
    while ( *msg == ' ' ) ++msg;

    // skip "<time> "
    while ( *msg != ' ' && *msg != '\0' ) ++msg;
    while ( *msg == ' ' ) ++msg;

    // skip sender
    while ( *msg != ' ' && *msg != '\0' ) ++msg;
    while ( *msg == ' ' ) ++msg;

    if ( *msg != '(' )
    {
        // not a clang message
        M_freeform_message.message_.assign( msg );
        if ( ! M_freeform_message.message_.empty()
             && *M_freeform_message.message_.rbegin() == ')' )
        {
            M_freeform_message.message_.erase( M_freeform_message.message_.length() - 1 );
        }

        return;
    }

    // clang message

    int n_read = 0;
    char msg_type[32];

    if ( std::sscanf( msg, "(%31[^ ] %n ",
                      msg_type, &n_read ) != 1 )
    {
        std::cerr << "***ERROR*** failed to parse clang message type. ["
                  << msg
                  << std::endl;
        return;
    }

    msg += n_read;

    if ( std::strcmp( msg_type, "freeform" ) != 0 )
    {
        // not a freeform message
        std::cerr << current << ": "
                  << "recv not a freeform message. "
                  << msg_type
                  << std::endl;
        return;
    }

    bool quoted = false;
    if ( *msg != '\0'
         && *msg == '\"' )
    {
        ++msg;
        quoted = true;
    }

    M_freeform_message.time_ = current;
    M_freeform_message.message_.assign( msg );

    // remove quotation
    if ( quoted )
    {
        std::string::size_type qpos = M_freeform_message.message_.find_last_of( '\"' );
        if ( qpos != std::string::npos )
        {
            M_freeform_message.message_.erase( qpos );
        }

        std::cerr << current << ": "
                  << "recv freeform message. ["
                  << M_freeform_message.message_ << ']'
                  << std::endl;
        return;
    }

    // remove last two paren
    for ( int i = 0; i < 2; ++i )
    {
        std::string::size_type ppos = M_freeform_message.message_.find_last_of( ')' );
        if ( ppos != std::string::npos )
        {
            M_freeform_message.message_.erase( ppos );
        }
    }
}


/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::parseTrainer( const char * msg,
                           const GameTime & current )
{

    // (hear <time> referee <msg>) : v7-
    // (hear <time> coach "<msg>")   : v7+
    // (hear <time> coach <clang>)   : v7+

    M_time = current;
    M_trainer_message.time_ = current;
    M_trainer_message.message_.erase();

    int n_read = 0;
    long cycle;
    char sender[32];

    if ( std::sscanf( msg, "(hear %ld %31s %n ",
                      &cycle, sender, &n_read ) != 2 )
    {
        std::cerr << "***ERRORR*** failed to parse trainer message. ["
                  << msg << ']'
                  << std::endl;
        return;
    }

    msg += n_read;

    bool quoted = false;
    if ( *msg != '\0'
         && *msg == '\"' )
    {
        ++msg;
        quoted = true;
    }

    M_trainer_message.message_.assign( msg );

    // remove quotation
    if ( quoted )
    {
        std::string::size_type qpos = M_trainer_message.message_.find_last_of( '\"' );
        if ( qpos != std::string::npos )
        {
            M_trainer_message.message_.erase( qpos );
        }
        return;
    }

    // remove last paren
    std::string::size_type ppos = M_trainer_message.message_.find_last_of( ')' );
    if ( ppos != std::string::npos )
    {
        M_trainer_message.message_.erase( ppos );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::parseTeammateAudioV8()
{
    if ( M_player_message.message_.empty() )
    {
        return;
    }

    const char * msg = M_player_message.message_.c_str();

    while ( *msg != '\0' )
    {
        int len = 0;
        PlayerMessage::Info info;

        switch ( *msg ) {
        case 'a': // attack request
            len = 1;
            info.type_ = PlayerMessage::ATTACK_REQUEST;
            break;
        case 'b': // ball info
            len = parseBallInfo( msg );
            if ( len == 0 ) break;
            info.type_ = PlayerMessage::BALL_INFO;
            break;
        case 'd': // defense line
            len = parseDefenseLineInfo( msg );
            if ( len == 0 ) break;
            info.type_ = PlayerMessage::DEFENSE_LINE_INFO;
            break;
        case 'g': // goalie info
            len = parseGoalieInfo( msg );
            if ( len == 0 ) break;
            info.type_ = PlayerMessage::OPPONENT_GOALIE_INFO;
            break;
        case 'h': // hey pass info
            len = parseHeyPassInfo( msg );
            if ( len == 0 ) break;
            info.type_ = PlayerMessage::HEY_PASS;
            break;
        case 'i': // intercept info
            len = parseInterceptInfo( msg );
            if ( len == 0 ) break;
            info.type_ = PlayerMessage::INTERCEPT_INFO;
            break;
        case 'o': // offside line
            len = parseOffsideLineInfo( msg );
            if ( len == 0 ) break;
            info.type_ = PlayerMessage::OFFSIDE_LINE_INFO;
            break;
        case 'p': // pass info
            len = parsePassInfo( msg );
            if ( len == 0 ) break;
            info.type_ = PlayerMessage::PASS_INFO;
            break;
        case 'w': // wait request
            len = 1;
            info.type_ = PlayerMessage::WAIT_REQUEST;
            break;
        default:
            break;
        }

        if ( len == 0 )
        {
            dlog.addText( Logger::SENSOR,
                          "AudioSensor: unsupported message? [%s] in [%s]",
                          msg, M_player_message.message_.c_str() );
            // assign the rest string
            M_player_message.message_ = msg;
            return;
        }

        info.str_.assign( msg, 0, len );
        M_player_message.info_.push_back( info );

        msg += len;
    }

}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodeBall( const Vector2D & ball_pos,
                         const Vector2D & ball_vel,
                         std::string & say_buf ) const
{
    if ( (int)say_buf.length() + 6 > ServerParam::i().playerSayMsgSize() )
    {
        dlog.addText( Logger::SENSOR,
                      "AudioSensor::encodeBallInfo(). over the message size = %d",
                      say_buf.length() );
        return;
    }

    std::string msg = codec().encodePosVelToStr6( ball_pos, ball_vel );
    if ( msg.empty() )
    {
        return;
    }

    say_buf += 'b';
    say_buf += msg;
}

/*-------------------------------------------------------------------*/
/*!

 */
int
AudioSensor::parseBallInfo( const char * msg )
{
    if ( std::strlen( msg ) < 7 )
    {
        std::cerr << "***ERROR*** AudioSensor::parseTeammateAudioV8"
                  << " Illega ball info [" << msg << "]"
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Illegal ball info [%s]",
                      msg );
        return 0;
    }

    std::string ball_str( msg, 1, 6 );

    if ( ! codec().decodeStr6ToPosVel( ball_str,
                                       &M_ball_pos, &M_ball_vel ) )
    {
        std::cerr << "***ERROR*** AudioSensor::parseTeammateAudioV8"
                  << " Failed to decode ball [" << msg
                  << "]" << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Ball Info [%s]",
                      msg );
        return 0;
    }

    dlog.addText( Logger::SENSOR,
                  "AudioSensor::parseBallInfo() success! pos(%.1f %.1f) vel(%.1f %.1f)",
                  M_ball_pos.x, M_ball_pos.y,
                  M_ball_vel.x, M_ball_vel.y );

    return 7;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodePass( const int receiver,
                         const Vector2D & receive_pos,
                         std::string & say_buf ) const
{
    if ( (int)say_buf.length() + 6 > ServerParam::i().playerSayMsgSize() )
    {
        dlog.addText( Logger::SENSOR,
                      "AudioSensor::encodePassInfo(). over the message size = %d",
                      say_buf.length() );
        return;
    }

    char hex = AudioCodec::unum2hex( receiver );
    if ( hex == '\0' )
    {
        std::cerr << "AudioSensor::encodePass. "
                  << "Unexpected receiver number "
                  << receiver << std::endl;
        return;
    }

    std::string pos_str = codec().encodePosToStr4( receive_pos );
    if ( pos_str.empty() )
    {
        std::cerr << "AudioSensor::encode XY error "
                  << receive_pos << " -> " << pos_str
                  << std::endl;
        return;
    }

    say_buf += 'p';
    say_buf += hex;
    say_buf += pos_str;
}

/*-------------------------------------------------------------------*/
/*!

 */
int
AudioSensor::parsePassInfo( const char * msg )
{
    if ( std::strlen( msg ) < 6
         || msg[0] != 'p' )
    {
        std::cerr << "AudioSensor::parsePassInfo. "
                  << "Unexpected pass message ["
                  << msg << "] len = " << std::strlen( msg )
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Pass Info [%s]",
                      msg );
        return 0;
    }
    ++msg;

    M_receiver_number = AudioCodec::hex2unum( *msg );
    if ( M_receiver_number == Unum_Unknown )
    {
        std::cerr << "AudioSensor::parsePassInfo. "
                  << "Unexpected receiver number. "
                  << msg
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Pass Info [%s]",
                      msg );
        return 0;
    }
    ++msg;

    M_receive_pos = codec().decodeStr4ToPos( msg );

    if ( ! M_receive_pos )
    {
        std::cerr << "AudioSensor::parsePassInfo. "
                  << "Unexpected value " << msg
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Pass Info [%s]",
                      msg );
        return 0;
    }

    dlog.addText( Logger::SENSOR,
                  "AudioSensor::parsePassInfo() success! receiver= %d  receive_pos(%f %f)",
                  M_receiver_number,
                  M_receive_pos.x, M_receive_pos.y );
    return 6;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodeGoalie( const int goalie_number,
                           const Vector2D & goalie_pos,
                           std::string & say_buf ) const
{
    if ( (int)say_buf.length() + 6 > ServerParam::i().playerSayMsgSize() )
    {
        dlog.addText( Logger::SENSOR,
                      "AudioSensor::encodeGoalieInfo(). over the message size = %d",
                      say_buf.length() );
        return;
    }

    char hex = AudioCodec::unum2hex( goalie_number );
    if ( hex == '\0' )
    {
        std::cerr << "AudioSensor::encodeGoalieInfo. "
                  << "Unexpected goalie number "
                  << goalie_number << std::endl;
        return;
    }

    std::string pos_str = codec().encodePosToStr4( goalie_pos );
    if ( pos_str.empty() )
    {
        std::cerr << "AudioSensor::encodeGoalieInfo. "
                  << "Failed to encode goalie info "
                  << "( " << goalie_pos.x << goalie_pos.y << ")"
                  << std::endl;
        return;
    }

    say_buf += 'g';
    say_buf += hex;
    say_buf += pos_str;
}

/*-------------------------------------------------------------------*/
/*!

 */
int
AudioSensor::parseGoalieInfo( const char * msg )
{
    if ( std::strlen( msg ) < 6
         || msg[0] != 'g' )
    {
        std::cerr << "AudioSensor::parseGoalieInfo. "
                  << "Unexpected goalie message ["
                  << msg << "] len = " << std::strlen( msg )
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Goalie Info [%s]",
                      msg );
        return 0;
    }
    ++msg;

    M_goalie_number = AudioCodec::hex2unum( *msg );
    if ( M_goalie_number == Unum_Unknown )
    {
        std::cerr << "AudioSensor::parseGoalieInfo. "
                  << "Unexpected goalie number " << msg
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Goalie Info [%s]",
                      msg );
        return 0;
    }
    ++msg;

    M_goalie_pos = codec().decodeStr4ToPos( msg );

    if ( ! M_goalie_pos )
    {
        std::cerr << "AudioSensor::parseGoalieInfo. "
                  << "Unexpected value " << msg
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Goalie Info [%s]",
                      msg );
        return 0;
    }

    dlog.addText( Logger::SENSOR,
                  "AudioSensor::parseGoalieInfo() success! goalie pos = (%.2f %.2f)",
                  M_goalie_pos.x, M_goalie_pos.y );
    return 6;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodeOffsideLineX( const double & offside_line_x,
                                 std::string & say_buf ) const
{
    if ( (int)say_buf.length() + 3 > ServerParam::i().playerSayMsgSize() )
    {
        dlog.addText( Logger::SENSOR,
                      "AudioSensor::encodeOffsideLineX(). over the message size = %d",
                      say_buf.length() );
        return;
    }

    std::string msg = codec().encodeCoordToStr2( offside_line_x,
                                                 AudioCodec::X_NORM_FACTOR );
    if ( msg.empty() )
    {
        std::cerr << "AudioSensor::encodeOffsideLineX. "
                  << " Failed to encode. value = " << offside_line_x
                  << std::endl;
    }

    say_buf += 'o';
    say_buf += msg;
}

/*-------------------------------------------------------------------*/
/*!

 */
int
AudioSensor::parseOffsideLineInfo( const char * msg )
{
    if ( std::strlen( msg ) < 3
         || msg[0] != 'o' )
    {
        std::cerr << "AudioSensor::parseOffsideLineInfo. "
                  << " Unexpected message ["
                  << msg << "]"
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Offside Line Info [%s]",
                      msg );
        return 0;
    }
    ++msg;

    double val = codec().decodeStr2ToCoord( msg[0], msg[1],
                                            AudioCodec::X_NORM_FACTOR );
    if ( val == AudioCodec::ERROR_VALUE )
    {
        std::cerr << "AudioSensor::parseOffsideLineInfo. "
                  << " Failed to read offside line"
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Offside Line Info [%s]",
                      msg );
        return 0;
    }

    M_offside_line_x = val;

    dlog.addText( Logger::SENSOR,
                  "AudioSensor::parseOffsideLineInfo() success! x=%.1f",
                  val );

    return 3;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodeDefenseLineX( const double & defense_line_x,
                                 std::string & say_buf ) const
{
    if ( (int)say_buf.length() + 3 > ServerParam::i().playerSayMsgSize() )
    {
        dlog.addText( Logger::SENSOR,
                      "AudioSensor::encodeBallInfo(). over the message size = %d",
                      say_buf.length() );
        return;
    }

    std::string msg = codec().encodeCoordToStr2( defense_line_x,
                                                 AudioCodec::X_NORM_FACTOR );
    if ( msg.empty() )
    {
        std::cerr << "AudioSensor::encodeDefenseLineX. "
                  << " Failed to encode. value = " << defense_line_x
                  << std::endl;
    }

    say_buf += 'd';
    say_buf += msg;
}

/*-------------------------------------------------------------------*/
/*!

 */
int
AudioSensor::parseDefenseLineInfo( const char * msg )
{
    if ( std::strlen( msg ) < 3
         || msg[0] != 'd' )
    {
        std::cerr << "AudioSensor::parseDefenseLineInfo. "
                  << " Unexpected message ["
                  << msg << "]"
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Defense Line Info [%s]",
                      msg );
        return 0;
    }
    ++msg;

    double val = codec().decodeStr2ToCoord( msg[0], msg[1],
                                            AudioCodec::X_NORM_FACTOR );
    if ( val == AudioCodec::ERROR_VALUE )
    {
        std::cerr << "AudioSensor::parseDefenseLineInfo. "
                  << " Failed to read offside line"
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Defense Line Info [%s]",
                      msg );
        return 0;
    }

    M_defense_line_x = val;

    dlog.addText( Logger::SENSOR,
                  "AudioSensor::parseDefenseLineInfo() success! x=%.1f",
                  val );

    return 3;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodeInterceptInfo( const int unum,
                                  const int cycle,
                                  std::string & say_buf ) const
{
    if ( (int)say_buf.length() + 3 > ServerParam::i().playerSayMsgSize() )
    {
        dlog.addText( Logger::SENSOR,
                      "AudioSensor::encodeInterceptInfo(). over the message size = %d",
                      say_buf.length() );
        return;
    }

    char hex = AudioCodec::unum2hex( unum );
    if ( hex == '\0' )
    {
        std::cerr << "AudioSensor::encodeInterceptInfo. "
                  << "Unexpected number "
                  << unum << std::endl;
        return;
    }

    try
    {
        char ch = codec().intToCharMap().at( cycle );

        say_buf += 'i';
        say_buf += hex;
        say_buf += ch;
    }
    catch ( ... )
    {
        std::cerr << "AudioSensor::encodeInterceptInfo."
                  << " Exception caught! Failed to encode cycle = "
                  << cycle
                  << std::endl;
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
int
AudioSensor::parseInterceptInfo( const char * msg )
{
    if ( std::strlen( msg ) < 3
         || msg[0] != 'i' )
    {
        std::cerr << "AudioSensor::parseInterceptInfo. "
                  << " Unexpected message = ["
                  << msg << "]"
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode intercept info [%s]",
                      msg );
        return 0;
    }
    ++msg;

    int number = AudioCodec::hex2unum( *msg );
    if ( number == Unum_Unknown )
    {
        std::cerr << "AudioSensor::parseInterceptInfo. "
                  << " Invalid player number. message = ["
                  << msg << "]"
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode intercept info [%s]",
                      msg );
        return 0;
    }
    ++msg;

    AudioCodec::CharToIntCont::const_iterator cycle
        = codec().charToIntMap().find( *msg );
    if ( cycle == codec().charToIntMap().end() )
    {
        std::cerr << "AudioSensor::parseInterceptInfo. "
                  << " Invalid cycle. message = ["
                  << msg << "]"
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode intercept info [%s]",
                      msg );
        return 0;
    }

    M_intercept_time = M_time;
    M_intercept_number = number;
    M_intercept_cycle = cycle->second;

    dlog.addText( Logger::SENSOR,
                  "AudioSensor::parseInterceptInfo() success! unum=%d cycle=%d",
                  number, cycle->second );

    return 3;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodeHeyPassInfo( const int unum,
                                const Vector2D & pos,
                                std::string & say_buf ) const
{
    if ( (int)say_buf.length() + 6 > ServerParam::i().playerSayMsgSize() )
    {
        dlog.addText( Logger::SENSOR,
                      "AudioSensor::encodeHeyPassInfo(). over the message size = %d",
                      say_buf.length() );
        return;
    }

    char hex = AudioCodec::unum2hex( unum );
    if ( hex == '\0' )
    {
        std::cerr << "AudioSensor::encodeHeyPassInfo. "
                  << "Failed to encode hey pass info. unum="
                  << unum
                  << " ( " << pos.x << pos.y << ")"
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to encode hey pass info" );
        return;
    }

    std::string pos_str = codec().encodePosToStr4( pos );
    if ( pos_str.length() != 4 )
    {
        std::cerr << "AudioSensor::encodeHeyPassInfo. "
                  << "Failed to encode hey pass info. unum="
                  << unum
                  << " ( " << pos.x << pos.y << ")"
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to encode hey pass info" );
        say_buf.erase();
        return;
    }

    say_buf += 'h';
    say_buf += hex;
    say_buf += pos_str;
}

/*-------------------------------------------------------------------*/
/*!

 */
int
AudioSensor::parseHeyPassInfo( const char * msg )
{
    if ( std::strlen( msg ) < 6
         || msg[0] != 'h' )
    {
        std::cerr << "AudioSensor::parseHeyPassInfo. "
                  << "Unexpected hey pass message ["
                  << msg << "] len = " << std::strlen( msg )
                  << std::endl;
        return 0;
    }
    ++msg;

    int number = AudioCodec::hex2unum( *msg );
    if ( number == Unum_Unknown )
    {
        std::cerr << "AudioSensor::parseGoalieInfo. "
                  << "Unexpected goalie number " << msg
                  << std::endl;
        return 0;
    }
    ++msg;

    Vector2D pos = codec().decodeStr4ToPos( msg );
    if ( ! pos )
    {
        std::cerr << "AudioSensor::parseHeyPassInfo. "
                  << " Failed to decode hey pass potiiton"
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode hey pass potiiton" );
        return 0;
    }

    M_hey_pass_number = number;
    M_hey_pass_pos = pos;
    M_hey_pass_time = M_time;

    dlog.addText( Logger::SENSOR,
                  "AudioSensor::parseHeyPassInfo() success! hey pass pos = (%.2f %.2f)",
                  M_hey_pass_pos.x, M_hey_pass_pos.y );

    return 0;
}

}
