﻿//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		CSTypeHelper.cs
 * @brief		Type Helper ファイル
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2013-2014 Takazumi Shirayanagi\n
 * This software is released under the MIT License,
 * see LICENSE
*/
//-----------------------------------------------------------------------
//======================================================================

//======================================================================
// using
using System;
using System.Reflection;
using System.Collections;

namespace Common
{
	public class CSTypeHelper
	{
		#region IsBaseType
		/// <summary>
		/// 継承関係にあるかどうか
		/// </summary>
		/// <param name="type"></param>
		/// <param name="base_type"></param>
		/// <returns></returns>
		public static bool IsBaseType(Type type, Type base_type)
		{
			if (type.BaseType != null)
			{
				if (type.BaseType == base_type)
					return true;
				if (IsBaseType(type.BaseType, base_type))
					return true;
			}
			return false;
		}
		#endregion

		#region IsDerived
		/// <summary>
		/// 継承関係にあるかどうか（自身も含む）
		/// </summary>
		/// <param name="type"></param>
		/// <param name="base_type"></param>
		/// <returns></returns>
		public static bool IsDerived(Type type, Type base_type)
		{
			if (type == base_type) return true;
			return IsBaseType(type, base_type);
		}
		#endregion

		#region IsDisposable
		/// <summary>
		/// IDisposable を継承しているか
		/// </summary>
		/// <param name="type"></param>
		/// <returns></returns>
		public static bool IsDisposable(Type type)
		{
			if (type.BaseType != null)
			{
				if (IsDisposable(type.BaseType))
				{
					if (type.GetMethod("Dispose") == null)
						return false;	// 内部で完結してるので false を返す
					return true;
				}
			}

			Type[] infs = type.GetInterfaces();
			foreach (Type it in infs)
			{
				if (it == typeof(IDisposable) || it.IsSubclassOf(typeof(IDisposable)))
				{
					return true;
				}
			}
			return false;
		}
		#endregion

		#region IsHasDisposable
		/// <summary>
		/// IDisposable なメンバがあるかどうか
		/// </summary>
		/// <param name="type"></param>
		/// <returns></returns>
		public static bool IsHasDisposable(Type type)
		{
			FieldInfo[] fis = type.GetFields(BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.Instance);
			foreach (FieldInfo fi in fis)
			{
				if (IsDisposable(fi.FieldType))
				{
					return true;
				}
			}
			return false;
		}
		#endregion

		#region インスタンスの検索
		/// <summary>
		/// インスタンスの検索
		/// </summary>
		/// <param name="asm"></param>
		/// <param name="type"></param>
		/// <returns></returns>
		public static object FindInstance(Assembly asm, Type type)
		{
			if( asm == null || type == null )
				return null;

			Type[] types = asm.GetTypes();
			try
			{
				foreach (Type t in types)
				{
					object ret = FindInstance(t, type);
					if (ret != null) return ret;
				}
			}
			catch (Exception e)
			{
				CSConsole.WriteLine(e.ToString());
				return null;
			}
			return null;
		}

		public static object FindInstance(Type target, Type type)
		{
			if (target == null || type == null) return null;
			try
			{
				FieldInfo[] fis = target.GetFields(BindingFlags.NonPublic | BindingFlags.Public | BindingFlags.Static);
				foreach (FieldInfo fi in fis)
				{
					if (fi.FieldType == type)
					{
						return fi.GetValue(target);
					}
				}
			}
			catch (Exception e)
			{
				CSConsole.WriteLine(e.ToString());
				return null;
			}
			return null;
		}
		#endregion

		#region インスタンスの取得
		public static object GetInstance(Assembly asm, Type type)
		{
			if (asm == null || type == null)
				return null;

			Type[] types = asm.GetTypes();
			try
			{
				foreach (Type t in types)
				{
					object ret = GetInstance(t, type);
					if (ret != null) return ret;
				}
			}
			catch (Exception e)
			{
				CSConsole.WriteLine(e.ToString());
				return null;
			}
			return null;
		}
		public static object GetInstance(Type target, Type type)
		{
			if (target == null || type == null) return null;
			try
			{
				object ret = null;// FindInstance(target, type);
				if (ret != null) return ret;
				MethodInfo[] mis = target.GetMethods(BindingFlags.NonPublic | BindingFlags.Public | BindingFlags.Static);
				foreach (MethodInfo mi in mis)
				{
					if (mi.ReturnType == type)
					{
						return mi.Invoke(null, null);
					}
				}
			}
			catch (Exception e)
			{
				CSConsole.WriteLine(e.ToString());
				return null;
			}
			return null;
		}
		#endregion

		#region EnumMember
		public static Type[] EnumMember(Type type, Type target)
		{
			ArrayList al = new ArrayList();
			FieldInfo[] fis = type.GetFields(BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.Instance);
			foreach (FieldInfo fi in fis)
			{
				if (IsDerived(fi.FieldType, target))
				{
					al.Add(fi.FieldType);
				}
			}
			return (Type[])(al.ToArray(typeof(Type)));
		}

		#endregion

		#region HasDisposable
		/// <summary>
		/// IDisposable なメンバの列挙
		/// </summary>
		/// <param name="type"></param>
		/// <returns></returns>
		public static Type[] HasDisposable(Type type)
		{
			ArrayList al = new ArrayList();
			FieldInfo[] fis = type.GetFields(BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.Instance);
			foreach (FieldInfo fi in fis)
			{
				if (IsDisposable(fi.FieldType))
				{
					al.Add(fi.FieldType);
				}
			}
			return (Type[])(al.ToArray(typeof(Type)));
		}
		#endregion
	}

}	// end of namespace Common

