﻿//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		CSFileStream.cs
 * @brief		FileStream ファイル
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2013-2014 Takazumi Shirayanagi\n
 * This software is released under the MIT License,
 * see LICENSE
*/
//-----------------------------------------------------------------------
//======================================================================

//======================================================================
// using
using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Windows.Forms;
using System.Drawing;
using System.Diagnostics;
using System.Reflection;
using System.Resources;
using System.Text.RegularExpressions;
using Common;
using win32;

namespace rcsharp
{
	namespace RES
	{
		#region WS
		/// <summary>
		/// リソース出力用にカスタムした Window Style
		/// </summary>
		[Flags]
		public enum WS : long
		{
			WS_POPUP = 0x80000000L,
			WS_CHILD = 0x40000000L,
			WS_MINIMIZE = 0x20000000L,
			WS_VISIBLE = 0x10000000L,
			WS_DISABLED = 0x08000000L,
			WS_CLIPSIBLINGS = 0x04000000L,
			WS_CLIPCHILDREN = 0x02000000L,
			WS_MAXIMIZE = 0x01000000L,
			WS_CAPTION = 0x00C00000L,		/* WS_BORDER | WS_DLGFRAME	*/
			WS_BORDER = 0x00800000L,
			WS_DLGFRAME = 0x00400000L,
			WS_VSCROLL = 0x00200000L,
			WS_HSCROLL = 0x00100000L,
			WS_SYSMENU = 0x00080000L,
			WS_THICKFRAME = 0x00040000L,
			WS_GROUP = 0x00020000L,
			WS_TABSTOP = 0x00010000L,

			WS_MINIMIZEBOX = 0x00020000L,
			WS_MAXIMIZEBOX = 0x00010000L,

			WS_OVERLAPPEDWINDOW = (	WS_CAPTION |
									WS_SYSMENU |
									WS_THICKFRAME |
									WS_MINIMIZEBOX |
									WS_MAXIMIZEBOX),

			WS_POPUPWINDOW = (WS_POPUP |
									WS_BORDER |
									WS_SYSMENU),

			WS_CHILDWINDOW = WS_CHILD,


			WS_ICONIC = WS_MINIMIZE,
			WS_SIZEBOX = WS_THICKFRAME,
			WS_TILEDWINDOW = WS_OVERLAPPEDWINDOW,

			MASK = 0xFFFF0000L,
		}
		#endregion

        #region ダイアログID
        public enum DLGID
        {
            IDOK = 1,
            IDCANCEL,
            IDABORT,
            IDRETRY,
            IDIGNORE,
            IDYES,
            IDNO,
            IDCLOSE,
            IDHELP,
            IDTRYAGAIN,
            IDCONTINUE,
            IDTIMEOUT = 32000,
        }
        #endregion
	}

	#region リソーススクリプトクラス
	/// <summary>
	/// リソーススクリプトファイルクラス
	/// </summary>
	class CSRCFileStream : CSFileStream
	{
		#region コンストラクタ
		public CSRCFileStream()
		{
		}
		#endregion

        #region コントロール
        public enum DLGSCRIPTTYPE
        {
            CONTROL,
            DEFPUSHBUTTON,
            PUSHBUTTON,
            CHECKBOX,
            RADIOBUTTON,
            COMBOBOX,
            LISTBOX,
            LTEXT,
            EDITTEXT,
            SCROLLBAR,
            GROUPBOX,
            UNKOWN
        }
        #endregion

        #region DLU
        public static int DLUX(int x, Form form)
		{
			return DLUX(x, form.Font);
		}
		public static int DLUY(int y, Form form)
		{
			return DLUY(y, form.Font);
		}
		public static int DLUX(int x, Font font)
		{
			return (int)(x / (CSFont.AveCharWidth(font) / 4.0f));
		}
		public static int DLUY(int y, Font font)
		{
			//int h = CSFont.AveCharHeight(form.Font);
			float h = font.GetHeight() / 8;
			return (int)(y / h);
		}
		#endregion

        #region 位置取得
        public static void GetFormRelativePosition(Form form, Control ctrl, out int left, out int top)
        {
            left = ctrl.Left;
            top = ctrl.Top;
            Control curr = ctrl.Parent;
            while (curr != null && curr != form)
            {
                left += curr.Left;
                top += curr.Top;
                curr = curr.Parent;
            }
        }
        #endregion

        #region プロパティ

        #region Separator
        public static string Separator
		{
			get { return "/////////////////////////////////////////////////////////////////////////////"; }
		}
		#endregion

		#endregion

		#region field
		public const string CRLF = "\r\n";
		public const string SEPARATE = "/////////////////////////////////////////////////////////////////////////////" + CRLF;

		public const string RESCOMMENT = SEPARATE + "//" + CRLF + "// NAME" + CRLF + "//" + CRLF + CRLF;
		#endregion
	};
	#endregion

    #region リソースID 管理
    public class CSResID
	{
		#region フィールド
        private SortedDictionary<int, string> m_ResIDs = new SortedDictionary<int, string>();
		private int m_IDD = 100;
		private int m_IDC = 1000;
		private int m_IDM = 6000;
		private int m_IDS = 10000;	// original
		#endregion

		#region コンストラクタ
		public CSResID() { }
		public CSResID(int IDD, int IDC, int IDS, int IDM) 
		{
			m_IDD = IDD;
			m_IDC = IDC;
			m_IDS = IDS;
			m_IDM = IDM;
		}
		#endregion

		#region ファイル出力
		public bool DumpFile(string path, string using_rc)
		{
			Sort();

			try
			{
				string buf = "";
				#region ヘッダ
				CSFileStream fs = new CSFileStream(path, FileMode.Create);
				fs.Write(HEADER, 0);

				buf += "// Used by " + Path.GetFileName(using_rc) + CRLF + "//" + CRLF;
				fs.Write(buf, 0);
				#endregion

				#region IDs
				foreach (KeyValuePair<int, string> ids in m_ResIDs)
				{
					string id = ids.Value;
					buf = "#define " + id;
					while (buf.Length < 40) buf += " ";
					buf += ids.Key;
					buf += CRLF;
					fs.Write(buf, 0);
				}
				#endregion

				#region IDC_STATIC
				{
					buf = "#ifndef IDC_STATIC";
					buf += CRLF;
					buf += "#define IDC_STATIC              -1";
					buf += CRLF;
					buf += "#endif";
					buf += CRLF;
					fs.Write(buf, 0);
				}
				#endregion

				#region フッター
				buf = CRLF + CRLF + "// Next default values for new objects" + CRLF + "// " + CRLF + "#ifdef APSTUDIO_INVOKED" + CRLF
					 + "#ifndef APSTUDIO_READONLY_SYMBOLS" + CRLF + "#define _APS_NEXT_RESOURCE_VALUE        " + m_IDD + CRLF
					 + "#define _APS_NEXT_COMMAND_VALUE         " + m_IDM + CRLF
					 + "#define _APS_NEXT_CONTROL_VALUE         " + m_IDC + CRLF
					 + "#define _APS_NEXT_SYMED_VALUE           " + m_IDD + CRLF
					 + "#endif" + CRLF + "#endif" + CRLF;
				fs.Write(buf, 0);
				#endregion
			}
			catch (Exception e)
			{
				CSConsole.Write(e.ToString());
				return false;
			}
			return true;
		}
		#endregion

		#region プロパティ
        public SortedDictionary<int, string> IDs
		{
			get { return m_ResIDs; }
		}
		#endregion

		#region Sort
		public void Sort()
		{
			//m_ResIDs.Sort(CompareIDs);
		}
		#endregion

		#region ID 追加
		public void Add(string strID)
		{
			if (strID == null || strID.Length == 0) return;

			// 既存の場合は増やさない
			if (m_ResIDs.ContainsValue(strID)) return;
            // 定義済み
            foreach (RES.DLGID id in Enum.GetValues(typeof(RES.DLGID)))
            {
                if (id.ToString() == strID) return;
            }

            int idnum = -1;
            if ((strID.IndexOf("IDD_") == 0) || (strID.IndexOf("IDR_") == 0) || (strID.IndexOf("IDI_") == 0) || (strID.IndexOf("IDB_") == 0)) idnum = m_IDD++;
			else if (strID.IndexOf("IDC_") == 0) idnum = m_IDC++;
			else if (strID.IndexOf("IDM_") == 0) idnum = m_IDM++;
            else if (strID.IndexOf("IDS_") == 0) idnum = m_IDS++;
            else
            {
                // 該当なしの場合は、IDC系統にしておく
                idnum = m_IDC++;
            }

			m_ResIDs.Add(idnum, strID);
		}
		#endregion

        #region ID String 生成
        public static string MakeIDString(string prefix, string name)
        {
            string idname = CSCharctors.StripAlphaNumber(name.ToUpper());
            Regex regex = new Regex("^ID[A-Z]_");
            if (regex.IsMatch(idname)) return idname;
            foreach (RES.DLGID id in Enum.GetValues(typeof(RES.DLGID)))
            {
                if (id.ToString() == idname) return idname;
            }
            return prefix + idname;
        }
        #endregion

        #region sort function
        private static int CompareIDs(KeyValuePair<int, string> x, KeyValuePair<int, string> y)
		{
            if (x.Key == y.Key) return 0;
            if (x.Key > y.Key) return 1;
			return -1;
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		private string HEADER = "//{{NO_DEPENDENCIES}}" + CRLF + "// " + Application.ProductName + " generated include file." + CRLF;
		#endregion
	}
	#endregion

	#region ダイアログリソース
	/// <summary>
	/// ダイアログリソース
	/// </summary>
	class CSRCDialog
	{
		#region Serialize
		public static bool Serialize(Form form, CSResID resIDs, out string buf)
		{
			buf = "";
			if (form == null) return false;

			try
			{
				// IDD
				buf += GetIDDLineString(form, resIDs);

				// STYLE
				buf += GetStyleString(form);

                // CAPTION
                buf += GetCaptionString(form);

				// MENU
				buf += GetMenuString(form);

				// FONT
				buf += GetFontString(form);

				// CONTROLS
				buf += GetContorlsString(form, resIDs);

				buf += CRLF;
			}
			catch
			{
				buf = "";
				return false;
			}
			return true;
		}
		#endregion

		#region GetIDDLineString
		private static string GetIDDLineString(Form form, CSResID resIDs)
		{
            string name = form.GetType().Name;

			string strIDD = CSResID.MakeIDString("IDD_", name);

			// リソースID 追加
			resIDs.Add(strIDD);

			strIDD += " DIALOGEX 0, 0, ";
			strIDD += CSRCFileStream.DLUX(form.Width, form);
			strIDD += ", ";
			strIDD += CSRCFileStream.DLUY(form.Height, form);
			strIDD += CRLF;
			return strIDD;
		}
		#endregion

		#region GetStyleString
		private static string GetStyleString(Form form)
		{
			string buf = "STYLE ";
			RES.WS style = (RES.WS)API.GetWindowLong(form, API.GWL.GWL_STYLE);
			API.DS dstyle = (API.DS)style & API.DS.MASK;
			style &= RES.WS.MASK;
			string wsname = style.ToString();
			string dsname = dstyle == 0 ? "" : dstyle.ToString();
			if (dsname.Length > 0 && wsname.Length > 0) wsname += "| ";
			buf += wsname.Replace(',', '|');
			buf += dsname.Replace(',', '|');
			buf += CRLF;
			return buf;
		}
		#endregion

        #region GetCaptionString
        private static string GetCaptionString(Form form)
        {
            string buf = "CAPTION ";
            buf += "\"" + form.Text + "\"";
            buf += CRLF;
            return buf;
        }
        #endregion
        
        #region GetMenuString
		private static string GetMenuString(Form form)
		{
			string buf = "MENU ";
			string name = "";
			if (form.Menu == null)
			{
				if (form.MainMenuStrip == null)
				{
					return "";
				}
				name = form.MainMenuStrip.Name;
			}
			else
			{
				name = form.Menu.Name;
			}
			buf += CSResID.MakeIDString("IDR_", name);
			buf += CRLF;
			return buf;
		}
		#endregion

		#region GetFontString
		private static string GetFontString(Form form)
		{
			string buf = "FONT ";
			buf += form.Font.SizeInPoints;
			buf += ", \"" + form.Font.Name + "\", ";
			buf += form.Font.Bold ? "700" : "400" + ", ";
			buf += form.Font.Italic ? "1" : "0" + ", ";
			buf += "0x" + form.Font.GdiCharSet.ToString("X");
			buf += CRLF;
			return buf;
		}
		#endregion

		#region GetContorlsString
		private static string GetContorlsString(Form form, CSResID resIDs)
		{
			string buf = "BEGIN" + CRLF;

			FieldInfo[] fis = form.GetType().GetFields(BindingFlags.NonPublic | BindingFlags.Public | BindingFlags.Instance);
			foreach (FieldInfo fi in fis)
			{
				string str = "";
				if (!CSTypeHelper.IsBaseType(fi.FieldType, typeof(Control))) continue;

                string name = fi.Name;
				object obj = null;
				obj = fi.GetValue(form);

                #region MenuStrip
                if (CSTypeHelper.IsDerived(fi.FieldType, typeof(MenuStrip))
                    || CSTypeHelper.IsDerived(fi.FieldType, typeof(ContextMenuStrip)))
                {
                    continue;
                }
                #endregion

                #region Control List

                #region ButtonBase
                if (CSTypeHelper.IsDerived(fi.FieldType, typeof(ButtonBase)))
				{
					if (!CSRCButton.Serialize(form, (ButtonBase)obj, name, resIDs, out str)) continue;
				}
				#endregion
				#region ComboBox
				else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(ComboBox)))
				{
					if (!CSRCComboBox.Serialize(form, (ComboBox)obj, name, resIDs, out str)) continue;
				}
				#endregion
				#region ListBox
				else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(ListBox)))
				{
					if (!CSRCListBox.Serialize(form, (ListBox)obj, name, resIDs, out str)) continue;
				}
				#endregion
				#region Label
				else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(Label)))
				{
					if (CSTypeHelper.IsDerived(fi.FieldType, typeof(LinkLabel)))
					{
						if (!CSRCLinkLabel.Serialize(form, (LinkLabel)obj, name, resIDs, out str)) continue;
					}
					else
					{
						if (!CSRCLabel.Serialize(form, (Label)obj, name, resIDs, out str)) continue;
					}
				}
				#endregion
				#region TextBoxBase
				else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(TextBoxBase)))
				{
					if (CSTypeHelper.IsDerived(fi.FieldType, typeof(TextBox)))
					{
						if (!CSRCTextBox.Serialize(form, (TextBox)obj, name, resIDs, out str)) continue;
					}
					else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(RichTextBox)))
					{
						if (!CSRCRichTextBox.Serialize(form, (RichTextBox)obj, name, resIDs, out str)) continue;
					}
                    else
                    {
                        if (!CSRCTextBox.Serialize(form, (TextBoxBase)obj, name, resIDs, out str)) continue;
                    }
				}
				#endregion
				#region ScrollBar
				else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(ScrollBar)))
				{
					if (!CSRCScrollBar.Serialize(form, (ScrollBar)obj, name, resIDs, out str)) continue;
				}
				#endregion
				#region GroupBox
				else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(GroupBox)))
				{
					if (!CSRCGroupBox.Serialize(form, (GroupBox)obj, name, resIDs, out str)) continue;
				}
				#endregion
				#region ProgressBar
				else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(ProgressBar)))
				{
					if (!CSRCProgressBar.Serialize(form, (ProgressBar)obj, name, resIDs, out str)) continue;
				}
				#endregion
				#region ListView
				else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(ListView)))
				{
					if (!CSRCListView.Serialize(form, (ListView)obj, name, resIDs, out str)) continue;
				}
				#endregion
				#region TreeView
				else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(TreeView)))
				{
					if (!CSRCTreeView.Serialize(form, (TreeView)obj, name, resIDs, out str)) continue;
				}
				#endregion
				#region TabControl
				else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(TabControl)))
				{
					if (!CSRCTabControl.Serialize(form, (TabControl)obj, name, resIDs, out str)) continue;
				}
				#endregion
				#region PictureBox
				else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(PictureBox)))
				{
					if (!CSRCPictureBox.Serialize(form, (PictureBox)obj, name, resIDs, out str)) continue;
				}
				#endregion
				#region TrackBar
				else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(TrackBar)))
				{
					if (!CSRCTrackBar.Serialize(form, (TrackBar)obj, name, resIDs, out str)) continue;
				}
				#endregion
				#region DateTimePicker
				else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(DateTimePicker)))
				{
					if (!CSRCDateTimePicker.Serialize(form, (DateTimePicker)obj, name, resIDs, out str)) continue;
				}
				#endregion
				#region MonthCalendar
				else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(MonthCalendar)))
				{
					if (!CSRCMonthCalendar.Serialize(form, (MonthCalendar)obj, name, resIDs, out str)) continue;
				}
				#endregion
				#region Panel
				else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(Panel)))
				{
					if( CSTypeHelper.IsDerived(fi.FieldType, typeof(TabPage)))
					{
					}
					else
					{
						if (!CSRCPanel.Serialize(form, (Panel)obj, name, resIDs, out str)) continue;
					}
				}
				#endregion
				#region UpDownBase
				else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(UpDownBase)))
				{
					if (CSTypeHelper.IsDerived(fi.FieldType, typeof(NumericUpDown)))
					{
						if (!CSRCNumericUpDown.Serialize(form, (NumericUpDown)obj, name, resIDs, out str)) continue;
					}
					else if (CSTypeHelper.IsDerived(fi.FieldType, typeof(DomainUpDown)))
					{
						if (!CSRCDomainUpDown.Serialize(form, (DomainUpDown)obj, name, resIDs, out str)) continue;
					}
				}
				#endregion
				#region not support
				if (str.Length == 0)
				{
					CSConsole.WriteLine(fi.FieldType.Name + " is not support.");
					continue;
				}
				#endregion

				#endregion

				buf += str;
			}

			buf += "END" + CRLF + CRLF;
			return buf;
		}
		#endregion

        #region MakeScriptString
        public static string MakeScriptString(Form form, CSRCFileStream.DLGSCRIPTTYPE type, Control ctrl
            , string text, string strID, string style)
        {
            int left, top;
            CSRCFileStream.GetFormRelativePosition(form, ctrl, out left, out top);

            string buf = type.ToString();
            while (buf.Length < 16)
            {
                buf += " ";
            }
            if (text != null && text.Length > 0)
            {
                buf += text;
                buf += ", ";
            }

            buf += strID;
            buf += ", ";
            buf += CSRCFileStream.DLUX(left, ctrl.Font);
            buf += ",";
            buf += CSRCFileStream.DLUY(top, ctrl.Font);
            buf += ",";
            buf += CSRCFileStream.DLUX(ctrl.Width, ctrl.Font);
            buf += ",";
            buf += CSRCFileStream.DLUY(ctrl.Height, ctrl.Font);

            if (style != null && style.Length > 0)
            {
                buf += ", ";
                buf += style;
            }
            return buf;
        }
        public static string MakeScriptString(Form form, CSRCFileStream.DLGSCRIPTTYPE type, Control ctrl
            , string text, string strID, string classname, string style)
        {
            int left, top;
            CSRCFileStream.GetFormRelativePosition(form, ctrl, out left, out top);

            string buf = type.ToString();
            while (buf.Length < 16)
            {
                buf += " ";
            }
            if (text != null && text.Length > 0)
            {
                buf += text;
                buf += ", ";
            }

            buf += strID;
            buf += ", ";

            if (classname != null && classname.Length > 0)
            {
                buf += "\"";
                buf += classname;
                buf += "\"";
                buf += ", ";
            }
            if (style != null && style.Length > 0)
            {
                buf += style;
                buf += ", ";
            }

            buf += CSRCFileStream.DLUX(left, ctrl.Font);
            buf += ",";
            buf += CSRCFileStream.DLUY(top, ctrl.Font);
            buf += ",";
            buf += CSRCFileStream.DLUX(ctrl.Width, ctrl.Font);
            buf += ",";
            buf += CSRCFileStream.DLUY(ctrl.Height, ctrl.Font);

            return buf;
        }
        #endregion

        #region private field
        private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	};
	#endregion

	#region Window
	class CSRCWindow
	{
		public static string GetNotStyleString(Control ctrl)
		{
			string style = "";
			if (!ctrl.Visible)
			{
				style += "NOT WS_VISIBLE";
			}
			return style;
		}
        public static string MakeStyleString(params string[] style)
		{
			string buf = "";
            foreach (string str in style)
            {
                if (str != null && str.Length > 0)
                {
                    if (buf.Length > 0)
                    {
                        buf += "| ";
                    }
                    buf += str;
                }
            }
            return buf.Replace(',', '|');
		}
    }
	#endregion

	#region コントロールリソース

	#region ボタンリソース
	class CSRCButton
	{
		#region Serialize
        public static bool Serialize(Form form, ButtonBase ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			string strID = CSResID.MakeIDString("IDC_", name);
			API.BS style = (API.BS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE) & (API.BS.MASK - 0xF);
			switch (ctrl.TextAlign)
			{
				case ContentAlignment.BottomCenter:
				case ContentAlignment.MiddleCenter:
				case ContentAlignment.TopCenter:
					style |= API.BS.BS_CENTER;
					break;
				case ContentAlignment.BottomRight:
				case ContentAlignment.MiddleRight:
				case ContentAlignment.TopRight:
					style |= API.BS.BS_RIGHT;
					break;
				case ContentAlignment.BottomLeft:
				case ContentAlignment.MiddleLeft:
				case ContentAlignment.TopLeft:
					style |= API.BS.BS_LEFT;
					break;
			}
			string notStyle = CSRCWindow.GetNotStyleString(ctrl);

			resIDs.Add(strID);
            CSRCFileStream.DLGSCRIPTTYPE type = CSRCFileStream.DLGSCRIPTTYPE.UNKOWN;
			if (CSTypeHelper.IsDerived(ctrl.GetType(), typeof(Button)))
			{
                if (form.AcceptButton == ctrl)
                {
                    type = CSRCFileStream.DLGSCRIPTTYPE.DEFPUSHBUTTON;
                }
                else
                {
                    type = CSRCFileStream.DLGSCRIPTTYPE.PUSHBUTTON;
                }
			}
			else if (CSTypeHelper.IsDerived(ctrl.GetType(), typeof(CheckBox)))
			{
                type = CSRCFileStream.DLGSCRIPTTYPE.CHECKBOX;
			}
			else if (CSTypeHelper.IsDerived(ctrl.GetType(), typeof(RadioButton)))
			{
                type = CSRCFileStream.DLGSCRIPTTYPE.RADIOBUTTON;
			}
			else
			{
				return false;
			}
            string text = "\"" + ctrl.Text + "\"";
            buf += CSRCDialog.MakeScriptString(form, type, ctrl, text, strID, CSRCWindow.MakeStyleString(style.ToString(), notStyle));
			buf += CRLF;

			return true;
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region コンボボックス
	class CSRCComboBox
	{
		#region Serialize
        public static bool Serialize(Form form, ComboBox ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			string strID = CSResID.MakeIDString("IDC_", name);
			API.CBS style = (API.CBS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE) & API.CBS.MASK;
			string notStyle = CSRCWindow.GetNotStyleString(ctrl);

			resIDs.Add(strID);
            buf += CSRCDialog.MakeScriptString(form, CSRCFileStream.DLGSCRIPTTYPE.COMBOBOX
                , ctrl, null, strID, CSRCWindow.MakeStyleString(style.ToString(), notStyle));
			buf += CRLF;

			return true;
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region リストボックス
	class CSRCListBox
	{
		#region Serialize
        public static bool Serialize(Form form, ListBox ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			string strID = CSResID.MakeIDString("IDC_", name);
			API.LBS style = (API.LBS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE) & API.LBS.MASK;
			string notStyle = CSRCWindow.GetNotStyleString(ctrl);

			resIDs.Add(strID);
            buf += CSRCDialog.MakeScriptString(form, CSRCFileStream.DLGSCRIPTTYPE.LISTBOX
                , ctrl, null, strID, CSRCWindow.MakeStyleString(style.ToString(), notStyle));
			buf += CRLF;

			return true;
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region ラベル
	class CSRCLabel
	{
		#region Serialize
        public static bool Serialize(Form form, Label ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			string strID = CSResID.MakeIDString("IDC_", name);
			API.SS style = (API.SS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE) & API.SS.MASK;
			style &= ~API.SS.SS_OWNERDRAW;
			string notStyle = CSRCWindow.GetNotStyleString(ctrl);

			resIDs.Add(strID);

            string text = "\"" + ctrl.Text + "\"";
            buf += CSRCDialog.MakeScriptString(form, CSRCFileStream.DLGSCRIPTTYPE.LTEXT
                , ctrl, text, strID, CSRCWindow.MakeStyleString(style.ToString(), notStyle));
			buf += CRLF;

			return true;
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region テキストボックスリソース
	class CSRCTexBoxBase
	{
		public static string GetNotStyleString(TextBoxBase ctrl)
		{
			string style = CSRCWindow.GetNotStyleString(ctrl);
			if (ctrl.BorderStyle == BorderStyle.None)
			{
				if (style.Length > 0) style += "| ";
				style += "NOT WS_BORDER";
			}
			return style;
		}
	};
	class CSRCTextBox
	{
		#region Serialize
        public static bool Serialize(Form form, TextBoxBase ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			string strID = CSResID.MakeIDString("IDC_", name);
			API.ES style = (API.ES)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE) & API.ES.MASK;
			string notStyle = CSRCTexBoxBase.GetNotStyleString(ctrl);

			resIDs.Add(strID);

            buf += CSRCDialog.MakeScriptString(form, CSRCFileStream.DLGSCRIPTTYPE.EDITTEXT
                , ctrl, null, strID, CSRCWindow.MakeStyleString(style.ToString(), notStyle));
			buf += CRLF;

			return true;
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region スクロールバー
	class CSRCScrollBar
	{
		#region Serialize
        public static bool Serialize(Form form, ScrollBar ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			string strID = CSResID.MakeIDString("IDC_", name);
			API.SBS style = (API.SBS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE) & API.SBS.MASK;
			string notStyle = CSRCWindow.GetNotStyleString(ctrl);

			resIDs.Add(strID);

            buf += CSRCDialog.MakeScriptString(form, CSRCFileStream.DLGSCRIPTTYPE.SCROLLBAR
                , ctrl, null, strID, CSRCWindow.MakeStyleString(style.ToString(), notStyle));
			buf += CRLF;

			return true;
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region グループボックス
	class CSRCGroupBox
	{
		#region Serialize
        public static bool Serialize(Form form, GroupBox ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			string strID = CSResID.MakeIDString("IDC_", name);
			API.BS style = (API.BS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE) & API.BS.MASK;
			string notStyle = CSRCWindow.GetNotStyleString(ctrl);

			resIDs.Add(strID);

            string text = "\"" + ctrl.Text + "\"";
            buf += CSRCDialog.MakeScriptString(form, CSRCFileStream.DLGSCRIPTTYPE.GROUPBOX
                , ctrl, text, strID, CSRCWindow.MakeStyleString(style.ToString(), notStyle));
			buf += CRLF;

			return true;
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region プログレスバー
	class CSRCProgressBar
	{
		#region Serialize
        public static bool Serialize(Form form, ProgressBar ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			RES.WS style = (RES.WS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE);
			API.PBS ctrlstyle = (API.PBS)style & API.PBS.MASK;
			style &= RES.WS.MASK;

			string strStyle = "";
			if (ctrlstyle != 0)
				strStyle += ctrlstyle.ToString().Replace(',', '|');
			if (style != 0)
			{
				if (ctrlstyle != 0) strStyle += "| ";
				strStyle += style.ToString().Replace(',', '|');
			}

			return CSRCControl.Serialize(form, ctrl, name, resIDs, "msctls_progress32", strStyle, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region リストビュー
	class CSRCListView
	{
		#region Serialize
        public static bool Serialize(Form form, ListView ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			RES.WS style = (RES.WS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE);
			API.LVS ctrlstyle = (API.LVS)style & API.LVS.MASK;
			style &= RES.WS.MASK;

			string strStyle = "";
			if (ctrlstyle != 0)
				strStyle += ctrlstyle.ToString().Replace(',', '|');
			if (style != 0)
			{
				if (ctrlstyle != 0) strStyle += "| ";
				strStyle += style.ToString().Replace(',', '|');
			}

			return CSRCControl.Serialize(form, ctrl, name, resIDs, "SysListView32", strStyle, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region ツリービュー
	class CSRCTreeView
	{
		#region Serialize
        public static bool Serialize(Form form, TreeView ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			RES.WS style = (RES.WS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE);
			API.TVS ctrlstyle = (API.TVS)style & API.TVS.MASK;
			style &= RES.WS.MASK;
            string strStyle = CSRCWindow.MakeStyleString(ctrlstyle.ToString(), style.ToString());

			return CSRCControl.Serialize(form, ctrl, name, resIDs, "SysTreeView32", strStyle, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region タブコントロール
	class CSRCTabControl
	{
		#region Serialize
        public static bool Serialize(Form form, TabControl ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			RES.WS style = (RES.WS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE);
			API.TCS ctrlstyle = (API.TCS)style & API.TCS.MASK;
			style &= RES.WS.MASK;
            string strStyle = CSRCWindow.MakeStyleString(ctrlstyle.ToString(), style.ToString());

			return CSRCControl.Serialize(form, ctrl, name, resIDs, "SysTabControl32", strStyle, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region ピクチャーボックス
	class CSRCPictureBox
	{
		#region Serialize
        public static bool Serialize(Form form, PictureBox ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			RES.WS style = (RES.WS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE);
			API.SS ctrlstyle = (API.SS)style & API.SS.MASK | API.SS.SS_BITMAP;
			style &= RES.WS.MASK;
            string strStyle = CSRCWindow.MakeStyleString(ctrlstyle.ToString(), style.ToString());

            string text = null;
			if (ctrl.ImageLocation == null)
			{
				Type res = CSRCResource.GetPropertiesResource(form.GetType().Assembly);
                string resname = CSRCResource.GetResourceName(res, ctrl.Image);
				if (resname == null)
				{
					text = "IDB_BITMAP1";	// TODO : 適当
				}
				else
				{
					text = "IDB_" + resname.ToUpper();
				}
			}
			else
			{
				text = "\"" + ctrl.ImageLocation + "\"";
			}
			return CSRCControl.Serialize(form, ctrl, name, resIDs, text, "Static", strStyle, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region リンクラベル
	class CSRCLinkLabel
	{
		#region Serialize
        public static bool Serialize(Form form, LinkLabel ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			RES.WS style = (RES.WS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE);
			API.SS ctrlstyle = 0;// (API.SS)style & API.SS.MASK;
			style &= RES.WS.MASK;
            string strStyle = CSRCWindow.MakeStyleString(ctrlstyle.ToString(), style.ToString());

			string text = "\"<a>" + ctrl.Text + "</a>\"";
            string strID = CSResID.MakeIDString("IDC_", name);
            string notStyle = CSRCWindow.GetNotStyleString(ctrl);
            buf += CSRCDialog.MakeScriptString(form, CSRCFileStream.DLGSCRIPTTYPE.LTEXT
                , ctrl, text, strID, CSRCWindow.MakeStyleString(style.ToString(), notStyle));
            buf += CRLF;

            resIDs.Add(strID);
            return true;
            //return CSRCControl.Serialize(form, ctrl, name, resIDs, text, "SysLink", strStyle, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region リッチテキスト
	class CSRCRichTextBox
	{
		#region Serialize
        public static bool Serialize(Form form, RichTextBox ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			RES.WS style = (RES.WS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE);
			API.ES ctrlstyle = (API.ES)style & API.ES.MASK;
			style &= RES.WS.MASK;
            string strStyle = CSRCWindow.MakeStyleString(ctrlstyle.ToString(), style.ToString());

			return CSRCControl.Serialize(form, ctrl, name, resIDs, "RichEdit20A", strStyle, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region トラックバー
	class CSRCTrackBar
	{
		#region Serialize
        public static bool Serialize(Form form, TrackBar ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			RES.WS style = (RES.WS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE);
			API.TBS ctrlstyle = (API.TBS)style & API.TBS.MASK;
			style &= RES.WS.MASK;
            string strStyle = CSRCWindow.MakeStyleString(ctrlstyle.ToString(), style.ToString());

			return CSRCControl.Serialize(form, ctrl, name, resIDs, "msctls_trackbar32", strStyle, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region DateTimeピッカー
	class CSRCDateTimePicker
	{
		#region Serialize
        public static bool Serialize(Form form, DateTimePicker ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			RES.WS style = (RES.WS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE);
			API.DTS ctrlstyle = (API.DTS)style & API.DTS.MASK;
			style &= RES.WS.MASK;
            string strStyle = CSRCWindow.MakeStyleString(ctrlstyle.ToString(), style.ToString());

			return CSRCControl.Serialize(form, ctrl, name, resIDs, "SysDateTimePick32", strStyle, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region Month カレンダー
	class CSRCMonthCalendar
	{
		#region Serialize
        public static bool Serialize(Form form, MonthCalendar ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			RES.WS style = (RES.WS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE);
			API.MCS ctrlstyle = (API.MCS)style & API.MCS.MASK;
			style &= RES.WS.MASK;
            string strStyle = CSRCWindow.MakeStyleString(ctrlstyle.ToString(), style.ToString());

			return CSRCControl.Serialize(form, ctrl, name, resIDs, "SysMonthCal32", strStyle, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region パネル
	class CSRCPanel
	{
		#region Serialize
        public static bool Serialize(Form form, Panel ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			RES.WS style = (RES.WS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE);
			API.SS ctrlstyle = (API.SS)style & API.SS.MASK | API.SS.SS_BLACKFRAME;
			style &= RES.WS.MASK;
            string strStyle = CSRCWindow.MakeStyleString(ctrlstyle.ToString(), style.ToString());

			return CSRCControl.Serialize(form, ctrl, name, resIDs, "Static", strStyle, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region コントロール
	class CSRCControl
	{
		#region Serialize
        public static bool Serialize(Form form, Control ctrl, string name, CSResID resIDs, string classname, string style, out string buf)
		{
			string text = "\"" + ctrl.Name + "\"";
			return Serialize(form, ctrl, name, resIDs, text, classname, style, out buf);
		}
		#endregion

		#region Serialize
        public static bool Serialize(Form form, Control ctrl, string name, CSResID resIDs, string text, string classname, string style, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			string strID = CSResID.MakeIDString("IDC_", name);
			resIDs.Add(strID);
			string notStyle = CSRCWindow.GetNotStyleString(ctrl);

            buf += CSRCDialog.MakeScriptString(form, CSRCFileStream.DLGSCRIPTTYPE.CONTROL
                , ctrl, text, strID, classname, CSRCWindow.MakeStyleString(style, notStyle));
			buf += CRLF;

			return true;
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region アップダウンコントロール

	#region UpDwonEdit
	class CSRCUpDownEdit
	{
		#region Serialize
        public static bool Serialize(Form form, TextBox ctrl, string name, CSResID resIDs, out string buf)
		{
            return CSRCTextBox.Serialize(form, ctrl, name + "_EDIT", resIDs, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region UpDownButton
	class CSRCUpDownButton
	{
		#region Serialize
        public static bool Serialize(Form form, Control ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			string strID = CSResID.MakeIDString("IDC_", name);
			RES.WS style = (RES.WS)API.GetWindowLong(ctrl, API.GWL.GWL_STYLE);
			API.UDS ctrlstyle = (API.UDS)style & API.UDS.MASK | API.UDS.UDS_AUTOBUDDY;
			string strStyle = "";
			if (ctrlstyle != 0)
				strStyle += ctrlstyle.ToString().Replace(',', '|');
			if (style != 0)
			{
				if (ctrlstyle != 0) strStyle += "| ";
				strStyle += style.ToString().Replace(',', '|');
			}

			return CSRCControl.Serialize(form, ctrl, name, resIDs, "msctls_updown32", strStyle, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region UpDownBase
	class CSRCUpDownBase
	{
		#region Serialize
        public static bool Serialize(Form form, UpDownBase ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			string buf2 = "";
			FieldInfo[] fis = ctrl.GetType().GetFields(BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.Instance);
			foreach (FieldInfo fi in fis)
			{
				if (fi.FieldType.Name == "UpDownEdit")
				{
                    TextBox edit = fi.GetValue(ctrl) as TextBox;
					if (edit == null) return false;
					if (!CSRCUpDownEdit.Serialize(form, edit, name, resIDs, out buf2)) return false;
					buf += buf2;
				}
				else if (fi.FieldType.Name == "UpDownButtons")
				{
					Control btn = fi.GetValue(ctrl) as Control;
					if (btn == null) return false;
					if (!CSRCUpDownButton.Serialize(form, btn, name, resIDs, out buf2)) return false;
					buf += buf2;
				}
			}
			return true;
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region NumericUpDown
	class CSRCNumericUpDown
	{
		#region Serialize
        public static bool Serialize(Form form, NumericUpDown ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			return CSRCUpDownBase.Serialize(form, ctrl, name, resIDs, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region DomainUpDown
	class CSRCDomainUpDown
	{
		#region Serialize
        public static bool Serialize(Form form, DomainUpDown ctrl, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (ctrl == null) return false;
			return CSRCUpDownBase.Serialize(form, ctrl, name, resIDs, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#endregion

	#region ツールストリップ
	class CSRCToolStrip
	{
		#region Serialize
        public static bool Serialize(ToolStrip strip, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			{
				ContextMenuStrip menu = strip as ContextMenuStrip;
				if (menu != null)
				{
					return SerializeMenuStrip(strip, name, resIDs, out buf);
				}
			}
			{
				MenuStrip menu = strip as MenuStrip;
				if (menu != null)
				{
					return SerializeMenuStrip(strip, name, resIDs, out buf);
				}
			}
			return false;
		}
		#endregion

		#region MenuStrip Serialize
        public static bool SerializeMenuStrip(ToolStrip strip, string name, CSResID resIDs, out string buf)
		{
			buf = "";
			if (strip.Items.Count == 0) return false;
			string strID = CSResID.MakeIDString("IDR_", name);
			resIDs.Add(strID);

			ContextMenuStrip cntx = strip as ContextMenuStrip;

			string indent = "    ";
			buf += strID + " MENU";
			buf += CRLF;
			buf += "BEGIN";
			buf += CRLF;
			if (cntx != null)
			{
				buf += indent;
				buf += "POPUP ";
				buf += "\"" + cntx.Name + "\"";
				buf += CRLF;

				buf += indent;
				buf += "BEGIN";
				buf += CRLF;
			}

			foreach (ToolStripItem item in strip.Items)
			{
				string str = "";
				if (CSRCToolStripMenuItem.Serialize(item, 1, resIDs, out str))
				{
					buf += str;
				}
			}

			if (cntx != null)
			{
				buf += indent;
				buf += "END";
				buf += CRLF;
			}
			
			buf += "END";
			buf += CRLF;
			buf += CRLF;
			return true;
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region コンテキストメニューストリップ
	class CSRCContextMenuStrip
	{
		#region Serialize
        public static bool Serialize(ContextMenuStrip menu, string name, CSResID resIDs, out string buf)
		{
			return CSRCToolStrip.SerializeMenuStrip(menu, name, resIDs, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region メニューストリップ
	class CSRCMenuStrip
	{
		#region Serialize
        public static bool Serialize(MenuStrip menu, string name, CSResID resIDs, out string buf)
		{
			return CSRCToolStrip.SerializeMenuStrip(menu, name, resIDs, out buf);
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#region メニューストリップアイテム
	class CSRCToolStripMenuItem
	{
		#region Serialize
		public static bool Serialize(ToolStripItem dropItem, int depth, CSResID resIDs, out string buf)
		{
			buf = "";
			if (dropItem == null) return false;

			string indent = "";
			for (int i = 0; i < depth; ++i)
			{
				indent += "    ";
            }

            #region SEPARATOR
            ToolStripSeparator sep = dropItem as ToolStripSeparator;
			if (sep != null)
			{
				buf += indent;
				buf += "MENUITEM SEPARATOR";
				buf += CRLF;
				return true;
            }
            #endregion

            ToolStripMenuItem menu = dropItem as ToolStripMenuItem;
			if (menu == null) return false;
						
			if (menu.DropDownItems.Count == 0)
            {
                #region MENUITEM
                buf += indent;
				buf += "MENUITEM ";
				if (menu.DisplayStyle == ToolStripItemDisplayStyle.Text
					|| menu.DisplayStyle == ToolStripItemDisplayStyle.ImageAndText)
				{
                    buf += "\"" + menu.Text;
                    #region ショートカットキーの表示
                    if (menu.ShowShortcutKeys && (menu.ShortcutKeys != Keys.None))
                    {
                        string strsc = menu.ShortcutKeyDisplayString;
                        if (strsc == null || strsc.Length == 0)
                        {
                            Keys vk = menu.ShortcutKeys;
                            if ((vk & Keys.Control) == Keys.Control)
                            {
                                strsc += "Ctrl+";
                                vk -= Keys.Control;
                            }
                            if ((vk & Keys.Shift) == Keys.Shift)
                            {
                                strsc += "Shift+";
                                vk -= Keys.Shift;
                            }
                            if ((vk & Keys.Alt) == Keys.Alt)
                            {
                                strsc += "Alt+";
                                vk -= Keys.Alt;
                            }

                            if (vk >= Keys.D0 && vk <= Keys.D9)
                            {
                                strsc += vk.ToString().Substring(1);
                            }
                            else
                            {
                                strsc += vk.ToString();
                            }
                        }
                        buf += "\\t" + strsc;
                    }
                    #endregion
                    buf += "\",    ";
				}
				string strID = CSResID.MakeIDString("IDM_", menu.Name);
				resIDs.Add(strID);
				buf += strID;
				buf += CRLF;
                #endregion
            }
			else
            {
                #region POPUP
                buf += indent;
				buf += "POPUP ";
				buf += "\"" + menu.Text + "\"";
				buf += CRLF;

				buf += indent;
				buf += "BEGIN";
				buf += CRLF;

				foreach (ToolStripItem item in menu.DropDownItems)
				{
					string str = "";
					if (CSRCToolStripMenuItem.Serialize(item, depth+1, resIDs, out str))
					{
						buf += str;
					}
				}

				buf += indent;
				buf += "END";
				buf += CRLF;
                #endregion
            }
			return true;
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

	#endregion

	#region バージョン情報リソース
	class CSRCVersionInfo
	{
		#region Serialize
		public static bool Serialize(FileVersionInfo vi, out string buf)
		{
			buf = "";
			if (vi == null) return false;
			buf += CSRCFileStream.RESCOMMENT.Replace("NAME", "Version");

			buf += "VS_VERSION_INFO VERSIONINFO" + CRLF;
			buf += "FILEVERSION " + vi.FileVersion.Replace('.', ',') + CRLF;
			buf += "PRODUCTVERSION " + vi.ProductVersion.Replace('.', ',') + CRLF;

			buf += "FILEFLAGSMASK 0x17L" + CRLF + "#ifdef _DEBUG" + CRLF + " FILEFLAGS 0x1L" + CRLF + "#else" + CRLF + " FILEFLAGS 0x0L" + CRLF + "#endif" + CRLF;

			buf += " FILEOS ";
			PlatformID pid = Environment.OSVersion.Platform;
			if (pid == PlatformID.WinCE)
			{
				buf += "VOS_WINCE";
			}
			else
			{
				buf += "VOS__WINDOWS32";
			}
			buf += CRLF;

			buf += " FILETYPE 0x01L" + CRLF;	// TODO : デフォルトで VFT_APP
			buf += " FILESUBTYPE 0x0L" + CRLF;	// TODO : デフォルト

			buf += "BEGIN" + CRLF;
			buf += GetStringFileInfo(vi);
			buf += GetVarFileInfo(vi);
			buf += "END" + CRLF;
			return true;
		}
		#endregion

		#region GetStringFileInfo
		private static string GetStringFileInfo(FileVersionInfo vi)
		{
			const string VALUE = "            VALUE ";
			string buf = "    BLOCK \"StringFileInfo\"" + CRLF;
			buf += "    BEGIN" + CRLF;
			buf += "        BLOCK " + "\"041104b0\"" + CRLF;	// TODO : 言語とか国のコードなんだろうけど
			buf += "        BEGIN" + CRLF;

			buf += VALUE + "\"Comments\", \"" + vi.Comments + "\"" + CRLF;
			buf += VALUE + "\"CompanyName\", \"" + vi.CompanyName + "\"" + CRLF;
			buf += VALUE + "\"FileDescription\", \"" + vi.FileDescription + "\"" + CRLF;
			buf += VALUE + "\"FileVersion\", \"" + vi.FileVersion + "\"" + CRLF;
			buf += VALUE + "\"InternalName\", \"" + vi.InternalName + "\"" + CRLF;
			buf += VALUE + "\"LegalCopyright\", \"" + vi.LegalCopyright + "\"" + CRLF;
			buf += VALUE + "\"OriginalFilename\", \"" + vi.OriginalFilename + "\"" + CRLF;
			buf += VALUE + "\"ProductName\", \"" + vi.ProductName + "\"" + CRLF;
			buf += VALUE + "\"ProductVersion\", \"" + vi.ProductVersion + "\"" + CRLF;
			if( vi.SpecialBuild != null && vi.SpecialBuild.Length > 0)
				buf += VALUE + "\"SpecialBuild\", \"" + vi.SpecialBuild + "\"" + CRLF;

			buf += "        END" + CRLF;
			buf += "    END" + CRLF;
			return buf;
		}
		#endregion

		#region GetVarFileInfo
		private static string GetVarFileInfo(FileVersionInfo vi)
		{
			const string VALUE = "        VALUE ";
			string buf = "    BLOCK \"VarFileInfo\"" + CRLF;
			buf += "    BEGIN" + CRLF;

			buf += VALUE + "\"Translation\", " + "0x411" + ", " + "1200" + "" + CRLF;	// TODO : 言語とか国のコード
	
			buf += "    END" + CRLF;
			return buf;
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

    #region アクセラレータ
    class CSRCAccelerators
    {
        #region Serialize
        public static bool Serialize(Dictionary<Keys, ToolStripItem> keys, string name, out string buf)
        {
            buf = "";
            if (keys == null) return false;
            if (keys.Count == 0) return false;

            buf += CSResID.MakeIDString("IDC_", name);
            buf += " ACCELERATORS";
            buf += CRLF;
            buf += "BEGIN";
            buf += CRLF;

            string indent = "    ";

            foreach (KeyValuePair<Keys, ToolStripItem> k in keys)
            {
                bool virtkey = false;
                string vkoption = "";
                string option = "";
                Keys vk = k.Key;
                if ((vk & Keys.Control) == Keys.Control)
                {
                    option += ", CONTROL";
                    vk -= Keys.Control;
                    virtkey = true;
                }
                if ((vk & Keys.Shift) == Keys.Shift)
                {
                    option += ", SHIFT";
                    vk -= Keys.Shift;
                    virtkey = true;
                }
                if ((vk & Keys.Alt) == Keys.Alt)
                {
                    option += ", ALT";
                    vk -= Keys.Alt;
                    virtkey = true;
                }

                string keystr = "";
                if (vk >= Keys.A && vk <= Keys.Z)
                {
                    keystr = "\"" + vk.ToString() + "\"";
                }
                else if (vk >= Keys.D0 && vk <= Keys.D9)
                {
                    keystr = "\"" + vk.ToString().Substring(1) + "\"";
                }
                else
                {
                    keystr = "VK_" + vk.ToString().ToUpper();
                    virtkey = true;
                }
                if (virtkey)
                {
                    vkoption = ", VIRTKEY";
                }
                else
                {
                    vkoption = ", ASCII";
                }

                buf += indent;
                while (keystr.Length < 8)
                {
                    keystr += " ";
                }
                buf += keystr;
                buf += " ," + CSResID.MakeIDString("IDM_", k.Value.Name);
                buf += vkoption;
                buf += option;
                buf += CRLF;
            }

            buf += "END";
            buf += CRLF;
            buf += CRLF;
            return true;
        }
        #endregion

        #region private field
        private const string CRLF = CSRCFileStream.CRLF;
        #endregion
    }
    #endregion

    #region その他リソース
    class CSRCResource
	{
		#region Serialize
		public static bool Serialize(Assembly asm, CSResID resIDs, string directory, out string buf)
		{
			buf = "";
			if (asm == null) return false;
			Type res = GetPropertiesResource(asm);
			if (res == null) return false;

			// bitmap
			#region bitmap
			{
				string str = "";
				if (GetBitmapsString(res, resIDs, directory, out str))
				{
					buf += str;
				}
			}
			#endregion

			#region Icon
			{
				string str = "";
				if (GetIconsString(res, resIDs, directory, out str))
				{
					buf += str;
				}
			}
			#endregion

			#region String Table
			{
				string str = "";
				if (GetStringTableString(res, resIDs, out str))
				{
					buf += str;
				}
			}
			#endregion

			return true;
		}
		#endregion

		#region GetPropertiesResource
		public static Type GetPropertiesResource(Assembly asm)
		{
			if (asm == null) return null;
			Type[] types = asm.GetTypes();
			try
			{
				foreach (Type t in types)
				{
					FieldInfo[] fis = t.GetFields(BindingFlags.NonPublic | BindingFlags.Public | BindingFlags.Static);
					foreach (FieldInfo fi in fis)
					{
						if (fi.FieldType == typeof(ResourceManager))
						{
							return fi.DeclaringType;
						}
					}
				}
			}
			catch (Exception e)
			{
				CSConsole.WriteLine(e.ToString());
				return null;
			}
			return null;
		}
		#endregion

		#region リソース名の取得
		public static string GetResourceName(Type resType, object target)
		{
			if (resType == null || target == null) return null;
			Type tt = target.GetType();
			PropertyInfo[] pis = resType.GetProperties(BindingFlags.Static | BindingFlags.Public | BindingFlags.NonPublic);
			foreach (PropertyInfo pi in pis)
			{
				if (pi.PropertyType == tt)
				{
					object o = pi.GetValue(null, null);
					Type ot = o.GetType();
					if (ot.GUID == tt.GUID)
					{
						return pi.Name;
					}
				}
			}
			return null;
		}
		#endregion

		#region CollectTypes
		private static object[] CollectTypes(Type resType, Type type)
		{
			PropertyInfo[] pis = resType.GetProperties(BindingFlags.Static | BindingFlags.Public | BindingFlags.NonPublic);
			ArrayList al = new ArrayList();
			foreach (PropertyInfo pi in pis)
			{
				if( pi.PropertyType == type )
				{
					object o = pi.GetValue(null, null);
					if( o != null )
						al.Add(o);
				}
			}
			return al.ToArray();
		}
		#endregion

		#region BITMAP
		private static bool GetBitmapsString(Type resType, CSResID resIDs, string directory, out string buf)
		{
			buf = "";
			if (resType == null) return false;
			bool ret = false;
			PropertyInfo[] pis = resType.GetProperties(BindingFlags.Static | BindingFlags.Public | BindingFlags.NonPublic);
			foreach (PropertyInfo pi in pis)
			{
				if (pi.PropertyType == typeof(Bitmap))
				{
					object o = pi.GetValue(null, null);
					if (o != null)
					{
						Bitmap bmp = (Bitmap)o;
						if (ret == false)
						{
							buf += CSRCFileStream.RESCOMMENT.Replace("NAME", "Bitmap");
							ret = true;
						}
                        string name = pi.Name;
						string fname = name + ".bmp";
						string strIDB = CSResID.MakeIDString("IDB_",  name);
						resIDs.Add(strIDB);
						while (strIDB.Length < 25) strIDB += " ";

						buf += strIDB;
						buf += "BITMAP                  ";
						buf += "\"" + fname + "\"" + CRLF;
						// TODO : パスの取得が分からないので、出力する
						string path = Path.Combine(directory, fname);
						bmp.Save(path);
					}
				}
			}
			if (ret) buf += CRLF + CRLF;
			return ret;
		}
		#endregion

		#region ICON
		private static bool GetIconsString(Type resType, CSResID resIDs, string directory, out string buf)
		{
			buf = "";
			if (resType == null) return false;
			bool ret = false;
			PropertyInfo[] pis = resType.GetProperties(BindingFlags.Static | BindingFlags.Public | BindingFlags.NonPublic);
			foreach (PropertyInfo pi in pis)
			{
				if (pi.PropertyType == typeof(Icon))
				{
					object o = pi.GetValue(null, null);
					if (o != null)
					{
						Icon resource = (Icon)o;
						if (ret == false)
						{
							buf += CSRCFileStream.RESCOMMENT.Replace("NAME", "Icon");
							ret = true;
						}
                        string name = pi.Name;
						string fname = name + ".ico";
						string strIDB = CSResID.MakeIDString("IDI_", name);
						resIDs.Add(strIDB);
						while (strIDB.Length < 25) strIDB += " ";

						buf += strIDB;
						buf += "ICON                    ";
						buf += "\"" + fname + "\"" + CRLF;
						// TODO : パスの取得が分からないので、出力する
						string path = Path.Combine(directory, fname);
						CSFileStream fs = new CSFileStream(path, FileMode.Create);
						resource.Save(fs.Stream);
					}
				}
			}
			if (ret) buf += CRLF + CRLF;
			return ret;
		}
		#endregion

		#region String Table
		private static bool GetStringTableString(Type resType, CSResID resIDs, out string buf)
		{
			buf = "";
			if (resType == null) return false;
			bool ret = false;
			PropertyInfo[] pis = resType.GetProperties(BindingFlags.Static | BindingFlags.Public | BindingFlags.NonPublic);
			foreach (PropertyInfo pi in pis)
			{
				if (pi.PropertyType == typeof(string))
				{
					object o = pi.GetValue(null, null);
					if (o != null)
					{
						string resource = (string)o;
						if (ret == false)
						{
							buf += CSRCFileStream.RESCOMMENT.Replace("NAME", "String Table");
							buf += "STRINGTABLE" + CRLF + "BEGIN" + CRLF;
							ret = true;
						}
                        string name = pi.Name;
						string strIDB = CSResID.MakeIDString("IDS_", name);
						resIDs.Add(strIDB);
						while (strIDB.Length < 25) strIDB += " ";

						buf += "    " + strIDB + "\"" + resource + "\"" + CRLF;
					}
				}
			}
			if (ret)
			{
				buf += "END" + CRLF + CRLF + CRLF;
			}
			return ret;
		}
		#endregion

		#region private field
		private const string CRLF = CSRCFileStream.CRLF;
		#endregion
	}
	#endregion

}	// end of namespace rcsharp

