/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 Part of this file was originally taken from the SoccerServer and
 modified. The SoccerServer is also released under the GNU Lesser
 General Public License.

 Copyright (C) 1996-2000 Electrotechnical Laboratory. 
 Itsuki Noda, Yasuo Kuniyoshi and Hitoshi Matsubara.
 Copyright (C) 2000, 2001 RoboCup Soccer Server Maintainance Group.
 Patrick Riley, Tom Howard, Daniel Polani, Itsuki Noda,
 Mikhail Prokopenko, Jan Wendler 

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/


#ifndef UTILITY_H_
#define UTILITY_H_

#include "spades-config.h"
#if SPADES_TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if SPADES_HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#include <unistd.h>
#include <string>
#include <vector>
#include <functional>
#include <iostream>
#include <iterator>

namespace spades
{
  
  std::string toString(int i);

  // finds and replaces all occurences of find with repl in str
  // operates recursively, so this could make an infinite loop if you
  // are not careful
  std::string findReplaceInStr(const std::string& str, const char* find, const char* repl);

  std::string trimWhiteSpace(const char* str);
  inline std::string trimWhiteSpace(const std::string& s) { return trimWhiteSpace(s.c_str()); }
  
  void seedRandom(int seed = -1, bool print = false);
  double rangeRandom(double lo, double hi);
  inline double probRandom() { return rangeRandom(0, 1); }

  // see page A-111 of 
  // http://www.geocities.com/~mikemclaughlin/math_stat/Dists/Compendium.html
  double triangularCDF(double pt, double A, double B, double C);
  
  /* Set the FD_CLOEXEC flag if value is true or clear the flag if value is false */
  bool set_cloexec_flag (int desc, bool value);
  /* same, but do it for all fd. Returns true iff all fds set */
  bool set_all_cloexec_flags (bool value);
  
  /* dynamically allocates */
  void setEnvVar(const char* var, const char* value);
  
  /* return 0 on error, the character representing the state otherwise */
  char get_process_status(int pid);
  
  std::ostream& operator << (std::ostream& o, const struct timeval & t) ;

  //breaks the strings into arguments which are space delimited
  /* The argument parsing is pretty simplistic. The rules are quite simple
     (space) spearates arguments
     " can be used to bound an argument, i.e. get a space in an argument name
     The " will not be passed to the executable
     If you want to include a " in a string, you must include that argument in " "
     and use "" to get a single "
     No other characters are special
     Note in particular that this means that \ does not mean anything special, nor
     can any shell style redirection or anything like that be done
     Example:
     foo -a b --code \\
     gives 'foo', '-a', 'b', '--code', '\\'
     bar -a "Some Name"
     gives 'bar', '-a', 'Some Name'
     baz -a "Some""Name"
     gives 'bar', '-a', 'Some"Name'
     bif -a """SomeName"""
     gives 'bif', '-a', '"Some Name"'

     end is the character one after the last one we are to process
  */
  bool breakIntoArgs(std::string s,
		     std::string::size_type begin, std::string::size_type end,
		     std::vector<char*>& vArgs);
  inline bool breakIntoArgs(std::string s,
			    std::vector<char*>& vArgs)
  { return breakIntoArgs(s, 0, s.size(), vArgs); }

  /* returns an idx such that begin - return (not including the character at return
     is a brace balanced string which includes at least
     1 character of string s
     end is one character after the last one to consider
     Returns string::npas if not such substring can be found */
  std::string::size_type findBraceBalanced(std::string s,
					   std::string::size_type begin,
					   std::string::size_type end);
  inline std::string::size_type findBraceBalanced(std::string s,
						  std::string::size_type begin)
  { return findBraceBalanced(s, begin, s.size()); }
  inline std::string::size_type findBraceBalanced(std::string s)
  { return findBraceBalanced(s, 0, s.size()); }

  //return true iff the first strlen(prefix) characters of str match prefix
  inline bool prefixMatch(const char* prefix, const char* str)
  { return strncmp(prefix, str, strlen(prefix)) == 0; }
  
  struct cstr_lt : public std::binary_function <const char *, const char *, bool>
  {
    bool operator () (const char * s1, const char * s2)
    { return strcmp(s1, s2) < 0; }
  };

  // Assigns str the value is cstr may be null terminated, but is at most len chars
  void assignStringFromMaybeNullTerm( std::string& str, const char* cstr, unsigned len);
  
  //useful for for_each to delete the contents of a container
  template<class T> struct deleteptr : public std::unary_function<T*, void>
  {
    void operator() (T* x) { delete x; }
  };

  //useful for for_each to delete the contents of a map container
  template<class Map> struct deletemapptr : public std::unary_function<typename Map::value_type, void>
  {
    void operator() (typename Map::value_type x) { delete x.second; }
  };


  //Only maintains info for mean
  class MeanSummary 
  {
  public:
    MeanSummary() { reset(); }
    MeanSummary(const MeanSummary& ms) : n(ms.n), sum(ms.sum) { }
    MeanSummary(int new_n, double new_sum)
    { n = new_n; sum = new_sum; }

    void reset();
  
    void addPoint(double x, int count = 1);
    double getMean() const;
    int getNumPoints() const { return n; }

    void addIn(const MeanSummary& ms) { n+=ms.n; sum+=ms.sum; }

    friend std::ostream& operator<<(std::ostream& os, const MeanSummary& ms)
    { os << "MeanSummary(n:" << ms.n << ", sum:" << ms.sum << ")"; return os; }
      
  private:
    int n;
    double sum;
  } ;
 
} //spades namespace

#endif
