/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* This class was inspired by the XmlStream class written by
   Oboltus and posted on codeproject.org */

/* Some notes:
 * you can not use endl, flush, ends
 */

#ifndef XML_WRITER_H_
#define XML_WRITER_H_

#include <iostream>
#include <stack>
#include <sstream>

namespace spades
{
  class XMLWriter
  {
  public:
    class StreamControl
    {
    public:
      StreamControl() {}
      virtual ~StreamControl() {}

      virtual bool handle(XMLWriter& writer) const = 0;

      virtual void print(std::ostream& os) const = 0;
      friend std::ostream& operator<<(std::ostream& os, const StreamControl& control)
      { control.print(os); return os; }
    };

    class Prolog
      : public StreamControl
    {
    public:
      Prolog(const char* str_version, const char* str_encoding)
	: StreamControl(), version(str_version), encoding(str_encoding) {}
      ~Prolog() {}
      bool handle(XMLWriter& writer) const { return writer.curr_state->handleProlog(writer, *this); }
      void print(std::ostream& os) const { os << "XML::Prolog(" << version << ", " << encoding << ")"; }
      const std::string& getVersion() const { return version; }
      const std::string& getEncoding() const { return encoding; }
    private:
      std::string version;
      std::string encoding;
    };

    class Tag
      : public StreamControl
    {
    public:
      Tag(const char* name = "") : StreamControl(), name(name) {}
      ~Tag() {}
      bool handle(XMLWriter& writer) const { return writer.curr_state->handleTag(writer, *this); }
      void print(std::ostream& os) const { os << "XML::Tag(" << name << ")"; }
      const std::string& getName() const { return name; }
    private:
      std::string name;
    };

    class BareTag
      : public StreamControl
    {
    public:
      BareTag(const char* name = "") : StreamControl(), name(name) {}
      ~BareTag() {}
      bool handle(XMLWriter& writer) const;
      void print(std::ostream& os) const { os << "XML::BareTag(" << name << ")"; }
      const std::string& getName() const { return name; }
    private:
      std::string name;
    };

    class EmptyElement
      : public StreamControl
    {
    public:
      EmptyElement(const char* name = "") : StreamControl(), name(name) {}
      ~EmptyElement() {}
      bool handle(XMLWriter& writer) const;
      void print(std::ostream& os) const { os << "XML::EmptyElement(" << name << ")"; }
      const std::string& getName() const { return name; }
    private:
      std::string name;
    };

    class CloseTag
      : public StreamControl
    {
    public:
      CloseTag() : StreamControl() {}
      ~CloseTag() {}
      bool handle(XMLWriter& writer) const { return writer.curr_state->handleCloseTag(writer, *this); }
      void print(std::ostream& os) const { os << "XML::CloseTag"; }
    };

    class EndTag
      : public StreamControl
    {
    public:
      EndTag() : StreamControl() {}
      ~EndTag() {}
      bool handle(XMLWriter& writer) const { return writer.curr_state->handleEndTag(writer, *this); }
      void print(std::ostream& os) const { os << "XML::EndTag"; }
    };

    class Attr
      : public StreamControl
    {
    public:
      Attr(const char* name) : StreamControl(), name(name) {}
      ~Attr() {}
      bool handle(XMLWriter& writer) const { return writer.curr_state->handleAttr(writer, *this); }
      void print(std::ostream& os) const { os << "XML::Attr(" << name << ")"; }
      const std::string& getName() const { return name; }
    private:
      std::string name;
    };
    
  public:
    static void test();
    
  public:
    XMLWriter();
    XMLWriter(std::ostream* pos);
    XMLWriter(const char* fn);
    ~XMLWriter();

    bool open(const char* fn);
    bool attach(std::ostream* pos);
    // this causes us to forget the stream
    bool close();

    XMLWriter& operator<<(const StreamControl& control);
    // this is kind of annoying, but if we're not explicit, the template version gets called
    XMLWriter& operator<<(const Prolog& control)
    { return *this << static_cast<const StreamControl&>(control); }
    XMLWriter& operator<<(const Tag& control)
    { return *this << static_cast<const StreamControl&>(control); }
    XMLWriter& operator<<(const BareTag& control)
    { return *this << static_cast<const StreamControl&>(control); }
    XMLWriter& operator<<(const EmptyElement& control)
    { return *this << static_cast<const StreamControl&>(control); }
    XMLWriter& operator<<(const EndTag& control)
    { return *this << static_cast<const StreamControl&>(control); }
    XMLWriter& operator<<(const CloseTag& control)
    { return *this << static_cast<const StreamControl&>(control); }
    XMLWriter& operator<<(const Attr& control)
    { return *this << static_cast<const StreamControl&>(control); }
    template <class T>
    XMLWriter& operator<<(const T& t)
    {
      std::ostringstream os;
      // Set our format flags from the base stream
      os.flags(pos->flags());
      os.fill(pos->fill());
      os.precision(pos->precision());
      os.width(pos->width());
      os << t;
      // Now, put the flags back on our base stream because the last op
      // could have changed them
      pos->flags(os.flags());
      pos->fill(os.fill());
      pos->precision(os.precision());
      pos->width(os.width());
      curr_state->handleString(*this, os.str());
      return *this;
    }
    
    void closeAllTags();

    bool wasFailure() const { return failure; }

  public:

  protected:
    class State
    {
    public:
      State() {}
      virtual ~State() {}

      // we pass the writer in so that we can keep only one copy of the state
      virtual bool handleProlog(XMLWriter& writer, const Prolog& prolog);
      virtual bool handleTag(XMLWriter& writer, const Tag& tag);
      virtual bool handleCloseTag(XMLWriter& writer, const CloseTag& close_tag);
      virtual bool handleEndTag(XMLWriter& writer, const EndTag& end_tag);
      virtual bool handleAttr(XMLWriter& writer, const Attr& attr);
      virtual bool handleString(XMLWriter& writer, const std::string& str);
      virtual bool handleClose(XMLWriter& writer);

      virtual void print(std::ostream& os) const = 0;
      friend std::ostream& operator<<(std::ostream& os, const State& state)
      { state.print(os); return os; }
    };

    class StateEmpty
      : public State
    {
    public:
      bool handleProlog(XMLWriter& writer, const Prolog& prolog);
      // we're allowing a completely empty file
      bool handleClose(XMLWriter& writer) { return true; }
      void print(std::ostream& os) const { os << "StateEmpty"; }
    };
    class StatePrologDone
      : public State
    {
    public:
      bool handleTag(XMLWriter& writer, const Tag& tag);
      void print(std::ostream& os) const { os << "StatePrologDone"; }
    };
    class StateOpenTag
      : public State
    {
    public:
      bool handleTag(XMLWriter& writer, const Tag& tag);
      bool handleCloseTag(XMLWriter& writer, const CloseTag& close_tag);
      bool handleEndTag(XMLWriter& writer, const EndTag& end_tag);
      bool handleAttr(XMLWriter& writer, const Attr& attr);
      // this affects the tag name!
      bool handleString(XMLWriter& writer, const std::string& str);
      bool handleClose(XMLWriter& writer);
      void print(std::ostream& os) const { os << "StateOpenTag"; }
    };
    class StateOpenAttr
      : public State
    {
    public:
      bool handleTag(XMLWriter& writer, const Tag& tag);
      bool handleCloseTag(XMLWriter& writer, const CloseTag& close_tag);
      bool handleEndTag(XMLWriter& writer, const EndTag& end_tag);
      bool handleAttr(XMLWriter& writer, const Attr& attr);
      bool handleString(XMLWriter& writer, const std::string& str);
      bool handleClose(XMLWriter& writer);
      void print(std::ostream& os) const { os << "StateOpenAttr"; }
    };
    class StateBody
      : public State
    {
    public:
      bool handleTag(XMLWriter& writer, const Tag& tag);
      bool handleEndTag(XMLWriter& writer, const EndTag& end_tag);
      bool handleString(XMLWriter& writer, const std::string& str);
      bool handleClose(XMLWriter& writer);
      void print(std::ostream& os) const { os << "StateBody"; }
    };
    class StateDone
      : public State
    {
    public:
      bool handleClose(XMLWriter& writer) { return true; }
      void print(std::ostream& os) const { os << "StateDone"; }
    };


  protected:
    void setFailBit();
    void writeArmoredString(const std::string& str);
    
  protected:
    friend class State;
    friend class StateEmpty;
    friend class StatePrologDone;
    friend class StateOpenTag;
    friend class StateOpenAttr;
    friend class StateBody;
    friend class StateBodyDone;

    typedef std::stack<std::string> TagStack;

    // the current writing state
    State* curr_state;

    // the stack of tags
    TagStack tag_stack;
    
    // this memory is NOT deleted
    std::ostream* pos;
    // this memory is deleted because we allocate it
    std::ofstream* pfile;

    // set to true if any protocol failure happened
    bool failure;
    
  protected:
    //static vars
    static StateEmpty static_state_empty;
    static StatePrologDone static_state_prolog_done;
    static StateOpenTag static_state_open_tag;
    static StateOpenAttr static_state_open_attr;
    static StateBody static_state_body;
    static StateDone static_state_done;

  }; // class XMLWriter
  
  
} // namespace spades


#endif
