/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* A class which provides a C++ interface to expat */

#ifndef XML_PARSER_H_
#define XML_PARSER_H_

#include <expat.h>
#include <string>

namespace spades
{
  /*******************************************************************/
  /* A base class for XML parsing routines */
  class XMLParser
  {
  public:
    XMLParser();
    virtual ~XMLParser();

    /* returns where parsing was sucessful. Return is !getError */
    bool parseFile(const char* fn);

    bool getError() const { return error; }
    void setError();
    void clearError() { error = false; }
    
    //Here is how we pass control from one parser to another
    // the parent gives control over parsing to this object
    void takeOver(XMLParser* parent);
    //called by the child to return control to the parent
    void returnControl(XMLParser* child);

  protected:
    //default is empty
    virtual void notifyStartParse() {}
    //default is empty
    virtual void notifyEndParse() {}
    //default just calls parent
    virtual void startElementHandler(const XML_Char *name, const XML_Char **atts);
    //defauly just calls parent
    virtual void endElementHandler(const XML_Char *name);
    //default adds to curr_text_string
    virtual void characterDataHandler(const XML_Char *s, int len);
    
    XMLParser* getParent() { return parent; }
    const char* getCurrentFN() const { return current_fn; }

    //verifies that the namespace is the one given
    // returns the non-namespace part of the string
    // or '' if the namespace doesn't match
    // If there is no namespace, just returns a string
    // This is to get around a possible bug of expat where it forgets
    // the namespace used
    static std::string extractAndVerifyNamespace(const char* name,
						 const char* expected_namespace);
    // Finds the given attribute in the list
    // If there is a namespace, it must be the expected namespace,
    // but if there is not namespace, that's okay too
    // returns a string of the value. "" on error
    const char* extractAttribute(const char** atts,
				 const char* expected_namespace,
				 const char* attr_name,
				 bool warning_on_not_found);
    
    // these virtual functions will be called BEFORE the real handling
    virtual void virtualTakeOver(XMLParser* parent) {}
    virtual void virtualReturnControl(XMLParser* child) {}
    
    XML_Parser parser; //the expat parser pointer
    std::string curr_text_string;
    
  private:
    void installHandlers();

    XMLParser* parent;
    bool error;
    const char* current_fn;

  private:
    // these are the handler we will register with expat
    static void s_startElementHandler(void* data, const XML_Char *name, const XML_Char **atts);
    static void s_endElementHandler(void* data, const XML_Char *name);
    static void s_characterDataHandler(void *userData, const XML_Char *s, int len);
    
    static const int BUF_SIZE = 1024;
    static const char NAMESPACE_SEP = '|';
  };
  
  /*******************************************************************/
  //a simple 
  class XMLParser_Print
    : public XMLParser
  {
  public:
    XMLParser_Print(std::ostream& os) : XMLParser(), os(os) {}
    ~XMLParser_Print() {}

  protected:
    void startElementHandler(const XML_Char *name, const XML_Char **atts);
    void endElementHandler(const XML_Char *name);

  private:
    std::ostream& os;
  };
  
} //spades namespace


#endif
