/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* This class provides an arbitrary amount of write buffering for the
   provided file descriptor */

#ifndef WRITE_BUFF_FD_H_
#define WRITE_BUFF_FD_H_

#include <deque>
#include <map>
#include <iostream>
#include "FDSet.hpp"
#include "Writable.hpp"
#include "TimeVal.hpp"

namespace spades
{

  class WriteBuffFD
    : public Writable
  {
  public:
    WriteBuffFD(int new_fd = -1);
    virtual ~WriteBuffFD();
  
    //returns false only on unrecoverable errors
    //This may buffer the message
    virtual bool write (const char* buf, int buflen);

    //*pmax is the largest value in the set
    void addToSet(fd_set* ps, int* pmax);
    bool isInSet(const fd_set* ps);

    int getFD() const { return fd; }
  
    virtual bool close();
  
    friend std::ostream& operator<<(std::ostream& o, const WriteBuffFD& f)
    { o << "WriteBuffFD(fd="<<f.fd<<")"; return o; }

    static const FDSet& getPendingWriteSet() { return fds_pending_writes; }
    //returns whether the socket still has queued messages
    static bool tryPendingWriteFor(int fd);
    //returns the number of descriptors processed
    static int tryPendingWriteFor(fd_set write_set);
    //waits the specified amount of time for all write to be finished, calls select
    //returns false if something didn't get flushed in time
    static bool waitForPendingWrites(const TimeVal& tv_max, bool show_progress_interval);
    
  protected:

    class WriteBufferElement
    {
    public:
      WriteBufferElement(const char* b, int l);
      WriteBufferElement(const WriteBufferElement& wbe);
      ~WriteBufferElement();

      const WriteBufferElement& operator=(const WriteBufferElement& wbe);
    
      char* getData() const { return buf; }
      int getDataLen() const { return buflen; }

      //removed this many bytes from the front of the element
      void cutInitialSegment(int count);
    
    private:
      char* buf;
      int buflen;
    };

    typedef std::deque<WriteBufferElement> WriteBuffer;

    /* returns whether the write queue still has something in it */
    bool tryWriteBuffer();
    void addToWriteBuffer(const char* b, int l);
  
    WriteBuffer qWriteBuffer;

    unsigned total_bytes_written;
  
    virtual void setFD(int new_fd);

    /* statis elements */
    static FDSet fds_pending_writes;
  
    /* maps file descriptors to the memory location */
    typedef std::map<int, WriteBuffFD*> FD2MemMap;
    static FD2MemMap s_fd_2_mem_map;

  private:
    //we have to make this private so that we can maintain the FD2MemMap correctly
    //The file descriptor of the socket
    int fd;

    // these this functions should not be used! It is only used to catch errors
    // it is not defined
    WriteBuffFD(const WriteBuffFD& s);
    const WriteBuffFD& operator=(const WriteBuffFD& s);


  };


} //spades namespace

#endif
