/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* This class provides the base class that a particular world model must
   implement in order to function with the simulation engine */

#ifndef WORLDMODEL_H_
#define WORLDMODEL_H_

#include <iostream>
#include "Event.hpp"
#include "sharedtypes.hpp"
#include "EngineParam.hpp"
#include "DataArray.hpp"
#include "AgentTypeDB.hpp"

namespace spades
{

  class SimEngine;
  class ActEvent;
  class SenseEvent;
  
  class WorldModel
  {
  public:
    WorldModel () {}
    virtual ~ WorldModel () {}

    /* You probably want to inherit some parameters from EngineParam and use that
       to parse the commandline. See the ParamReader class */
    virtual EngineParam *parseParameters (const int argc, const char *const *argv) = 0;

    /* Most of the real initialization/finalization work should be done here
       We do this so that the SimEngine can initialize after this class is
       created */
    virtual bool initialize (SimEngine * pSE) = 0;
    virtual bool finalize () = 0;

    /* This function should advance the world forward. Events can be enqueed from
       this function. The time does *not* have to advance all the way to the
       time_desired, and if you put events in the queue, you should not advance
       past the time of those events */
    virtual SimTime simToTime (SimTime time_curr, SimTime time_desired) = 0;

    /* we used to have a realizeEvent method. Now all of that functionality is
       delegated to the events themselves */

    /* Monitors are programs that can open up a connection to the engine and
       get periodic updates about the state of the world. These next three functions
       support this */
    /* This function is called once for every monitor. It should return any header/setup
       information that is needed */
    virtual DataArray getMonitorHeaderInfo () = 0;
    /* This function will be called periodically to get information about the current
       state of the world. The format is completely determined by what the monitors
       will expect; no processing is done by the simulation engine */
    virtual DataArray getMonitorInfo (SimTime time) = 0;
    /* If a monitor sends information to the world model, this function is called to
       process it. Note that only the data section of the message (not the ID part which
       indicates that it is a message for the world model and not the simulation engine)
       is included here. If you need to keep the data, you must copy it */ 
    virtual void parseMonitorMessage (const char* data, unsigned datalen) = 0;
    
    /* There is a latency (in simulation time) between when an action is sent by the
       agent and when it takes effect. This function needs to return the minimum of
       all possible values of that latency. This is used to reason about causality */
    virtual SimTime getMinActionLatency() const = 0;

    /* Similar to getMinActionLatency, but the latency between when the sensation
       is generated from the world and when it is received by the agent */
    virtual SimTime getMinSenseLatency() const = 0;

    /* This function parses the action string received from the agents
       Note that this method is const! parsing the act can *not* affect the world
       at all, since this could then violate causality.
       Also, at some point this functionality may be moved to the communication server
       (which manages communication with the agents) */
    virtual ActEvent* parseAct(SimTime t, AgentID a, const char* data, unsigned datalen) const = 0;
  
    /* When the simulation is paused, this is called peridically
       At some point, this function should unpause the simulation
       The simulation starts out in paused mode */
    virtual void pauseModeCallback () = 0;

    /* The world model initiates startup requests for agents. This function is called
       once the agent has been started successfully */
    virtual bool agentConnect (AgentID agent, AgentTypeDB::AgentTypeConstIterator at) = 0;
    /* Any time an agent leaves the simulation (whether by crash, graceful exit request,
       etc.), this function is called so that the world model can update it's data
       structures */
    virtual bool agentDisappear (AgentID agent, AgentLostReason reason) = 0;

    /* Called every time a comm server connects. Usually, this can be
       ignored as the WorldModel does not need to know when and how
       many commservers connect, unless it wants to do it's own load balancing */
    virtual void notifyCommserverConnect (ServerID s) = 0;
    /* Called every time a comm server dsiconnects. Usually, this can be
       ignored as the WorldModel does not need to know when and how
       many commservers connect, unless it wants to do it's own load balancing */
    virtual void notifyCommserverDisconnect (ServerID s) = 0;
    
  };

} //spades namespace

#endif
