/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/


#ifndef TO_CS_MESSAGE_H_
#define TO_CS_MESSAGE_H_

#include "DataArray.hpp"
#include "LengthPrefixedWriter.hpp"
#include "IntegratedAgent.hpp"

namespace spadesalib
{

  /******************************************************************/

  class ToCSMessage
  {
  public:
    ToCSMessage() {}
    virtual ~ToCSMessage() {}

    virtual void print (std::ostream & o) const = 0;
    friend std::ostream & operator << (std::ostream & o, const ToCSMessage & m)
    { m.print (o); return o; }

    // How this message affects the count of outstanding think messages
    virtual int getOutstandingThinkMod() const = 0;

    virtual bool writeTo(spades::LengthPrefixedWriter& lp) = 0;

    virtual void sendIntegrated(spades::IntegratedAgentActions* pcallbacks) = 0;
  
  private:
  
  };

  /******************************************************************/

  class ToCSMessage_Act
    : public ToCSMessage
  {
  public:
    ToCSMessage_Act() : ToCSMessage() {}
    ToCSMessage_Act(const spades::DataArray& d) : ToCSMessage(),data(d) {}
    ~ToCSMessage_Act() {}

    void print (std::ostream & o) const;
    int getOutstandingThinkMod() const { return 0; }
    bool writeTo(spades::LengthPrefixedWriter& lp);
    void sendIntegrated(spades::IntegratedAgentActions* pcallbacks) 
    { pcallbacks->act(data.toString()); }

  private:
    spades::DataArray data;
  };

  /******************************************************************/

  class ToCSMessage_RequestTimeNotify
    : public ToCSMessage
  {
  public:
    ToCSMessage_RequestTimeNotify(int t = -1) : ToCSMessage(),time(t) {}
    ~ToCSMessage_RequestTimeNotify() {}

    void print (std::ostream & o) const;
    int getOutstandingThinkMod() const { return 0; }
    bool writeTo(spades::LengthPrefixedWriter& lp);
    void sendIntegrated(spades::IntegratedAgentActions* pcallbacks) 
    { pcallbacks->requestTimeNotify(time); }

  private:
    int time;
  };

  /******************************************************************/

  class ToCSMessage_MigrationData
    : public ToCSMessage
  {
  public:
    ToCSMessage_MigrationData() : ToCSMessage() {}
    ToCSMessage_MigrationData(const spades::DataArray& d) : ToCSMessage(),data(d) {}
    ~ToCSMessage_MigrationData() {}

    void print (std::ostream & o) const;
    int getOutstandingThinkMod() const { return 0; }
    bool writeTo(spades::LengthPrefixedWriter& lp);
    void sendIntegrated(spades::IntegratedAgentActions* pcallbacks) 
    { pcallbacks->migrationData(data.toString()); }

  private:
    spades::DataArray data;
  };

  /******************************************************************/

  class ToCSMessage_DoneThinking
    : public ToCSMessage
  {
  public:
    ToCSMessage_DoneThinking() : ToCSMessage() {}
    ~ToCSMessage_DoneThinking() {}

    void print (std::ostream & o) const;
    int getOutstandingThinkMod() const { return -1; }
    bool writeTo(spades::LengthPrefixedWriter& lp);
    void sendIntegrated(spades::IntegratedAgentActions* pcallbacks) 
    { pcallbacks->doneThinking(); }

  private:
  };

  /******************************************************************/

  class ToCSMessage_Exit
    : public ToCSMessage
  {
  public:
    ToCSMessage_Exit() : ToCSMessage() {}
    ~ToCSMessage_Exit() {}

    void print (std::ostream & o) const;
    int getOutstandingThinkMod() const { return -1; }
    bool writeTo(spades::LengthPrefixedWriter& lp);
    void sendIntegrated(spades::IntegratedAgentActions* pcallbacks) 
    { pcallbacks->exit(); }

  private:
  };

  /******************************************************************/

  class ToCSMessage_InitDone
    : public ToCSMessage
  {
  public:
    ToCSMessage_InitDone() : ToCSMessage() {}
    ~ToCSMessage_InitDone() {}

    void print (std::ostream & o) const;
    int getOutstandingThinkMod() const { return 0; }
    bool writeTo(spades::LengthPrefixedWriter& lp);
    void sendIntegrated(spades::IntegratedAgentActions* pcallbacks) 
    { pcallbacks->initDone(); }

  private:
  };

  /******************************************************************/

  class ToCSMessage_RequestCurrentThinkTime
    : public ToCSMessage
  {
  public:
    ToCSMessage_RequestCurrentThinkTime() : ToCSMessage() {}
    ~ToCSMessage_RequestCurrentThinkTime() {}

    void print (std::ostream & o) const;
    int getOutstandingThinkMod() const { return 0; }
    bool writeTo(spades::LengthPrefixedWriter& lp);
    void sendIntegrated(spades::IntegratedAgentActions* pcallbacks) 
    { pcallbacks->requestCurrentThinkTime(); }

  private:
  };

  /******************************************************************/
  /******************************************************************/
  /******************************************************************/

} //spadesalib namespace

#endif
