/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* these classes provide a messages to be sent to the agent */

#ifndef TO_AGENT_MESSAGE_H_
#define TO_AGENT_MESSAGE_H_

#include <string.h>
#include <iostream>
#include "sharedtypes.hpp"
#include "DataArray.hpp"

namespace spades
{

  class AgentInterface; //defined in AgentInterface.hpp

  /* A general note on how this is structured:
     The work involved in setting the agent interface to process the message like starting
     a thinking cycle, updating the simulation time, etc is done in the process method here
     The actual work of sending bytes to the agent is done by the AgentInterface::sendMessageTo
     method. This is to support an AgentInterface through pipes as well as a DL loaded one
     in the same process (and any other interface I want to add later) */

  /* for now, we'll privide no ordering, since the order should be the same as they came in */
  
  /* the virtual base class */
  class ToAgentMessage
  {
  public:
    ToAgentMessage() {}
    virtual ~ToAgentMessage() {}

    // If true, then this message should be sent immediately, even if the
    // agent is thinking
    virtual bool forImmediateSend() const = 0;
    
    virtual bool process(AgentInterface* inter) = 0;
    
    virtual void print (std::ostream & o) const = 0;
    friend std::ostream & operator << (std::ostream & o, const ToAgentMessage& m)
    { m.print (o); return o; }
  };

  /* ToAgentMessage_TimeNotify */
  class ToAgentMessage_TimeNotify
    : public ToAgentMessage
  {
  public:
    ToAgentMessage_TimeNotify(ThinkingType th, SimTime t = SIMTIME_INVALID)
      : ToAgentMessage(), thinking(th), time(t) {}
    ~ToAgentMessage_TimeNotify() {}

    bool forImmediateSend() const { return false; }
    
    bool process(AgentInterface* inter);

    void print (std::ostream & o) const
    { o << "ToAgentMessage_TimeNotify("
	<< "th: " << thinking << ", "
	<< "t: " << time << ")"; }

    ThinkingType getThinking() const { return thinking; }
    SimTime getTime() const { return time; }
    
  private:
    ThinkingType thinking;
    SimTime time;
  };

  /* ToAgentMessage_Sense */
  class ToAgentMessage_Sense
    : public ToAgentMessage
  {
  public:
    ToAgentMessage_Sense(ThinkingType th, SimTime st, SimTime at, const DataArray& d)
      : ToAgentMessage(), thinking(th), sendtime(st), arrivetime(at), data(d) {}
    ~ToAgentMessage_Sense() {}

    bool forImmediateSend() const { return false; }

    bool process(AgentInterface* inter);

    void print (std::ostream & o) const
    { o << "ToAgentMessage_Sense( st: "
	<< sendtime << ", at: " << arrivetime << ", th: " << thinking << ")"; }

    ThinkingType getThinking() const { return thinking; }
    SimTime getSendTime() const { return sendtime; }
    SimTime getArriveTime() const { return arrivetime; }
    const DataArray& getData() const { return data; }
    
  private:
    ThinkingType thinking;
    SimTime sendtime;
    SimTime arrivetime;
    DataArray data;
  };

  /* ToAgentMessage_MigrationRequest */
  class ToAgentMessage_MigrationRequest
    : public ToAgentMessage
  {
  public:
    ToAgentMessage_MigrationRequest()
      : ToAgentMessage() {}
    ~ToAgentMessage_MigrationRequest() {}

    bool forImmediateSend() const { return false; }

    bool process(AgentInterface* inter);

    void print (std::ostream & o) const { o << "ToAgentMessage_MigrationRequest()"; }

  private:
  };

  class ToAgentMessage_Exit
    : public ToAgentMessage
  {
  public:
    ToAgentMessage_Exit()
      : ToAgentMessage() {}
    ~ToAgentMessage_Exit() {}

    bool forImmediateSend() const { return false; }

    bool process(AgentInterface* inter);

    void print (std::ostream & o) const { o << "ToAgentMessage_Exit()"; }

  private:
  };

  class ToAgentMessage_InitData
    : public ToAgentMessage
  {
  public:
    ToAgentMessage_InitData(const DataArray& d) 
      : ToAgentMessage(),data(d) {}
    ~ToAgentMessage_InitData() {}

    bool forImmediateSend() const { return false; }

    bool process(AgentInterface* inter);

    void print (std::ostream & o) const
    { o << "ToAgentMessage_InitData(size: " << data.getSize() << ")"; }

    const DataArray& getData() const { return data; }

  private:
    DataArray data;
    
  };

  class ToAgentMessage_Error
    : public ToAgentMessage
  {
  public:
    enum ErrorType
      {
	ET_None,
	ET_NoToken,
	ET_ActWhenNotThinking,
	ET_RTNWhenNotThinking,
	ET_BadRTNTime,
	ET_DoneThinkingWhenNotThinking,
	ET_MigDataWhenNotMig,
	ET_BadToken,
	ET_InitDoneWhenNotInit,
        ET_ReqCurrThinkTimeWhenNotThinking
      };
    
  public:
    ToAgentMessage_Error(ErrorType et = ET_None) : ToAgentMessage(), et(et) {}
    ~ToAgentMessage_Error() {}

    bool forImmediateSend() const { return true; }

    bool process(AgentInterface* inter);

    void print (std::ostream & o) const
    { o << "ToAgentMessage_Error(et: " << et << ")"; }

    ErrorType getErrorType() const { return et; }
    
  private:
    ErrorType et;
  };

  inline std::ostream & operator << (std::ostream & o, const ToAgentMessage_Error::ErrorType& et)
  {
    static const char* ERROR_TYPE_STRINGS[] = 
      { "none",
	"no_token_on_line",
	"act_when_not_thinking",
	"rtn_when_not_thinking",
	"bad_time_in_rtn",
	"done_thinking_when_not_thinking",
	"mig_data_when_not_mig",
	"bad_token",
	"init_done_when_not_init"
	"req_curr_think_time_when_not_thinking"
      };
    o << ERROR_TYPE_STRINGS[static_cast<int>(et)];
    return o;
  }
  
  /* ToAgentMessage_ThinkTime */
  class ToAgentMessage_ThinkTime
    : public ToAgentMessage
  {
  public:
    ToAgentMessage_ThinkTime(SimTime t, bool inter)
      : ToAgentMessage(), time(t), intermediate(inter) {}
    ~ToAgentMessage_ThinkTime() {}

    // if it's an intermediate think time point,
    // we want to send it immediately
    bool forImmediateSend() const { return intermediate; }

    bool process(AgentInterface* inter);

    void print (std::ostream & o) const
    { o << "ToAgentMessage_ThinkTime(" << time << ", " << intermediate << ")"; }

    SimTime getTime() const { return time; }
    bool getIntermediate() const { return intermediate; }
    
  private:
    SimTime time;
    // if true, this message is a response to mid-thinking request
    // by an agent for it's current simtime
    bool intermediate;
  };

  
} //spades namespace

#endif
