/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* a cover around struct timeval */

#ifndef TIME_VAL_H_
#define TIME_VAL_H_

#include "spades-config.h"
#if SPADES_TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if SPADES_HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#include <iostream>

namespace spades
{

  /* Conceptually, this class represents the amount of time since some point
     gettimeofday (for which there is a cover function below) considers this from
     some particular date in the past. This function can also be used to specify
     timeouts and such.
     It is the user of this classes responsibility to keep track of what this time
     is relative to.
  */
  class TimeVal
  {
  public:
    TimeVal() { timerclear(&base_tv); }
    TimeVal(double seconds) { setFromSeconds(seconds); }
    TimeVal(long tv_sec, long tv_usec) { base_tv.tv_sec = tv_sec; base_tv.tv_usec = tv_usec; }
    TimeVal(const struct timeval& tv) : base_tv(tv) {}
    ~TimeVal() {}

    //the default op=, copy constructor etc all work fine

    void clear() { timerclear(&base_tv); }
    
    bool isSet() const { return timerisset(&base_tv); }
    
    //comparison operators
    bool operator==(const TimeVal& tv) const { return timercmp(&base_tv, &tv.base_tv, ==); }
    bool operator!=(const TimeVal& tv) const { return timercmp(&base_tv, &tv.base_tv, !=); }
    bool operator<(const TimeVal& tv) const { return timercmp(&base_tv, &tv.base_tv, <); }
    bool operator<=(const TimeVal& tv) const { return timercmp(&base_tv, &tv.base_tv, <=); }
    bool operator>(const TimeVal& tv) const { return timercmp(&base_tv, &tv.base_tv, >); }
    bool operator>=(const TimeVal& tv) const { return timercmp(&base_tv, &tv.base_tv, >=); }

    // math ops
    TimeVal operator+(const TimeVal& tv) const
    { TimeVal rettv(*this); rettv += tv; return rettv; }
    TimeVal operator-(const TimeVal& tv) const
    { TimeVal rettv(*this); rettv -= tv; return rettv; }
    const TimeVal& operator+=(const TimeVal& tv);
    const TimeVal& operator-=(const TimeVal& tv);
    
    double   toSeconds() const;
    TimeVal& setFromSeconds(double sec);

    struct timeval* getBaseTimeVal() { return &base_tv; }
    const struct timeval* getBaseTimeVal() const { return &base_tv; }

    const TimeVal& setFromTimeOfDay();
    
    friend std::ostream& operator<<(std::ostream& os, const TimeVal& tv);
    friend std::istream& operator>>(std::istream& is, TimeVal& tv);
    
  private:
    struct timeval base_tv;
  };
  
  
} //spades namespace

#endif
