/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* This class provides a small abstraction over a TCP socket. There are several purposes
   to making this abstraction
   * in order to put more of the machine specific code in one place.
   * to provide additional read buffering (in case partially completed messages are received_
   ( this is unimplemented, and it is unclear whether it needs to be)
   * to provide extra write buffering
   */

#ifndef TCPSOCKET_H_
#define TCPSOCKET_H_

#include "spades-config.h"
#include <iostream>
#include "WriteBuffCollFD.hpp"
#include "ReadBuffFD.hpp"

namespace spades
{

  /* on creation, the TCPSocket is in an invalid state
     the function bindToPort and connect are used to create valid sockets */
  /* WriteBuffCollectorFD gather a bunch of writes together until a max size is reached
     or explictly flushed */
  class TCPSocket
    : public WriteBuffCollectorFD
  {
  public:
    static void printSocketInfoTo(std::ostream& os);
    
  public:
    TCPSocket();
    ~TCPSocket();

    //sets up the socket to bind to a port and listen for future connections
    bool bindToPort(int port);

    //connects this socket to a particular port on a machine
    bool connect(const char* hostname, int port);
  
    //returns a new socket if there is an connection waiting, NULL otherwise (or on error)
    TCPSocket* accept();

    //This turns off the TCP delay so that there is less latency
    bool setNoDelay();

    //sets the close on exec flag. By default, TCPSocket turns this on
    bool setCloseOnExec(bool turn_on = true);
    
    ReadBuffFD* getReader() { return &reader; }
    
    /* write functions come from WriteBuffCollectorFD */

    bool close();

    // returns whether the socket is still good
    bool check();
    
  
    static const int MAX_CONNECTIONS;
    static const int MAX_SEND_BUF_SIZE;
    
    friend std::ostream& operator<<(std::ostream& o, const TCPSocket& s)
    { o << "TCPSocket(fd="<<s.getFD()<<")"; return o; }

  protected:
    //also defined in WriteBuffCollectorFD
    void setFD(int new_fd);

    struct hostent* lookupHost(const char* name);

    // this is private because, according to tcp(7), you have to call this before connect or listen
    bool maximizeSendBuf();
    
  private:
    // should not be used -- not defined
    const TCPSocket& operator=(const TCPSocket& s);
    // should not be used -- not defined
    TCPSocket(const TCPSocket& s);
    
    TCPSocket(int new_fd);

    ReadBuffFD reader;
  };

} //spades namespace


#endif
