/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* This file contains a class which is the main coordinator for the simulator
   engine */

#ifndef SIMENGINE_H_
#define SIMENGINE_H_

#include <set>
#include <list>
#include <functional>
#include "WorldModel.hpp"
#include "Communication.hpp"
#include "AgentCSAllocManager.hpp"
#include "Event.hpp"
#include "FixedAgentEvent.hpp"
#include "MonitorManager.hpp"

namespace spades
{

  /******************************************************************/
  enum SimulationMode {
    SM_None, //should never be the actual mode
    SM_RunNormal,
    SM_RunLimitedRate,
    SM_PausedInitial,
    SM_PausedMonitor,
    SM_PausedWorldModel,
    SM_Shutdown
  };

  inline bool isPauseMode(SimulationMode m)
  { return (m == SM_PausedInitial) || (m == SM_PausedMonitor) ||
      (m == SM_PausedWorldModel) || (m == SM_Shutdown); }

  /******************************************************************/
  /* Helpers that need to be before SimEngine */

  struct event_lt : public std::binary_function <Event *, Event *, bool>
  {
    bool operator () (Event * pe1, Event * pe2);
  };

  /******************************************************************/
  /* these are defined below */
  class AgentInfo;
  class SimEngine; //the main class
  
  /******************************************************************/
  class SimEngineMessageProcessor
    : public CSEMessageVisitor
  {
  public:
    SimEngineMessageProcessor(SimEngine* p)
      : CSEMessageVisitor(), pSE(p), server(SERVERID_INVALID) {}
    ~SimEngineMessageProcessor() {}

    void visitAgentInitialized(CSEMessage_AgentInitialized* p);
    void visitAgentInitError(CSEMessage_AgentInitError* p);
    void visitExit(CSEMessage_Exit* p);
    void visitAct(CSEMessage_Act* p);
    void visitRequestTimeNotify(CSEMessage_RequestTimeNotify* p);
    void visitDoneThinking(CSEMessage_DoneThinking* p);
    void visitTimeUpdate(CSEMessage_TimeUpdate* p);
    void visitAgentLost(CSEMessage_AgentLost* p);
    void visitMigrationData(CSEMessage_MigrationData* p);
    void visitMachineLoad(CSEMessage_MachineLoad* p);
    void visitAgentTypes(CSEMessage_AgentTypes* p);
    
    void setServer(ServerID s) { server = s; }
    ServerID getServer() const { return server; }

  private:
    SimEngine* pSE;
    ServerID server;
  };
  
  
  /******************************************************************/
  /* The main class */

  class SimEngine
    : protected AgentCSAllocCallbacks
  {
  public:
    SimEngine (WorldModel * pWM);
    ~SimEngine ();

    bool mainLoop ();

    SimTime getSimulationTime () const { return sim_time; }
    int getNumAgents () const { return lAgent.size(); }
    int getNumCommServers () const;

    //This is the world models primary mode of talking back to the simulation engine
    bool enqueueEvent (Event * e);

    /* This moves an event to the queue in AgentInfo */
    bool enqueueAgentEvent (FixedAgentEvent* e);

    SimulationMode getSimulationMode() const { return sim_mode; }
    
    void changeSimulationMode(SimulationMode new_mode);
    bool inPauseMode() const { return isPauseMode(sim_mode); }
    bool inShutdownMode() const { return sim_mode == SM_Shutdown; }
    
    AgentID startNewAgent (AgentTypeDB::AgentTypeConstIterator at, ServerID server = SERVERID_INVALID);

    bool areAllAgentsInitialized();

    /* This function should be used to kill an agent which has started cleanly
       It sends a message to the agent notifying it of exit */
    bool killAgent (AgentID a);

    // returns whether this agent has been lost
    bool hasAgentBeenLost(AgentID a)
    { return lLostAgent.count(a) != 0; }
    
    // a shortcut function for the WorldModel
    void sendExtraMonitorInfo( const DataArray& d) { monitor_manager.sendExtraMonitorInfo(d); }
    
    void initiateShutdown() { changeSimulationMode(SM_Shutdown); }

    const TimeVal& getCurrWallClockTime() const { return tv_curr; }

    WorldModel* getWorldModel() const { return pWorldModel; }
    Communication* getCommunication() { return &commengine; }
    MonitorManager* getMonitorManager() { return &monitor_manager; }
    AgentCSAllocManager* getAgentCSAllocManager() { return &agent_alloc_manager; }

    AgentTypeDB* getAgentTypeDB() { return &agent_type_db; }
    const AgentTypeDB* getAgentTypeDB() const { return &agent_type_db; }
    
    // sends a message to a communication server. returns true on success
    bool sendCommMessage (ServerID s, ECSMessage * m)
    { return commengine.sendCommMessage(s, m); }
    /* this function is used internally by the SimEngine also */
    bool sendAgentMessage (AgentID agent, ECSMessage * m);

  protected:
    friend class AgentCSAllocManager;
    /* This set of functions implementes AgentCSAllocCallbacks */
    void notifyCommserverReady(ServerID s);
    bool canAgentMigrate(AgentID a) const;
    void notifyAgentMigrating(AgentID a, ServerID s);
    void notifyAgentDoneMigrating(AgentID a);

    const char* getAgentTypeName(AgentID agent);

    void disconnectCommServer(ServerID s) { commengine.disconnectCommServer(s); }

  protected:
    /* These are functions meant be called by Communication */
    friend class Communication;
    void notifyCommserverConnect(ServerID s);
    void notifyCommserverDisconnect(ServerID s);

  protected:
    // will delete the event if the realizeEvent method says to
    bool realizeEvent(Event* e);

    friend class SenseEvent;
    friend class TimeNotifyEvent;
    void startThink(SimTime t, AgentID a, ThinkingType thinking, AgentInfo* painfo);
    
    friend class AgentInfo;
    
  protected:
    friend class SimEngineMessageProcessor;
    
    AgentInfo* getAgentInfo(AgentID a);
    
    /* This should be used internally to actually remove the agent data from the simulation
       engine. The only reason this is protected and not private is so the MessageProcessor
       can call it */
    bool removeAgent(AgentID a, AgentLostReason reason);
    
  private:

    /* Pulls out any events in the pending event queue which are agent events are have times
       less than max time and puts them into the agent queue */
    /* returns the number of events moved */
    int scanForAgentEvents (SimTime max_time);
  
    int processMessages ();

    //calculates the minimum time that any agent can affect the simulation
    //returns SIMTIME_INVALID for infinity
    SimTime calculateMinAgentTime ();

    //calculates the minimum time where any new sensation could be queued for an agent
    SimTime calculateMinSenseTime ();
  
    void actionlogPendingEvents(int level);

    void clearEventQueue();

    //if force, status will be printed, unless it was already printed at this time
    void printStatusUpdate(bool force = false);

    // this function will (if EngineParam::instance()->getErrorTrace)
    // print out lots of usefule debugging information to stderr about the current state
    // of the simulation
    void errorTrace();
    
    /* These functions are variables are used to time the wall clock time of the execution
       of the simulation.
       We do not want to add the time when the simulation is paused, for example, or the startup
       and shutdown time */
    void startTiming();
    void stopTiming();

    /* These functions are used for the rate limited mode of the simulation engine */
    //returns whether rebasing was done
    bool checkForRebaseLimitedRate();
    void rebaseLimitedRate();
    bool hasLimitedRateTimeArrived(SimTime t);
    TimeVal limitedRateTimeFor(SimTime t);

    Communication commengine;
    MonitorManager monitor_manager;
    AgentCSAllocManager agent_alloc_manager;
    AgentTypeDB agent_type_db;
    
    bool timing; 
    TimeVal elapsed_real_time;
    TimeVal curr_start_time;

    /* this is set after every waiting for events */
    TimeVal tv_curr;
  
    WorldModel *pWorldModel;

    SimTime sim_time;

    SimulationMode sim_mode;
    
    //The next agent ID to assign
    AgentID next_agent_id;
  
    // the time when the most recent pause was started
    // in a run mode, the time of the last resume
    TimeVal tv_pause_mode;

    //the last time we sent a status update
    TimeVal tv_last_status_update;
    SimTime last_status_simtime;
    
    //this is the last time that an event was realized
    TimeVal tv_last_event_time;

    //counts the number of events realized
    int total_events_processed;

    //these are for time limited mode
    double       limited_rate_sim_time_per_second;
    TimeVal      tv_limited_rate_base;
    SimTime      sim_time_limited_rate_base;
    //how much time to wait at the next select call
    TimeVal      tv_limited_rate_next;
    
    typedef std::multiset<Event *, event_lt> PendingEvents;
    PendingEvents pending_events;

    std::ofstream* ptext_event_log;
    
    typedef std::map<AgentID, AgentInfo, std::less<AgentID> > AgentList;
    typedef std::set<AgentID, std::less<AgentID> > LostAgentList;

    AgentList lAgent;
    LostAgentList lLostAgent;

    bool removeAgent(AgentList::iterator iter, AgentLostReason reason);

  };

} //spades namespace




#endif
