/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* This is an abstract base class for timing processes
   There are two virtual base classes: ProcessTimer and Timepoint
   A ProcessTimer is used to get timing information about a process and a
   Timepoint is used as a memento to mark the state of the process in time
   Timepoints have two public indexes to refer to them, known as major and
   minor number. Each time a new thinking cycle begins, the major index is
   incremented, Each time the agent manager or agent interface takes a new
   timepoint, the minor index is incremented
   The first minor index for any major index  has value 0

   All subclasses should have a constructor that does not take arguments
   This is so we can do a creation from a stream
*/

#ifndef PROCESS_TIMER_H_
#define PROCESS_TIMER_H_

// My config.h file -- all defines have a SPADES_ prefix
#include "spades-config.h"

#include <set>
#include <deque>
#include <map>
#include <iostream>
#include "sharedtypes.hpp"
#include "ThinkTimeFile.hpp"
#include "MessageServer.hpp"
#include "ProcessKillList.hpp"
#ifdef SPADES_HAVE_PERFCTR
namespace spades 
{
  class ExtPerfCtrFD;
}
#endif

namespace spades {
  
  /********************************************************************************/
  /* The base classes */
  
  class ProcessTimer
    : public ForkWatcher // from MessageServer
  {
  public:
    enum ProcessTimerType {
      PTT_None,
      PTT_Fixed,
      PTT_Replay,
      PTT_MultiTrack,
      PTT_Jiffies,
      PTT_PerfCtrInstr
    };

    typedef unsigned MajorIndex;
    typedef unsigned MinorIndex;
    
    class Timepoint
    {
    public:
      Timepoint(ProcessTimerType timer_type, MajorIndex major_idx, MinorIndex minor_idx)
	: timer_type(timer_type), major_idx(major_idx), minor_idx(minor_idx) {}
      virtual ~Timepoint() {}

      bool operator==(const Timepoint& t)
      { return timer_type == t.timer_type && major_idx == major_idx && minor_idx == minor_idx; }

      bool operator<(const Timepoint& t) const
      { if (major_idx < t.major_idx) return true; else return (minor_idx < t.minor_idx); }
      bool operator>(const Timepoint& t) const
      { if (major_idx > t.major_idx) return true; else return (minor_idx > t.minor_idx); }
      bool operator<=(const Timepoint& t) const
      { if (major_idx <= t.major_idx) return true; else return (minor_idx <= t.minor_idx); }
      bool operator>=(const Timepoint& t) const
      { if (major_idx >= t.major_idx) return true; else return (minor_idx >= t.minor_idx); }
      
      virtual void print (std::ostream & o) const
      { o << "Timepoint(" << major_idx << ", " << minor_idx << ")"; }
      friend std::ostream & operator << (std::ostream & o, const Timepoint& t)
      { t.print (o); return o; }

      MajorIndex getMajorIndex() const { return major_idx; }
      MinorIndex getMinorIndex() const { return minor_idx; }

    protected:
      friend class ProcessTimer;
      friend class ProcessTimer_Fixed;
      friend class ProcessTimer_Replay;
      friend class ProcessTimer_MultiTrack;
      friend class ProcessTimer_Jiffies;
      friend class ProcessTimer_PerfCtrInstr;
      ProcessTimerType getTimerType() const { return timer_type; }

    private:
      ProcessTimerType timer_type;
      MajorIndex major_idx;
      MinorIndex minor_idx;
    };

    // Now the ProcessTimer methods
  public:
    ProcessTimer(ProcessTimerType timer_type, ProcessTimerType timepoint_type)
      : pid_set(), removed_pid_set(), ppkl(NULL),
	timer_type(timer_type), timepoint_type(timepoint_type),
	last_major_idx(0), last_minor_idx(0) {}
    virtual ~ProcessTimer() {}

    // before the timer is actually used, the agentid has to be set, in case any
    // of the subclasses need it
    virtual void setAgentID(AgentID a) {}

    // whether this timer can be used for an agent which is integrated in the same process
    virtual bool acceptableForIntegratedAgent() = 0;

    // You have to be careful with this function!
    // This function is called after the fork but before the exec for each agent
    // This means that any transient data here will vanish at the exec
    // and you should NOT depend on the agentID or process list being correct
    // returns false on error
    virtual bool setupPostForkPreExec(const Message::MsgUniqueID& id) { return true; }
    
    // if a subclass needs a special Timepoint subclass, it can redefine this
    virtual Timepoint* getInitialTimepoint() const { return new Timepoint(timer_type, 0, 0); }
    
    Timepoint* getNextMajorTimepoint();
    Timepoint* getNextMinorTimepoint();

    //returns pt2 - pt1
    SimTime diffTimepoints(const Timepoint* pt1, const Timepoint* pt2);

    // clone will not copy the process list
    virtual ProcessTimer* clone() const = 0;

    // if pid_parent is -1, then the id is invalid
    bool addPid(MessageServer* pmserver, int pid,
		int pid_parent, const Message::MsgUniqueID& id);
    bool removePid(int pid);
    int getNumProcesses() const { return pid_set.size(); }

    //returns the number of valid processes left
    int checkProcesses();
    // returns the status of the process (the character in the status field of
    // /proc/<pid>/stat
    // a string with each appended
    std::string getProcessStatus();

    // This removes us as a watcher for all pids that we know about
    void unregisterForkWatchers(MessageServer* pmserver);

    // This removes all pids from the message server queue
    void flushReceivedFD(MessageServer* pmserver);
    
    // this implements the ForkWatcher interface
    void notifyFork(MessageServer* pmserver, int pid_parent, int pid_child, const Message::MsgUniqueID& id);

    //does NOT take over the memory
    void setProcessKillList(ProcessKillList* p) { ppkl = p; }
    
    virtual void print (std::ostream & o) const = 0;
    friend std::ostream & operator << (std::ostream & o, const ProcessTimer& pt)
    { pt.print (o); return o; }

    static ProcessTimer* createFromStream(std::istream& in);
    static ProcessTimer* getDefaultTimer();
    static void setDefaultTimer(ProcessTimer* pt);
    static void deleteDefaultTimer() { if (s_default_pt) delete s_default_pt; s_default_pt = NULL; }
    
  protected:
    ProcessTimerType getTimerType() const { return timer_type; }

    virtual Timepoint* prot_getNewTimepoint(MajorIndex maj, MinorIndex min) = 0;
    //pt2-pt1
    virtual SimTime prot_diffTimepoints(const Timepoint* pt1, const Timepoint* pt2) = 0;

    // return true on success
    // pid_parent can be -1 to indicate no parent
    virtual bool prot_notifyAddPid(MessageServer* pmserver, int pid,
				   int pid_parent, const Message::MsgUniqueID& id) = 0;
    virtual bool prot_notifyRemovePid(int pid) = 0;
    virtual bool readArgs(std::istream& in) = 0;

    // we let process kill list access our pids
    friend class ProcessKillList;
    
    typedef std::set<int, std::less<int> > PidSet;

    PidSet pid_set;
    PidSet removed_pid_set; // set of pids that have been removed
    
    bool removePid(PidSet::iterator iter);

    ProcessKillList* ppkl;
    
  private:
    ProcessTimerType timer_type;
    ProcessTimerType timepoint_type;
    MajorIndex last_major_idx;
    MinorIndex last_minor_idx;

    static ProcessTimer* s_default_pt;
  };


  /********************************************************************************/
  /* ProcessTimer_Fixed */
  // we don't need any special subclass of Timepoint for this
  class ProcessTimer_Fixed
    : public ProcessTimer
  {
  public:
    ProcessTimer_Fixed(SimTime fixed_time = SIMTIME_INVALID)
      : ProcessTimer(PTT_Fixed, PTT_Fixed), fixed_time(fixed_time) {}
    ProcessTimer_Fixed(const ProcessTimer_Fixed& pt)
      : ProcessTimer(PTT_Fixed, PTT_Fixed), fixed_time(pt.fixed_time) {}
    ~ProcessTimer_Fixed() {}

    bool acceptableForIntegratedAgent() { return true; }
    
    ProcessTimer_Fixed* clone() const { return new ProcessTimer_Fixed(*this); }

    void print (std::ostream & o) const { o << "ProcessTimer_Fixed(" << fixed_time << ")"; }
    
  protected:
    Timepoint* prot_getNewTimepoint(MajorIndex maj, MinorIndex min)
    { return new Timepoint(PTT_Fixed, maj, min); }
    SimTime prot_diffTimepoints(const Timepoint* pt1, const Timepoint* pt2);
    bool prot_notifyAddPid(MessageServer* pmserver, int pid,
			   int pid_parent, const Message::MsgUniqueID& id) { return true; }
    bool prot_notifyRemovePid(int pid) { return true; }
    bool readArgs(std::istream& in);
    
  private:
    SimTime fixed_time;
  };
  
  /********************************************************************************/
  /* ProcessTimer_Replay */
  // we don't need any special subclass of Timepoint for this
  class ProcessTimer_Replay
    : public ProcessTimer
  {
  public:
    ProcessTimer_Replay(std::string file_pattern = "");
    ProcessTimer_Replay(const ProcessTimer_Replay& pt);
    ~ProcessTimer_Replay() { think_time_file.close(); }

    bool acceptableForIntegratedAgent() { return true; }

    void setAgentID(AgentID a);

    ProcessTimer_Replay* clone() const { return new ProcessTimer_Replay(*this); }

    void print (std::ostream & o) const { o << "ProcessTimer_Replay('" << file_pattern << "')"; }
    
  protected:
    Timepoint* prot_getNewTimepoint(MajorIndex maj, MinorIndex min)
    { return new Timepoint(PTT_Replay, maj, min); }
    SimTime prot_diffTimepoints(const Timepoint* pt1, const Timepoint* pt2);
    bool prot_notifyAddPid(MessageServer* pmserver, int pid,
			   int pid_parent, const Message::MsgUniqueID& id) { return true; }
    bool prot_notifyRemovePid(int pid) { return true; }
    bool readArgs(std::istream& in);
    
  private:
    //returns number read
    int readMore();
    bool haveDataForMajorIndex(MajorIndex idx);

    typedef std::deque<SimTime> TimeStorage;

    std::string file_pattern;
    ThinkTimeFile think_time_file;
    TimeStorage time_storage;
    MajorIndex start_idx;
    MajorIndex end_idx; //one after the last one stored
  };

  /********************************************************************************/
  /* ProcessTimer_MultiTrack */
  /* This is a virtual class! It handles some of the yuckiness of handling
     the tracking of multiple processing (which real tracking needs to be done for
     each process). The above timers (Fixed and Replay) don't need this because they
     don't have to do anything special based on how many processes there.
     ProcessTimer s below this do need to do something special for each process */
  class ProcessTimer_MultiTrack
    : public ProcessTimer
  {
  public:
    class Timepoint_MultiTrack
      : public ProcessTimer::Timepoint
    {
    public:
      Timepoint_MultiTrack(int maj, int min)
	: Timepoint(PTT_MultiTrack, maj, min), maptp() {}
      ~Timepoint_MultiTrack();

      void print (std::ostream & o) const;

      void addTP(int pid, Timepoint* tp);
    protected:
      friend class ProcessTimer_MultiTrack;

      // map pids to timepoints
      typedef std::map<int, Timepoint*> PidTimepointMap;
      PidTimepointMap maptp;
    };
    
  public:
    ProcessTimer_MultiTrack(ProcessTimerType ptype)
      // says that the timer is ptype but the timepoints will be PTT_MultiTrack
      : ProcessTimer(ptype, PTT_MultiTrack) {}
    ~ProcessTimer_MultiTrack() {}

    Timepoint* getInitialTimepoint() const;
    
    //clone and print are left virtual
  protected:
    Timepoint* prot_getNewTimepoint(MajorIndex maj, MinorIndex min);
    SimTime prot_diffTimepoints(const Timepoint* pt1, const Timepoint* pt2);
    virtual bool prot_notifyAddPid(MessageServer* pmserver, int pid,
				   int pid_parent, const Message::MsgUniqueID& id) { return true; }
    virtual bool prot_notifyRemovePid(int pid) { return true; }
    //readArgs is left virtual

    // Now new virtual methods
    virtual Timepoint* getTimepointForPid(MajorIndex maj, MinorIndex min,
					  int pid, bool is_removed) = 0;
    virtual SimTime diffTimepointsForPid(const Timepoint* pt1, const Timepoint* pt2, int pid) = 0;
    virtual Timepoint* getInitialTimepointForPid(int pid) = 0;
    // This returns the amount of time that should be used if the process
    // associated with this timepoint vanishes
    virtual SimTime finalTimeFrom(const Timepoint* pt, int pid) = 0;

  };

  /********************************************************************************/
  /* ProcessTimer_Jiffies */
  class ProcessTimer_Jiffies
    : public ProcessTimer_MultiTrack
  {
  public:
#if SPADES_HAVE_JIFFIES
    class Timepoint_Jiffies
      : public ProcessTimer::Timepoint
    {
    public:
      // Initialized to the initial timepoint
      Timepoint_Jiffies(int maj, int min)
	: Timepoint(PTT_Jiffies, maj, min),
	  user_jiffies(0), kernel_jiffies(0),
	  child_user_jiffies(0), child_kernel_jiffies(0) {}
      ~Timepoint_Jiffies() {}

      void print (std::ostream & o) const
      { o << "Timepoint_Jiffies(" << getMajorIndex() << ", " << getMinorIndex()
	  << ", u:" << user_jiffies << ", k:" << kernel_jiffies
	  << ", cu:" << child_user_jiffies << ", ck:" << child_kernel_jiffies
	  << ")"; }

      bool readForPid(int pid);
      
      SimTime getInterval(const Timepoint_Jiffies& tp, int kinst_per_sim_step) const;
      
    private:
      int user_jiffies;
      int kernel_jiffies;
      // These represent the times of the child that have exited and been waited on
      int child_user_jiffies;
      int child_kernel_jiffies;
    };
    
  public:
    ProcessTimer_Jiffies(int kinst_per_sim_step = -1)
      : ProcessTimer_MultiTrack(PTT_Jiffies), kinst_per_sim_step(kinst_per_sim_step) {}
    ProcessTimer_Jiffies(const ProcessTimer_Jiffies& pt)
      : ProcessTimer_MultiTrack(PTT_Jiffies), kinst_per_sim_step(pt.kinst_per_sim_step) {}
    ~ProcessTimer_Jiffies() {}

    bool acceptableForIntegratedAgent() { return false; }

    ProcessTimer_Jiffies* clone() const { return new ProcessTimer_Jiffies(*this); }

    void print (std::ostream & o) const { o << "ProcessTimer_Jiffies(" << kinst_per_sim_step << ")"; }
    
  protected:
    bool readArgs(std::istream& in);
    Timepoint* getTimepointForPid(MajorIndex maj, MinorIndex min, int pid, bool is_removed);
    SimTime diffTimepointsForPid(const Timepoint* pt1, const Timepoint* pt2, int pid);
    Timepoint* getInitialTimepointForPid(int pid);
    // This returns the amount of time that should be used if the process
    // associated with this timepoint vanishes
    SimTime finalTimeFrom(const Timepoint* pt, int pid);
    //this is test code for trying out the code where pids get rejected
    //bool prot_notifyAddPid(MessageServer* pmserver, int pid, int pid_parent, const Message::MsgUniqueID& id) { return (pid_set.size() > 1) ? false : true; }
    
  private:
    int kinst_per_sim_step;

#else
  public:
    ProcessTimer_Jiffies() : ProcessTimer_MultiTrack(PTT_Jiffies) {}
    ProcessTimer_Jiffies(const ProcessTimer_Jiffies& pt) : ProcessTimer_MultiTrack(PTT_Jiffies) {}
    ~ProcessTimer_Jiffies() {}

    bool acceptableForIntegratedAgent() { return false; }

    ProcessTimer_Jiffies* clone() const { return new ProcessTimer_Jiffies(*this); }

    void print (std::ostream & o) const { o << "ProcessTimer_Jiffies(DUMMY)"; }
    
  protected:
    bool readArgs(std::istream& in) { return true; }
    Timepoint* getTimepointForPid(MajorIndex maj, MinorIndex min, int pid, bool is_removed);
    SimTime diffTimepointsForPid(const Timepoint* pt1, const Timepoint* pt2, int pid);
    Timepoint* getInitialTimepointForPid(int pid);
    // This returns the amount of time that should be used if the process
    // associated with this timepoint vanishes
    SimTime finalTimeFrom(const Timepoint* pt, int pid);
    bool prot_notifyAddPid(MessageServer* pmserver, int pid,
			   int pid_parent, const Message::MsgUniqueID& id);
    bool prot_notifyRemovePid(int pid) { return false; }

  private:
    static const char* ERRMSG;

#endif	
  };  

  /********************************************************************************/
  /* ProcessTimer_PerfCtrInstr */
  class ProcessTimer_PerfCtrInstr
    : public ProcessTimer_MultiTrack
  {
#ifdef SPADES_HAVE_PERFCTR	
  public:
    class Timepoint_PerfCtrInstr
      : public ProcessTimer::Timepoint
    {
    public:
      Timepoint_PerfCtrInstr(int maj, int min);
      ~Timepoint_PerfCtrInstr() {}

      void print (std::ostream & o) const
      { o << "Timepoint_PerfCtrInstr(" << getMajorIndex() << ", " << getMinorIndex()
	  << ", i:" << instr_count << ")"; }

      SimTime getInterval(const Timepoint_PerfCtrInstr& tp, int kinst_per_sim_step) const;

      // expects a fil descriptor for a perfctr 
      bool readFrom(ExtPerfCtrFD& pcfd);
      
    private:
      unsigned long long instr_count;
    };

  public:
    ProcessTimer_PerfCtrInstr(int kinst_per_sim_step = -1)
      : ProcessTimer_MultiTrack(PTT_PerfCtrInstr),
	kinst_per_sim_step(kinst_per_sim_step)
    { }
    ProcessTimer_PerfCtrInstr(const ProcessTimer_PerfCtrInstr& pt)
      : ProcessTimer_MultiTrack(PTT_PerfCtrInstr),
	kinst_per_sim_step(pt.kinst_per_sim_step)
    { }
    ~ProcessTimer_PerfCtrInstr();

    bool acceptableForIntegratedAgent() { return false; }

    // Important! see the comments in ProcessTimer about this function
    bool setupPostForkPreExec(const Message::MsgUniqueID& id);
    
    ProcessTimer_PerfCtrInstr* clone() const { return new ProcessTimer_PerfCtrInstr(*this); }

    void print (std::ostream & o) const { o << "ProcessTimer_PerfCtrInstr(" << kinst_per_sim_step << ")"; }
    
  protected:
    bool readArgs(std::istream& in);
    Timepoint* getTimepointForPid(MajorIndex maj, MinorIndex min, int pid, bool is_removed);
    SimTime diffTimepointsForPid(const Timepoint* pt1, const Timepoint* pt2, int pid);
    Timepoint* getInitialTimepointForPid(int pid);
    // This returns the amount of time that should be used if the process
    // associated with this timepoint vanishes
    SimTime finalTimeFrom(const Timepoint* pt, int pid);
    bool prot_notifyAddPid(MessageServer* pmserver, int pid,
			   int pid_parent, const Message::MsgUniqueID& id);
    bool prot_notifyRemovePid(int pid);
    
  private:
    int kinst_per_sim_step;

    // pid to perfctr fd
    typedef std::map<int, ExtPerfCtrFD*> PidPerfCtrMap;
    PidPerfCtrMap pid_perfctr_map;

    // pid to the last timepoint for a process
    typedef std::map<int, Timepoint_PerfCtrInstr*> FinalTimepointMap;
    FinalTimepointMap final_timepoint_map;
    
#else

  public:
    ProcessTimer_PerfCtrInstr() : ProcessTimer_MultiTrack(PTT_PerfCtrInstr) {}
    ProcessTimer_PerfCtrInstr(const ProcessTimer_PerfCtrInstr& pt)
      : ProcessTimer_MultiTrack(PTT_PerfCtrInstr) {}
    ~ProcessTimer_PerfCtrInstr() {}

    bool acceptableForIntegratedAgent() { return false; }

    ProcessTimer_PerfCtrInstr* clone() const { return new ProcessTimer_PerfCtrInstr(*this); }

    void print (std::ostream & o) const { o << "ProcessTimer_PerfCtrInstr(DUMMY)"; }
    
  protected:
    bool readArgs(std::istream& in) { return true; }
    Timepoint* getTimepointForPid(MajorIndex maj, MinorIndex min, int pid, bool is_removed);
    SimTime diffTimepointsForPid(const Timepoint* pt1, const Timepoint* pt2, int pid);
    Timepoint* getInitialTimepointForPid(int pid);
    // This returns the amount of time that should be used if the process
    // associated with this timepoint vanishes
    SimTime finalTimeFrom(const Timepoint* pt, int pid);
    bool prot_notifyAddPid(MessageServer* pmserver, int pid,
			   int pid_parent, const Message::MsgUniqueID& id);
    bool prot_notifyRemovePid(int pid) { return false; }

  private:
    static const char* ERRMSG;
#endif	
  };

} //namespace spades

#endif
