/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* This is a class to handle the killing of processes in a uniform and nice way,
   giving the processes time to shut down */

#ifndef PROCESSKILLLIST_H_
#define PROCESSKILLLIST_H_

#include <set>
#include <iostream>
#include "utility.hpp"
#include "MessageServer.hpp"
#include "TimeVal.hpp"

namespace spades
{

  class ProcessTimer; //in ProcessTimer.hpp

  class ProcessKillList
  {
  public:
    ProcessKillList();
    ~ProcessKillList();

    void addProcess(int pid, double max_shutdown_time);
    void addProcess(int pid, double max_shutdown_time, const TimeVal& tv_curr);
    void addProcesses(const ProcessTimer* pt, double max_shutdown_time, const TimeVal& tv_curr);

    /* with check_if_alive false, check just looks for processes that have hit their
       max shutdown time
       with check_if_alive true, check will look to make sure that all processes still
       in the list actually still exist
       The first variant is faster, but will keep processes in the list longer */
    void check(const TimeVal& tv_curr, bool check_if_alive);
    void check(bool check_if_alive);

    bool empty() { return process_set.empty(); }
    bool size() { return process_set.size(); }
  
  private:

    class PKL_Item
    {
    public:
      PKL_Item() : pid(0),tv_limit(), exists(false) { }
      PKL_Item(int p, const TimeVal& tv) : pid(p),tv_limit(tv), exists(true) { }

      //returns false on unrecoverable errors, returns true if process is gone!
      bool killSoft();
      //returns false on unrecoverable errors, returns true if process is gone!
      bool killHard();

      //returns whether the process is alive
      bool checkAlive();
    
      bool getExists() const { return exists; }
      const TimeVal& getTvLimit() const { return tv_limit; }
      int getPid() const { return pid; }
    
      friend std::ostream & operator << (std::ostream & o, const PKL_Item & i)
      { o << "PKL_Item(" << i.pid << ", " << i.tv_limit << ", " << i.exists << ")"; return o; }

    protected:
      int pid;
      TimeVal tv_limit; //when to forcibly kill the process
      bool exists; //whether the process still exists
    };

    struct lt_PKL_Item : public std::binary_function <PKL_Item*, PKL_Item*, bool>
    {
      bool operator () (PKL_Item* i1, PKL_Item* i2)
      { if (i1->getTvLimit() != i2->getTvLimit()) return i1->getTvLimit() < i2->getTvLimit();
      return (i1->getPid() < i2->getPid()); }
    };

    //I have to use a set of pointers because iterators into the set can not modify the element
    typedef std::set<PKL_Item*, lt_PKL_Item> ProcessSet;
    ProcessSet process_set;
  
  };

  // a ForkWatcher that simple adds the chldren to the process kill list
  class PKLAddChildForkWatcher
    : public ForkWatcher
  {
  public:
    PKLAddChildForkWatcher(ProcessKillList* plist,
			   double max_shutdown_time)
      : ForkWatcher(),
	plist(plist),
	max_shutdown_time(max_shutdown_time)
    {}
    ~PKLAddChildForkWatcher() {};

    void notifyFork(MessageServer* pserver, int pid_parent, int pid_child, const Message::MsgUniqueID& id)
    { plist->addProcess(pid_child, max_shutdown_time); }

  private:
    ProcessKillList* plist;
    double max_shutdown_time;
  };
  
  
  
} //spades namespace


#endif
