/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

#ifndef PARAMREADER_H_
#define PARAMREADER_H_

#include <iostream>
#include <fstream>
#include <map>
#include <string>
#include <vector>
#include <algorithm>
#include <iterator>
#include "FileReader.hpp"
#include "XMLWriter.hpp"

namespace spades
{
  /****************************************************************************************/
  /* Helper Functions */
  template<class T> void readVector(std::istream& is, std::vector<T>& v)
  {
    v.clear();
    while (is.good())
      {
	T val;
	is >> val;
	if (is.fail())
	  break;
	v.push_back(val);
      }
  }

  template<class T> std::ostream & operator << (std::ostream & o, const std::vector<T>& v)
  { std::copy(v.begin(), v.end(), std::ostream_iterator<T>(o, " ")); return o; }

  template<class T> bool readVectorWBounds(std::istream& is, std::vector<T>& v, int min, int max)
  {
    readVector(is, v);
    return (static_cast<signed>(v.size()) >= min &&
	    (max < 0 || static_cast<signed>(v.size()) <= max));
  }
  /****************************************************************************************/

  class ParamReader
    : public FileReader
  {
  public:
    // A class that knows how to process parameters and store them
    // We'll subclass this below
    class ParamStorer
    {
    public:
      //ReadReturn values. These are ints so that sucess can be anything >= 0
      static const int RR_None = -3;
      static const int RR_NoMatch = -2;
      static const int RR_FormatErr = -1;
      // all values >= 0 indicate success
      
    public:
      ParamStorer() {}
      virtual ~ParamStorer() {}

      // return 0 on success
      // otherwise, see the RR_ values above
      virtual int readArgsFrom(const std::string& key, const char* fileid, const char* path,
			       std::istream& is, bool have_argument) = 0;
      // returns the number of args processed
      // otherwise see the RR_ values above
      virtual int readCmdLineArgs(const std::string& key,
				  int argc, const char* const* argv) = 0;

      // returns whether this storer set the data for this
      // you should only override the ones that make sense for you
      // return 0 on success,
      //        RR_FormatErr on type mismatch
      //        RR_NoMatch if key DNE
      virtual int setParam (const std::string & key, const int& value)
      { return RR_FormatErr; }
      virtual int setParam (const std::string & key, const bool& value)
      { return RR_FormatErr; }
      // Any time you override this method, you shoudl override the pair one below
      // as ParamReader may put the current directory as the second element for you
      virtual int setParam (const std::string & key, const std::string& value)
      { return RR_FormatErr; }
      // This variant is intended for a file path entry. The string is the value
      // and the const char* is a starting file/directory (if it is a directory, it
      // must end in / otherwise the final component will be removed)
      virtual int setParam (const std::string & key,
			    const std::pair<const std::string, const char*>& value)
      { return RR_FormatErr; }
      virtual int setParam (const std::string & key, const double& value)
      { return RR_FormatErr; }
      virtual int setParam (const std::string & key, const std::vector<int>& value)
      { return RR_FormatErr; }
      virtual int setParam (const std::string & key, const std::vector<double>& value)
      { return RR_FormatErr; }
      virtual int setParam (const std::string & key, const std::vector<std::string>& value)
      { return RR_FormatErr; }

      virtual void print(std::ostream& o) const = 0;

      // the only reason that this isn't pure virtual is to make it easier
      // for those who already wrote ParamStorer
      virtual void writeXML(XMLWriter& writer) const {}
    };

  public:

    class ParamStorerByMap
      : public ParamStorer
    {
    public:
      //an abstract base class
      class ParamStorage
      {
      public:
	ParamStorage() {}
	virtual ~ParamStorage() {}

	// only success return should be 0
	// the key has already been read; it is provided solely to generate error messages
	// otherwise, see the RR_ values above, though RR_NoMatch should not be returned
	virtual int readArgsFrom(const std::string& key,
				 const char* fileid, const char* path,
				 std::istream& is, bool have_argument) = 0;
	// returns the number of args processed
	// the key has already been read; it is provided solely to generate error messages
	// otherwise, see the RR_ values above, though RR_NoMatch should not be returned
	virtual int readCmdLineArgs(const std::string& key,
				    int argc, const char* const* argv) = 0;

	// returns whether this storage accepted the data
	virtual bool setParam (const int& value)
	{ return false; }
	virtual bool setParam (const bool& value)
	{ return false; }
	// Any time you override this method, you shoudl override the pair one below
	// as ParamReader may put the current directory as the second element for you
	virtual bool setParam (const std::string& value)
	{ return false; }
	// This variant is intended for a file path entry. The string is the value
	// and the const char* is a starting file/directory (if it is a directory, it
	// must end in / otherwise the final component will be removed)
	virtual bool setParam (const std::pair<const std::string, const char*>& value)
	{ return false; }
	virtual bool setParam (const double& value)
	{ return false; }
	virtual bool setParam (const std::vector<int>& value)
	{ return false; }
	virtual bool setParam (const std::vector<double>& value)
	{ return false; }
	virtual bool setParam (const std::vector<std::string>& value)
	{ return false; }

	virtual void print(std::ostream& o) const = 0;
        virtual void writeXML(XMLWriter& writer, const std::string& key) const = 0;
      };

    public:
      ParamStorerByMap();
      ~ParamStorerByMap();

      int readArgsFrom(const std::string& key, const char* fileid, const char* path,
		       std::istream& is, bool have_argument);
      // returns the number of args processed
      int readCmdLineArgs(const std::string& key,
			  int argc, const char* const* argv);

      // returns whether this storer set the data for this
      // return 0 on success,
      //        RR_FormatErr on type mismatch
      //        RR_NoMatch if key DNE
      int setParam (const std::string & key, const int& value);
      int setParam (const std::string & key, const bool& value);
      int setParam (const std::string & key, const std::string& value);
      int setParam (const std::string & key,
		    const std::pair<const std::string, const char*>& value);
      int setParam (const std::string & key, const double& value);
      int setParam (const std::string & key, const std::vector<int>& value);
      int setParam (const std::string & key, const std::vector<double>& value);
      int setParam (const std::string & key, const std::vector<std::string>& value);

      // returns whether a parameter already had this name
      bool addParam (const std::string& key, ParamStorage* p);
      
      void print(std::ostream& o) const ;

      void writeXML(XMLWriter& writer) const;
      
    private:
      template <class T> int templateSetParam (const std::string& key, const T& value);

      typedef std::map<std::string, ParamStorage*> MapParam;
      MapParam map_param;
    };

    
    /****************************************************************************/
    /****************************************************************************/
    /****************************************************************************/
  public:

    ParamReader (float max_version);
    virtual ~ParamReader ();

    // returns the directory of the executable.
    // Only valid after getOptions call or after setExecutableDir called
    const std::string& getExecutableDir() const;
    // You should only need to do this if you are not going to be calling
    // getOptions
    void setExecutableDir(const std::string& dir);

    // This class takes over the memory
    void addParamStorer(ParamStorer* p) { vec_param_storer.push_back(p); }
    
    void getOptions (const int &argc, const char *const *argv);

    bool setParam (const std::string & key, const int& value);
    bool setParam (const std::string & key, const bool& value);
    bool setParam (const std::string & key, const char* val)
    { return setParam(key, std::string(val)); }
    bool setParam (const std::string & key, const std::string& value,
		   const char* starting_dir = "./");
    bool setParam (const std::string & key, const double& value);
    bool setParam (const std::string & key, const std::vector<int>& value);
    bool setParam (const std::string & key, const std::vector<double>& value);
    bool setParam (const std::string & key, const std::vector<std::string>& value);
    
    void printAll(std::ostream& out);    
    void writeXML(XMLWriter& writer) const;
    
    /* This function will be called after the values have been read.
       It may be called several times!
       The only guaruntee that you get is that this function will be called after the last value
       has been read in */
    virtual void postReadProcessing() = 0;

  protected:

    static const int MaxArgLength;

    /* this is the call back for the FileReader */
    bool processLine(std::istream& line, const char* fileid, const char* path, float version);

    virtual void addAll2Maps () = 0;
    virtual void setDefaultValues () = 0;

    void add2Maps (const std::string & key, int *value);
    void add2Maps (const std::string & key, std::string *value, bool file_path = false);
    void add2Maps (const std::string & key, bool *value);
    void add2Maps (const std::string & key, double *value);
    void add2Maps (const std::string & key, std::vector<int> *value, int min, int max);
    void add2Maps (const std::string & key, std::vector<double> *value, int min, int max);
    void add2Maps (const std::string & key, std::vector<std::string> *value, int min, int max);
    
  private:
    template <class T> bool templateSetParam (const std::string& key, const T& value);

    // we make these private to insure that postReadProcessing is not called unneccsarily
    void readParams (const char *path) { readFile(path, max_version); }
    
    //getOptions sets up all the appropriate stuff
    void readOptions (const int &argc, const char *const *argv);

    typedef std::vector<ParamStorer*> VecParamStorer;
    VecParamStorer vec_param_storer;
    ParamStorerByMap* p_param_storer_by_map;
    
    //we just use this to make sure that the -file on the command line does not give an error
    std::string conf_file; 

    //stores the directory of the executable
    std::string exec_dir;
    bool exec_dir_set;
    
    float max_version;

  public:
    static const float XML_FORMAT_VERSION;
  };

} //spades namespace

#endif
