/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* This class provides a SPADES specific interface to the SysV message
   queue IPC stuff and to a unix domain socket for sending file descriptors
   from the children. It reads messages off the queue and stores the data
   to be processed later */

#ifndef MESSAGE_SERVER_H_
#define MESSAGE_SERVER_H_

#include "spades-config.h"

#ifdef SPADES_HAVE_SYSV_IPC
# include <sys/types.h>
# include <sys/ipc.h>
# include <sys/msg.h>
#endif
#include <map>
#include <string>
#include "UnixSocket.hpp"
#include "message_types.hpp"
#include "TimeVal.hpp"

namespace spades
{
  class ForkWatcher;
  class MessageServer;
  class IgnoreForkWatcher;
  class ActionLogForkWatcher;
  
  /**************************************************************************/
  /* This virtual base class should be implemented to get notifications
     when a process forks */
  class ForkWatcher
  {
  public:
    ForkWatcher() {}
    virtual ~ForkWatcher() {}

    virtual void notifyFork(MessageServer* pserver, int pid_parent, int pid_child,
			    const Message::MsgUniqueID& id) = 0;
  };
  
  
  /**************************************************************************/
  class MessageServer
  {
  public:
    static void testRunServer();
    
  public:
    MessageServer();
    ~MessageServer();

    // we have a separate initialize so that errors can be checked
    bool initialize();

    // returns the number of messages received
    int receiveForkMessages();

    // returns the number of messages received
    int receivePidFDMessages();
    
    // returns the number of messages received
    int receiveAllMessages();

    // returns the previous ForkWatcher for this pid
    // does NOT take over the memory
    // Note that calls to notifyFork can be made before this function returns
    // Passing NULL will clear the value
    ForkWatcher* setWatcherFor(int pid_parent, ForkWatcher* p);

    // Takes all pending notifications and sends them to this ForkWatcher
    // returns the number of notifications sent
    int notifyAllPendingTo(ForkWatcher* p);

    // returns -1 if no pending fd for this pid
    // removes this from the pending set
    int getReceivedFDForPid(int pid)
    { return getReceivedFDForPidWorker(pid, Message::MsgUniqueID(), false); }
    // this variant only returns a FD if the associated unique ID matches
    int getReceivedFDForPid(int pid, const Message::MsgUniqueID& id)
    { return getReceivedFDForPidWorker(pid, id, true); }

    // if the given pid has not sent us a fd yet, we'll wait (at most tv)
    // for a fd to be received.
    // The FD has to match the given unique id
    // We'll also check if secondary_pid has a FD waiting with the correct unique ID
    // See the NOTES file for a disucssion of why we have to do this
    // Removes the fd from the map just like getReceivedFDForPid
    // Returns -1 if we timeout
    int waitForFDFromPid(int pid, const Message::MsgUniqueID& id, int secondary_pid,
			 const TimeVal& tv);

    // returns the next pid >= lower_bound which has a received fd
    // returns -1 if no such pid;
    int getNextPendingPid(int lower_bound) const;
    
  private:
    bool createKey();

    void handleFork(int pid_parent, int pid_child, const Message::MsgUniqueID& id);
    
    // returns -1 if no pending fd for this pid
    // removes this from the pending set
    int getReceivedFDForPidWorker(int pid, const Message::MsgUniqueID& id, bool id_valid);

    bool initialized;
    
    key_t mqueue_key;
    int mqueue_id;

    struct PendingForkValue
    {
      PendingForkValue()
	: pid_child(-1), uniqid() {}
      PendingForkValue(pid_t pid_child, const Message::MsgUniqueID& uniqid)
	: pid_child(pid_child), uniqid(uniqid) {}
      pid_t pid_child;
      Message::MsgUniqueID uniqid;
    };
    
    //stuff for storing forks still to be notified
    typedef std::multimap<pid_t, PendingForkValue> PendingForkMap;
    PendingForkMap pending_fork_map;

    //stores the fork watchers
    typedef std::map<pid_t, ForkWatcher*> ForkWatcherMap;
    ForkWatcherMap fork_watcher_map;

    struct PendingFDValue
    {
      PendingFDValue()
	: fd(-1), uniqid() {}
      PendingFDValue(int fd, const Message::MsgUniqueID& uniqid)
	: fd(fd), uniqid(uniqid) {}
      int fd;
      Message::MsgUniqueID uniqid;
    };
    
    typedef std::multimap<pid_t, PendingFDValue> PendingPidFDMap;
    PendingPidFDMap pending_pid_fd_map;
    
    UnixSocket recv_socket;
  };
  
  /**************************************************************************/
  /* Now a couple of simple ForkWatchers */

  class IgnoreForkWatcher
    : public ForkWatcher
  {
  public:
    IgnoreForkWatcher() : ForkWatcher() {}
    ~IgnoreForkWatcher() {}

    void notifyFork(MessageServer* pserver, int pid_parent, int pid_child, const Message::MsgUniqueID& id) {}
  };

  class ActionLogForkWatcher
    : public ForkWatcher
  {
  public:
    ActionLogForkWatcher(const char* leader, int level)
      : ForkWatcher(), leader(leader), level(level) {}
    ~ActionLogForkWatcher() {}

    void notifyFork(MessageServer* pserver, int pid_parent, int pid_child, const Message::MsgUniqueID& id);
  private:
    std::string leader;
    int level;
  };

  class OSForkWatcher
    : public ForkWatcher
  {
  public:
    OSForkWatcher(std::ostream& os, const char* leader)
      : ForkWatcher(), os(os), leader(leader) {}
    ~OSForkWatcher() {}

    void notifyFork(MessageServer* pserver, int pid_parent, int pid_child, const Message::MsgUniqueID& id);
  private:
    std::ostream& os;
    std::string leader;
  };
  
  // routines to help manipulate my unique IDs
  bool operator==(const Message::MsgUniqueID& id1, const Message::MsgUniqueID& id2);
  inline bool operator!=(const Message::MsgUniqueID& id1, const Message::MsgUniqueID& id2)
  { return !(id1 == id2); }
  bool operator<(const Message::MsgUniqueID& id1, const Message::MsgUniqueID& id2);
  
} //spades namespace

/**************************************************************************/
  

#endif
