/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

#ifndef LOGGING_H_
#define LOGGING_H_

#include <iostream>
#include <fstream>
#include <string>
#include <set>
#include <vector>
#include <string.h>

/* Why is this function here? There is some namespace weirdness with the << operator
   If T is inside a namespace (maybe a nested one), then in the spades namespace
   that operator can not be found. We have this function out in the global namespace
   so that it can find all of the appropriate operator<< in all other namespaces */
template <class T> void globalWriteToForLogger (std::ostream& os, const T & a)
{ os << a; }

namespace spades
{

  /**************************************************************************/

  /* The virtual base class for TagFunctions */
  class TagFunction
  {
  public:
    TagFunction () {}
    virtual ~ TagFunction () {}
    //The first getTag is provided for backwards compatibility.
    // If you redefine the other getTag variant
    // then this definition can be nothing and should never be called
    virtual std::string getTag (int level) const = 0;
    virtual std::string getTag (int level, const std::string& channel_name) const
    { return getTag(level); }
    virtual std::string getErrorTag (int level) const = 0;
  };

  /* This is used if there is no other valid TagFunction available */
  class DefaultTagFunction : public TagFunction
  {
  public:
    DefaultTagFunction () {}
    ~DefaultTagFunction () {}
    std::string getTag (int level) const;
    std::string getTag (int level, const std::string& channel_name) const;
    std::string getErrorTag (int level) const;
  };
  
  /**************************************************************************/

  /* all log lines must end with ende */
  class LogManip_ende
  {
  public:
    LogManip_ende () {}
  };

  extern const LogManip_ende ende;


  class Logger
  {
  public:
    class ActionLog;
    class ErrorLog;

  protected:
    class Channel;
    
    friend class ActionLog;
    friend class ErrorLog;
    friend class Channel;
    
  public:
    typedef int ChannelID;
    static const int CHANNEL_ID_INVALID = -1;
    
    //Class definitions
    class ErrorLog
    {
    public:
      ErrorLog (): tag_next (true), next_level(-1), pos_log(NULL) {}
      ~ErrorLog () {}

      void setLog(std::ostream* pos) { pos_log = pos; }
      std::ostream* getLogStream() { return pos_log; }

      /* the levels in the error log should be used to reflect the severity of the error
	 0 is a critical error, and higher numbers are less severe. Exact details are up
	 to the user of this class. */
      void setNextLevel (int level) { next_level = level; }

      friend ErrorLog & operator << (ErrorLog & l, const LogManip_ende & ende);
      template < class T > friend ErrorLog & operator << (ErrorLog & l, const T & a);

      static const int ERROR_LOG_LEVEL;

    private:
      bool tag_next;
      int next_level;
      std::ostream* pos_log;
    };

    class ActionLog
    {
    public:
      ActionLog ()
	: tag_next(true), next_level(-1), next_chan(NULL), max_level(-1), pos_log(NULL) {}
      ~ActionLog () {}

      bool openFile(const std::string& action_log_fn);
      void setLog(std::ostream* pos) { pos_log = pos; }
      std::ostream* getLogStream() { return pos_log; }
      
      int getMaxLevel() const { return max_level; }
      void setMaxLevel(int m) { max_level = m; }
      
      void setNextLevel (int level) { next_level = level; }
      void setNextChannel (ChannelID id);

      friend ActionLog & operator << (ActionLog & l, const LogManip_ende & ende);
      template < class T > friend ActionLog & operator << (ActionLog & l, const T & a);

    private:
      bool tag_next;
      int next_level;
      Channel* next_chan;

      int max_level;

      std::ostream* pos_log;
      std::ofstream logfile; //this is only used if we open a logfile
    };

  public:

    ErrorLog & errorLogFunc (const int level)
    { elog.setNextLevel(level); return elog; }
    ActionLog & actionLogFunc (const int level, ChannelID chan)
    { alog.setNextLevel (level); alog.setNextChannel(chan); return alog; }

    int getMaxActionLogLevel() const { return alog.getMaxLevel(); }
    void setMaxActionLogLevel(int l) { alog.setMaxLevel(l); }

    bool openFiles(const std::string& action_log_fn)
    { return alog.openFile(action_log_fn); }

    // Note that we do NOT take over the memory
    void setLoggingStreams(std::ostream* pos_action_log, std::ostream* pos_error_log);

    std::ostream* getActionLogStream() { return alog.getLogStream(); }
    std::ostream* getErrorLogStream() { return elog.getLogStream(); }
    
    void setTagFunction(TagFunction* f) { delete pTag; pTag = f; }
    TagFunction* getTagFunction() const { return (pTag == NULL) ? &tagDefault : pTag; }

    ChannelID createLoggingChannel(ChannelID parent, const char* name);
    ChannelID getChannelID(const char* name);

    ChannelID getAllChannel() const { return chan_id_all; }
    ChannelID getDefaultChannel() const { return chan_id_default; }
    ChannelID getErrorChannel() const { return chan_id_error; }

    void showChannelHierarchy(std::ostream& os);

    // A channel command is of the form +foo or -foo where foo is the name of a channel
    // returns true if the command was processed successfully
    bool processSingleChannelCommand(const std::string& com_str);
    int processChannelCommands(const std::vector<std::string>& vcom);
    
  protected:
    // each logging message will be associated with a channel that can be turned on or off
    class Channel
    {
    public:
      Channel(Logger* p, ChannelID id, const std::string& name)
	: plogger(p), id(id), name(name), children(), is_on(true) {}
      ~Channel() {}

      ChannelID getID() const { return id; }

      const std::string& getName() const { return name; }
      
      void addChild(const ChannelID& id) { children.insert(id); }

      bool isOn() const { return is_on; }
      void turnOn() { setState(true); }
      void turnOff() { setState(false); }
      void setState(bool is_on);

      // returns number of channels printed
      int showChannelHierarchy(std::ostream& os, const std::string& prefix);

      void print(std::ostream& os) const;
      friend std::ostream& operator<<(std::ostream& os, const Channel& c)
      { c.print(os); return os; }
      
    private:
      typedef std::set<ChannelID> Children;

      Logger* plogger;
      ChannelID id;
      std::string name;
      Children children;
      bool is_on;
    };

    Channel* lookupChannel(ChannelID id);
    
  private:
    typedef std::vector<Channel> ChannelStorage;
    
    TagFunction *pTag;
    static DefaultTagFunction tagDefault;

    ErrorLog elog;
    ActionLog alog;

    ChannelStorage channels;

    ChannelID chan_id_all;
    ChannelID chan_id_default;
    ChannelID chan_id_error;
    
  public:
    static Logger* instance();
    static void removeInstance();
    
  private:
    /* these are private so that the instance method is the only one that can allocate */
    Logger ();
    ~Logger ();

    static Logger* s_instance;
    
  };

  
  /*********** Shortcuts into this logger *************************/

#ifdef NO_ACTION_LOG
#define actionlog(x) if(1) ; else spades::Logger::instance()->actionLogFunc(x, spades::Logger::CHANNEL_ID_INVALID)
#define actlog(x, chan) if(1) ; else spades::Logger::instance()->actionLogFunc(x, chan)
#else
#define actionlog(x) if(x>spades::Logger::instance()->getMaxActionLogLevel()) ; else spades::Logger::instance()->actionLogFunc(x, spades::Logger::instance()->getDefaultChannel())
#define actlog(x, chan) if(x>spades::Logger::instance()->getMaxActionLogLevel()) ; else spades::Logger::instance()->actionLogFunc(x, chan)
#endif

  /* 0 is a critical error. all others should go to warning log */
#define errorlog spades::Logger::instance()->errorLogFunc(0)
#define warninglog(x) spades::Logger::instance()->errorLogFunc(x)

  /*********** Implementations of templated function *************************/

  template <class T> Logger::ActionLog & operator << (Logger::ActionLog & l, const T & a)
  {
    if (l.next_level > l.max_level || !l.next_chan->isOn())
      return l;
    
    if (l.tag_next)
      {
	std::string tag (Logger::instance()->getTagFunction()->getTag (l.next_level, l.next_chan->getName()));
	l.tag_next = false;
	if (l.pos_log)
	  (*l.pos_log) << tag.c_str();
	else
	  std::cout << tag.c_str();
      }

    if (l.pos_log)
      /* see note at top of file */
      globalWriteToForLogger(*l.pos_log, a);
    else
      /* see note at top of file */
      globalWriteToForLogger(std::cout, a);
    return l;
  }

  template <class T> Logger::ErrorLog & operator << (Logger::ErrorLog & l, const T & a)
  {
    if (l.tag_next)
      {
	std::string tag (Logger::instance()->getTagFunction()->getErrorTag (l.next_level));
	l.tag_next = false;
	if (l.pos_log)
	  (*l.pos_log) << tag.c_str();
	else
	  std::cerr << tag.c_str();
	Logger::instance()->actionLogFunc (l.ERROR_LOG_LEVEL, Logger::instance()->getErrorChannel()) << tag.c_str();
      }

    /* see note at top of file */
    if (l.pos_log)
      globalWriteToForLogger(*l.pos_log, a);
    else
      globalWriteToForLogger(std::cerr, a);
    Logger::instance()->actionLogFunc (l.ERROR_LOG_LEVEL, Logger::instance()->getErrorChannel()) << a;

    return l;
  }

} //spades namespace

#endif
