/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/


#ifndef FROM_CS_MESSAGE_H_
#define FROM_CS_MESSAGE_H_

#include "sharedtypes.hpp"
#include "DataArray.hpp"

namespace spadesalib
{
  
  /******************************************************************/

  class FromCSMessage;
  class FromCSMessage_InitData;
  class FromCSMessage_Exit;
  class FromCSMessage_Sense;
  class FromCSMessage_TimeNotify;
  class FromCSMessage_MigrationRequest;
  class FromCSMessage_Error;
  class FromCSMessage_ThinkTime;

  /******************************************************************/

  class FromCSMessage
  {
  public:
    FromCSMessage() {}
    virtual ~FromCSMessage() {}

    virtual void print (std::ostream & o) const = 0;
    friend std::ostream & operator << (std::ostream & o, const FromCSMessage & m)
    { m.print (o); return o; }

    // How this message affects the count of outstanding think messages
    virtual int getOutstandingThinkMod() const = 0;
  
    class Visitor 
    {
    public:
      Visitor() {}
      virtual ~Visitor() {}

      virtual void visit(FromCSMessage_InitData* m) = 0;
      virtual void visit(FromCSMessage_Exit* m) = 0;
      virtual void visit(FromCSMessage_Sense* m) = 0;
      virtual void visit(FromCSMessage_TimeNotify* m) = 0;
      virtual void visit(FromCSMessage_MigrationRequest* m) = 0;
      virtual void visit(FromCSMessage_Error* m) = 0;
      virtual void visit(FromCSMessage_ThinkTime* m) = 0;

    };
  
    class ConstVisitor 
    {
    public:
      ConstVisitor() {}
      virtual ~ConstVisitor() {}

      virtual void visit(const FromCSMessage* m) = 0;
      virtual void visit(const FromCSMessage_InitData* m) = 0;
      virtual void visit(const FromCSMessage_Exit* m) = 0;
      virtual void visit(const FromCSMessage_Sense* m) = 0;
      virtual void visit(const FromCSMessage_TimeNotify* m) = 0;
      virtual void visit(const FromCSMessage_MigrationRequest* m) = 0;
      virtual void visit(const FromCSMessage_ThinkTime* m) = 0;
    };
  
    virtual void accept(Visitor* v) = 0;
    virtual void accept(ConstVisitor* v) const = 0;

  private:
  
  };

  /******************************************************************/

  class FromCSMessage_InitData
    : public FromCSMessage
  {
  public:
    FromCSMessage_InitData() : FromCSMessage(), data() {}
    FromCSMessage_InitData(const spades::DataArray& d) : FromCSMessage(), data(d) {}
    ~FromCSMessage_InitData() {}

    void print (std::ostream & o) const
    { o << "FromCSMessage_InitData(data_size=" << data.getSize() << ")"; }
  
    int getOutstandingThinkMod() const { return 0; }

    void accept(Visitor* v) { v->visit(this); }
    void accept(ConstVisitor* v) const { v->visit(this); }

    const spades::DataArray& getData() const { return data; }
  
  private:
    spades::DataArray data;
  
  };

  /******************************************************************/

  class FromCSMessage_Exit
    : public FromCSMessage
  {
  public:
    FromCSMessage_Exit() : FromCSMessage() {}
    ~FromCSMessage_Exit() {}

    void print (std::ostream & o) const
    { o << "FromCSMessage_Exit()"; }
  
    int getOutstandingThinkMod() const { return 0; }

    void accept(Visitor* v) { v->visit(this); }
    void accept(ConstVisitor* v) const { v->visit(this); }

  private:
  
  };

  /******************************************************************/

  class FromCSMessage_Sense
    : public FromCSMessage
  {
  public:
    FromCSMessage_Sense() : FromCSMessage(), data(), send_time(-1), arrive_time(-1), thinking(spades::TT_Invalid) {}
    FromCSMessage_Sense(const spades::DataArray& d, int send_time, int arrive_time, spades::ThinkingType thinking)
      : FromCSMessage(), data(d), send_time(send_time), arrive_time(arrive_time), thinking(thinking) {}
    ~FromCSMessage_Sense() {}

    void print (std::ostream & o) const
    { o << "FromCSMessage_Sense("
	<< "data_size=" << data.getSize() << ","
	<< "st=" << send_time << ","
	<< "at=" << arrive_time << ","
	<< "th=" << thinking << ")"; }
  
    int getOutstandingThinkMod() const;

    void accept(Visitor* v) { v->visit(this); }
    void accept(ConstVisitor* v) const { v->visit(this); }

    const spades::DataArray& getData() const { return data; }
    int getSendTime() const { return send_time; }
    int getArriveTime() const { return arrive_time; }
    spades::ThinkingType getThinking() const { return thinking; }
  
  private:
    spades::DataArray data;
    int send_time;
    int arrive_time;
    spades::ThinkingType thinking;
  };

  /******************************************************************/

  class FromCSMessage_TimeNotify
    : public FromCSMessage
  {
  public:
    FromCSMessage_TimeNotify() : FromCSMessage(), arrive_time(-1), thinking(spades::TT_Invalid) {}
    FromCSMessage_TimeNotify(int arrive_time, spades::ThinkingType thinking)
      : FromCSMessage(), arrive_time(arrive_time), thinking(thinking) {}
    ~FromCSMessage_TimeNotify() {}

    void print (std::ostream & o) const
    { o << "FromCSMessage_TimeNotify("
	<< "at=" << arrive_time << ","
	<< "th=" << thinking << ")"; }
  
    int getOutstandingThinkMod() const;
  
    void accept(Visitor* v) { v->visit(this); }
    void accept(ConstVisitor* v) const { v->visit(this); }

    int getArriveTime() const { return arrive_time; }
    spades::ThinkingType getThinking() const { return thinking; }
  
  private:
    int arrive_time;
    spades::ThinkingType thinking;
  };

  /******************************************************************/

  class FromCSMessage_MigrationRequest
    : public FromCSMessage
  {
  public:
    FromCSMessage_MigrationRequest() : FromCSMessage() {}
    ~FromCSMessage_MigrationRequest() {}

    void print (std::ostream & o) const
    { o << "FromCSMessage_MigrationRequest()"; }
  
    int getOutstandingThinkMod() const { return 0; }

    void accept(Visitor* v) { v->visit(this); }
    void accept(ConstVisitor* v) const { v->visit(this); }

  private:
  
  };

  /******************************************************************/

  class FromCSMessage_Error
    : public FromCSMessage
  {
  public:
    FromCSMessage_Error(std::string str) : FromCSMessage(), err_str(str) {}
    ~FromCSMessage_Error() {}

    void print (std::ostream & o) const
    { o << "FromCSMessage_Error()"; }
  
    int getOutstandingThinkMod() const { return 0; }

    void accept(Visitor* v) { v->visit(this); }
    void accept(ConstVisitor* v) const { v->visit(this); }

    std::string getString() const { return err_str; }

  private:

    std::string err_str;
  };

  /******************************************************************/

  class FromCSMessage_ThinkTime
    : public FromCSMessage
  {
  public:
    FromCSMessage_ThinkTime()
      : FromCSMessage(), think_time(-1), intermediate(false) {}
    FromCSMessage_ThinkTime(int think_time, bool intermediate)
      : FromCSMessage(), think_time(think_time), intermediate(intermediate) {}
    ~FromCSMessage_ThinkTime() {}

    void print (std::ostream & o) const
    { o << "FromCSMessage_ThinkTime("
	<< "t=" << think_time << ")"; }
  
    int getOutstandingThinkMod() const { return 0; }
  
    void accept(Visitor* v) { v->visit(this); }
    void accept(ConstVisitor* v) const { v->visit(this); }

    int getTime() const { return think_time; }
    bool getIntermediate() const { return intermediate; }
    
  private:
    int think_time;
    bool intermediate;
  };


  /******************************************************************/
  /******************************************************************/
  /******************************************************************/
  
} //spadesalib namespace


#endif
