/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* these classes provide a queue of messages received from an agent */

#ifndef FROM_AGENT_MESSAGE_H_
#define FROM_AGENT_MESSAGE_H_

#include <string.h>
#include "CSEMessage.hpp"

namespace spades
{
  /* Here's the deal with the orderings
     First, we want all RequestTimeNotifies to come first, and in particular, we want them to
     be earliest first (since the receipt of a time notify could immediately start
     off a thinking cycle. We then want the actions (which we want to come out in the same order
     as they were put in -- this is the main reason for the secondary order value
     Lastly should be the done thinking
  */
  /* One other note: there is no class for a Request Current Think Time message
     because this message is processed immediately and never ends up in the
     message queue */
  
  /* the virtual base class */
  class FromAgentMessage
  {
  public:
    FromAgentMessage() {}
    virtual ~FromAgentMessage() {}

    /* the lower numbers are should come later */
    virtual int getOrderNum() const = 0;
    virtual int getSecondaryOrderNum() const = 0;
    virtual CSEMessage* convertToCSEMessage(SimTime endtime, AgentID agent) = 0;

    virtual void print (std::ostream & o) const = 0;
    friend std::ostream & operator << (std::ostream & o, const FromAgentMessage& m)
    { m.print (o); return o; }

  protected:
    enum OrderConstant
      {
	OC_RequestTimeNotify,
	OC_Act,
	OC_DoneThinking,
	OC_InitDone
      };
  };

  /* FromAgentMessage_RequestTimeNotify */
  class FromAgentMessage_RequestTimeNotify
    : public FromAgentMessage
  {
  public:
    FromAgentMessage_RequestTimeNotify(SimTime t = SIMTIME_INVALID)
      : FromAgentMessage(), time(t) {}
    ~FromAgentMessage_RequestTimeNotify() {}

    int getOrderNum() const { return static_cast<int>(OC_RequestTimeNotify); };
    int getSecondaryOrderNum() const { return static_cast<int>(time); }
    CSEMessage* convertToCSEMessage(SimTime endtime, AgentID agent)
    { return new CSEMessage_RequestTimeNotify(time, agent); }

    void print (std::ostream & o) const { o << "FromAgentMessage_RequestTimeNotify(" << time << ")"; }

  private:
    SimTime time;
  };

  /* FromAgentMessage_Act */
  class FromAgentMessage_Act
    : public FromAgentMessage
  {
  public:
    // The secondary order val should go up monotonically as acts are put into the queue
    FromAgentMessage_Act(DataArray d, int secondary_order_num)
      : FromAgentMessage(), data(d), secondary_order_num(secondary_order_num) { }
    ~FromAgentMessage_Act() { }
  
    int getOrderNum() const { return static_cast<int>(OC_Act); };
    int getSecondaryOrderNum() const { return secondary_order_num; }
    CSEMessage* convertToCSEMessage(SimTime endtime, AgentID agent);

    void print (std::ostream & o) const { o << "FromAgentMessage_Act(" << data.getSize() << ")"; }

  private:
    DataArray data;
    int secondary_order_num;
  };

  /* FromAgentMessage_DoneThinking */
  class FromAgentMessage_DoneThinking
    : public FromAgentMessage
  {
  public:
    FromAgentMessage_DoneThinking() : FromAgentMessage() {}
    ~FromAgentMessage_DoneThinking() {}
  
    int getOrderNum() const { return static_cast<int>(OC_DoneThinking); };
    int getSecondaryOrderNum() const;
    CSEMessage* convertToCSEMessage(SimTime endtime, AgentID agent)
    { return new CSEMessage_DoneThinking(endtime, agent); }

    void print (std::ostream & o) const { o << "FromAgentMessage_DoneThinking"; }

  private:
  };

  /* FromAgentMessage_InitDone */
  class FromAgentMessage_InitDone
    : public FromAgentMessage
  {
  public:
    FromAgentMessage_InitDone() : FromAgentMessage() {}
    ~FromAgentMessage_InitDone() {}
  
    int getOrderNum() const { return static_cast<int>(OC_InitDone); };
    int getSecondaryOrderNum() const;
    CSEMessage* convertToCSEMessage(SimTime endtime, AgentID agent)
    { return new CSEMessage_AgentInitialized(agent); }

    void print (std::ostream & o) const { o << "FromAgentMessage_InitDone"; }

  private:
  };

  /**********************************************************************************/

  struct lt_pFromAgentMessage
  {
    bool operator()(const FromAgentMessage* p1, const FromAgentMessage* p2) 
    {
      if (p1->getOrderNum() != p2->getOrderNum())
	return p1->getOrderNum() < p2->getOrderNum();
      return p1->getSecondaryOrderNum() < p2->getSecondaryOrderNum();
    }
  };

} //spades namespace

#endif
