/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* This file defines message that go between the engine and communication server */

#ifndef ECSMESSAGE_H_
#define ECSMESSAGE_H_

#include <iostream>
#include <netinet/in.h>
#include "sharedtypes.hpp"
#include "DataArray.hpp"
#include "MessageParser.hpp"
#include "Writable.hpp"

namespace spades
{

  enum ECSMessageType
    {
      ECSM_InitAck,
      ECSM_StartAgent,
      ECSM_KillAgent,
      ECSM_Exit,
      ECSM_Sense,
      ECSM_TimeNotify,
      ECSM_MigrationRequest,
    };

  /******************************************************************************************/
  /******************************************************************************************/

  class ECSMessage;

  inline ECSMessageType ecsm_ntoh (ECSMessageType t) { return static_cast<ECSMessageType>(ntohl (t)); }
  inline ECSMessageType ecsm_hton (ECSMessageType t) { return static_cast<ECSMessageType>(htonl (t)); }

  /******************************************************************************************/
  /******************************************************************************************/
  /******************************************************************************************/

  /* forward decls for ECSMessageVisitor */
  class ECSMessage_InitAck;
  class ECSMessage_StartAgent;
  class ECSMessage_KillAgent;
  class ECSMessage_Exit;
  class ECSMessage_Sense;
  class ECSMessage_TimeNotify;
  class ECSMessage_MigrationRequest;

  /* a pure abstract class to implement a Visitor pattern for the messages */
  class ECSMessageVisitor 
  {
  public:
    ECSMessageVisitor() {}
    virtual ~ECSMessageVisitor() {}

    virtual void visitInitAck(ECSMessage_InitAck* p) = 0;
    virtual void visitStartAgent(ECSMessage_StartAgent* p) = 0;
    virtual void visitKillAgent(ECSMessage_KillAgent* p) = 0;
    virtual void visitExit(ECSMessage_Exit* p) = 0;
    virtual void visitSense(ECSMessage_Sense* p) = 0;
    virtual void visitTimeNotify(ECSMessage_TimeNotify* p) = 0;
    virtual void visitMigrationRequest(ECSMessage_MigrationRequest* p) = 0;
  };

  /******************************************************************************************/
  /******************************************************************************************/
  /******************************************************************************************/

  
  class ECSMessage
  {
  public:
    ECSMessage (ECSMessageType t):mtype (t) {}
    virtual ~ ECSMessage () {}

    ECSMessageType getType () const { return mtype; }

    int writeTo (Writable* out) const;

    /* These functions need to convert to network byte order */
    //returns the number of bytes written
    virtual int writeData (Writable* out) const = 0;
    //returns the number of bytes read
    virtual int readData (char *buffer, unsigned max_len) = 0;
    //if this returns -1, then this message has a variable data length
    virtual int getDataLength() const = 0;
    virtual bool isEnoughData (char *buffer, unsigned max_len) const;

    virtual void print (std::ostream & o) const = 0;
    friend std::ostream & operator << (std::ostream & o, const ECSMessage & m)
    { m.print (o); return o; }

    virtual void accept(ECSMessageVisitor* p) = 0;
    
    static ECSMessage *getNew (ECSMessageType t);
    static ECSMessage *newFromBuffer (char *buffer, unsigned dimension,
				      int *pbytes_consumed, MessageParser::MessageError * perr);

  protected:
    ECSMessageType mtype;

  };

  /******************************************************************************************/
  /******************************************************************************************/


  class ECSMessage_InitAck
    : public ECSMessage
  {
  public:
    ECSMessage_InitAck (int s = SERVERID_INVALID):ECSMessage (ECSM_InitAck), serverid (s) {}
    ~ECSMessage_InitAck () {}

    int writeData (Writable* out) const;
    int readData (char *buffer, unsigned max_len);
    int getDataLength() const;

    void print (std::ostream & o) const
    { o << "ECSM:InitAck(s:" << serverid << ")"; }

    void accept(ECSMessageVisitor* p) { p->visitInitAck(this); }

    ServerID getServerID () const { return serverid; }
    void setServerID (ServerID s) { serverid = s; }

  private:
    int serverid;
  };


  /******************************************************************************************/

  class ECSMessage_StartAgent
    : public ECSMessage
  {
  public:
    ECSMessage_StartAgent ()
      : ECSMessage (ECSM_StartAgent), sendtime (0), agentid (AGENTID_INVALID),
	agent_type_name(), init_data()
    { }
    ECSMessage_StartAgent (SimTime time, AgentID a, const char* n)
      : ECSMessage (ECSM_StartAgent), sendtime (time), agentid (a), agent_type_name(n),
	init_data()
    { }
    ECSMessage_StartAgent (SimTime time, AgentID a, const char* n,
			   DataArray data)
      :  ECSMessage (ECSM_StartAgent), sendtime (time), agentid(a), agent_type_name(n),
	 init_data(data)
    { }
    ~ECSMessage_StartAgent ();

    int writeData (Writable* out) const;
    int readData (char *buffer, unsigned max_len);

    int getDataLength() const { return -1; }
    bool isEnoughData (char *buffer, unsigned max_len) const;

    void print (std::ostream & o) const
    {
      o << "ECSM:StartAgent("
	<< "t:" << sendtime << ", "
	<< "a:" << agentid << ", "
	<< "type:" << agent_type_name << ", "
	<< "dlen:" << init_data.getSize() << ")";
    }

    void accept(ECSMessageVisitor* p) { p->visitStartAgent(this); }

    AgentID getAgentID () const { return agentid; }
    void setAgentID (AgentID a) { agentid = a; }

    const std::string& getAgentTypeName() const { return agent_type_name; }
    void setAgentTypeName(const char* name) { agent_type_name = name; }
  
    const DataArray& getInitData () const { return init_data; }

    SimTime getSendTime () const { return sendtime; }
    void setSendTime (SimTime t) { sendtime = t; }

  private:
    SimTime sendtime;
    AgentID agentid;
    std::string agent_type_name;
    DataArray init_data;
  };

  /******************************************************************************************/

  class ECSMessage_KillAgent
    : public ECSMessage
  {
  public:
    ECSMessage_KillAgent (SimTime t = SIMTIME_INVALID,
			  AgentID a = AGENTID_INVALID)
      : ECSMessage(ECSM_KillAgent), simtime(t), agentid(a) {}
    ~ECSMessage_KillAgent () {}

    int writeData (Writable* out) const;
    int readData (char *buffer, unsigned max_len);
    int getDataLength() const;

    void print (std::ostream & o) const
    { o << "ECSM:KillAgent("
	<< "t:" << simtime << ", "
	<< "a:" << agentid << ")"; }

    void accept(ECSMessageVisitor* p) { p->visitKillAgent(this); }

    AgentID getAgentID () const { return agentid; }
    void setAgentID (AgentID a) { agentid = a; }

    SimTime getTime () const { return simtime; }
    void setTime (SimTime t) { simtime = t; }

  private:
    SimTime simtime;
    AgentID agentid;
  };

  /******************************************************************************************/

  class ECSMessage_Exit
    : public ECSMessage
  {
  public:
    ECSMessage_Exit ():ECSMessage (ECSM_Exit) {}
    ~ECSMessage_Exit () {}

    //returns the number of bytes written
    int writeData (Writable* out) const { return 0; }
    //returns the number of bytes read
    int readData (char *buffer, unsigned max_len) { return 0; }
    int getDataLength() const { return 0; }

    void print (std::ostream & o) const { o << "ECSM:Exit"; }

    void accept(ECSMessageVisitor* p) { p->visitExit(this); }

  private:
  };

  /******************************************************************************************/

  class ECSMessage_Sense
    : public ECSMessage
  {
  public:
    ECSMessage_Sense ():ECSMessage (ECSM_Sense), thinking(TT_Regular), 
			sendtime (SIMTIME_INVALID), arrivetime (SIMTIME_INVALID),
			agentid (AGENTID_INVALID), data()
    { }
    ECSMessage_Sense (ThinkingType th, SimTime st, SimTime at, AgentID a, DataArray d);
    ~ECSMessage_Sense () {}

    int writeData (Writable* out) const;
    int readData (char *buffer, unsigned max_len);
    int getDataLength() const { return -1; }
    bool isEnoughData (char *buffer, unsigned max_len) const;

    void print (std::ostream & o) const
    { o << "ECSM:Sense("
	<< "th: " << thinking << ", "
	<< "st:" << sendtime << ", "
	<< "at:" << arrivetime << ", "
	<< "a:" << agentid << ", "
	<< "dlen:" << data.getSize() << ")"; }

    void accept(ECSMessageVisitor* p) { p->visitSense(this); }

    ThinkingType getThinking() const { return thinking; }
    void setThinking(ThinkingType t) { thinking = t; }
  
    SimTime getSendTime () const { return sendtime; }
    void setSendTime (SimTime t) { sendtime = t; }
    SimTime getArriveTime () const { return arrivetime; }
    void setArriveTime (SimTime t) { arrivetime = t; }

    AgentID getAgentID () const { return agentid; }
    void setAgentID (AgentID a) { agentid = a; }

    const DataArray& getSenseData () const { return data; }

  private:
    ThinkingType thinking;
    SimTime sendtime;
    SimTime arrivetime;
    AgentID agentid;
    DataArray data;
  };

  /******************************************************************************************/

  class ECSMessage_TimeNotify
    : public ECSMessage
  {
  public:
    ECSMessage_TimeNotify (ThinkingType th = TT_Regular,
			   SimTime t = SIMTIME_INVALID,
			   AgentID a = AGENTID_INVALID)
      : ECSMessage(ECSM_TimeNotify), thinking(th), simtime(t), agentid(a)
    {}
    ~ECSMessage_TimeNotify () {}

    int writeData (Writable* out) const;
    int readData (char *buffer, unsigned max_len);
    int getDataLength() const;

    void print (std::ostream & o) const
    { o << "ECSM:TimeNotify("
	<< "th: " << thinking << ", "
	<< "t:" << simtime << ", "
	<< "a:" << agentid << ")"; }

    void accept(ECSMessageVisitor* p) { p->visitTimeNotify(this); }

    ThinkingType getThinking() const { return thinking; }
    void setThinking(ThinkingType t) { thinking = t; }
  
    AgentID getAgentID () const { return agentid; }
    void setAgentID (AgentID a) { agentid = a; }

    SimTime getTime () const { return simtime; }
    void setTime (SimTime t) { simtime = t; }

  private:
    ThinkingType thinking; //if a time notify is old, it can be a non-thinking sensation
    SimTime simtime;
    AgentID agentid;
  };

  /******************************************************************************************/

  class ECSMessage_MigrationRequest
    : public ECSMessage
  {
  public:
    ECSMessage_MigrationRequest ()
      : ECSMessage (ECSM_MigrationRequest) {}
    ECSMessage_MigrationRequest (AgentID a)
      : ECSMessage (ECSM_MigrationRequest), agentid (a) {}

    ~ECSMessage_MigrationRequest () {}

    int writeData (Writable* out) const;
    int readData (char *buffer, unsigned max_len);
    int getDataLength() const;

    void print (std::ostream & o) const
    { o << "ECSM:MigrationRequest(" << "a:" << agentid << ")"; }

    void accept(ECSMessageVisitor* p) { p->visitMigrationRequest(this); }

    AgentID getAgentID () const { return agentid; }
    void setAgentID (AgentID a) { agentid = a; }

  private:
    AgentID agentid;
  };

  /******************************************************************************************/
  /******************************************************************************************/
  /******************************************************************************************/

  
} //spades namespace

#endif
