/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

#ifndef COMMUNICATION_H_
#define COMMUNICATION_H_

#include <deque>
#include <map>
#include <string>
#include "sharedtypes.hpp"
#include "ECSMessage.hpp"
#include "CSEMessage.hpp"
#include "MessageParser.hpp"
#include "FDSet.hpp"
#include "CommServerProxy.hpp"

namespace spades
{

  /******************************************************************************/

  class MessageQueueElement
  {
  public:
    MessageQueueElement (ServerID s = -1, CSEMessage * p = NULL):server (s), pMess (p)
    {}
    ServerID server;
    CSEMessage *pMess;
  };

  /******************************************************************************/

  class CSEMessageParser;
  class CommServerInfo;
  class SimEngine;
  class Communication;

  /******************************************************************************/

  class CSEMessageParser : public MessageParser
  {
  public:
    CSEMessageParser (Communication * pComm);
    ~CSEMessageParser ();

    bool changeServer(ServerID s, TCPSocket* psocket)
    { server = s; changeSocket(psocket); return true; }
    
  protected:
    MessageParser::MessageError processBuffer (char *buffer, int datalen, int *pbytes_consumed);

  private:
    Communication * pComm;
    ServerID server;
  };

  /******************************************************************************/
  class Communication
    : public CommServerProxyCallbacks // an interface from CommServerProxy.hpp
  {
  public:
    Communication (SimEngine* p);
    ~Communication ();

    //returns whether communication has succesfully started
    bool isStarted () { return started; }

    // returns the current number of connected Communication Servers
    int getNumCommServers () const { return mapCommServ.size(); }

    // returns whether a server with the given ID is valid
    bool isServerConnected(ServerID s) const { return mapCommServ.count(s) != 0; }
    
    //initializes the communication
    bool initConnection ();

    // does any further initialization needed after everything else is also initialized
    bool initFinal ();
    
    // close the connection
    void closeConnection ();

    // sends a message to a communication server. returns true on success
    bool sendCommMessage (ServerID comm_server, ECSMessage * m);

    //returns the number of message put into the queue
    //return -1 on error
    int checkReceivedData (TimeVal tv_max_wait);

    /* Message queue functions */
    MessageQueueElement popFrontMessage ();
    MessageQueueElement peekFrontElement () const;
    bool messageQueueEmpty () const { return mqueue.empty (); }
    int messageQueueSize () const { return mqueue.size (); }

    int getNumRecentlyQueuedMessages() const { return num_recently_queued_messages; }
    void resetNumRecentlyQueuedMessages() { num_recently_queued_messages = 0; }
    
    // This implements the CommServerProxyCallbacks interface
    FDSet* getReadSet() { return &fds_read_set; }
    /* this used to be a private method, but now that we have an exit message from the comm
       server, the sim engine needs to be able to call this */
    void disconnectCommServer(ServerID s);
    // this function takes over the responsibility for the memory here
    void enqueueMessage(ServerID s, CSEMessage* m)
    { mqueue.push_back(MessageQueueElement(s,m)); num_recently_queued_messages++; }
    const TimeVal& getCurrWallClockTime() const;
    AgentTypeDB* getAgentTypeDB();
    
    void errorTrace();
    
  protected:
    typedef std::deque<MessageQueueElement> MessageQueue;

    friend class CSEMessageParser;

  private:
    void checkAll();
    
    bool started;
    TCPSocket sock_connect;

    ServerID nextid;

    typedef std::map<ServerID, CommServerProxy*> CommServerMap;
    CommServerMap mapCommServ;

    CSEMessageParser mparser;
    
    SimEngine* pSimEngine;

    FDSet fds_read_set;

    MessageQueue mqueue;

    int num_recently_queued_messages;
    
    void acceptNewCommServerConnection();
  };


} //spades namespace

#endif
