/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

#ifndef COMM_SERVER_PROXY_H_
#define COMM_SERVER_PROXY_H_

#include "ECSMessage.hpp"
#include "CSEMessage.hpp"
#include "TCPSocket.hpp"
#include "FDSet.hpp"
#include "AgentManager.hpp"
#include "Load.hpp"

namespace spades
{

  class CSEMessageParser; //defined in Communication.hpp
  
  /*****************************************************************/
  class CommServerProxyCallbacks
  {
  public:
    CommServerProxyCallbacks() {}
    virtual ~CommServerProxyCallbacks() {}

    virtual FDSet* getReadSet() = 0;

    virtual void disconnectCommServer(ServerID s) = 0;

    // this function takes over the responsibility for the memory here
    virtual void enqueueMessage(ServerID s, CSEMessage* m) = 0;

    virtual const TimeVal& getCurrWallClockTime() const = 0;

    virtual AgentTypeDB* getAgentTypeDB() = 0;
    
  };
  /*****************************************************************/

  class CommServerProxy
  {
  public:
    CommServerProxy(ServerID s, CommServerProxyCallbacks* p)
      : serverID(s), pCallbacks(p) {}
    virtual ~CommServerProxy() {}

    virtual bool disconnect() = 0;
    virtual bool send(ECSMessage* m) = 0;

    virtual void errorTrace() = 0;
    
    //returns the number of fd handled
    // pmessages_received will be incremented with the number of messages processed
    //  from this CommServer
    virtual int processReadSet(const fd_set* read_set, int* pmessages_received) = 0;

    //performs a qucik check for any queued message. Does not do any reading
    // from file descriptors
    // Can call enqueueMessage to process messages
    // returns the number of messages processed
    virtual int quickQueuedMessageCheck() = 0;
    
    // called when select times out; Should check connections
    virtual void check() = 0;
    
  protected:
    ServerID serverID;
    CommServerProxyCallbacks* pCallbacks;
  
  private:
  };

  /*****************************************************************/

  class ExternalCommServerProxy
    : public CommServerProxy
  {
  public:
    ExternalCommServerProxy(ServerID s,
			    CommServerProxyCallbacks* p,
			    TCPSocket *psocket);
    ~ExternalCommServerProxy();

    bool disconnect();
    bool send(ECSMessage* m);

    void errorTrace() {}

    //returns the number of fd handled
    int processReadSet(const fd_set* read_set, int* pmessages_received);

    int quickQueuedMessageCheck() { return 0; }

    // called when select times out; Should check connections
    void check();

  private:
    CSEMessageParser* pmparser;
    TCPSocket* psocket;

    //static functions
  public:
    // does NOT take over the memory here
    static void setMessageParser(CSEMessageParser* pmparser) { s_pmparser = pmparser; }
    static void forgetMessageParser() { s_pmparser = NULL; }

  private:
    static CSEMessageParser* s_pmparser;
    
  };
  
  
  /*****************************************************************/

  class IntegratedCommServerProxy
    : public CommServerProxy,
      public AgentManagerCallbacks //an interface from AgentManager
  {
  public:
    IntegratedCommServerProxy(ServerID s,
			      CommServerProxyCallbacks* p);
    ~IntegratedCommServerProxy();

    // this is a special initialize mehtod that must be called.
    // If false is returned, this object is invalid and you should
    // delete it
    bool initialize();
    
    bool disconnect();
    bool send(ECSMessage* m);

    void errorTrace() { agent_manager.errorTrace(); }

    //returns the number of fd handled
    int processReadSet(const fd_set* read_set, int* pmessages_received);

    //returns number of messages processed
    int quickQueuedMessageCheck();

    // called when select times out; Should check connections
    void check();

    // This implements the AgentManagerCallbacks interface
    // This class must take over the responsibility for the memory
    bool sendToEngine (CSEMessage * m) { pCallbacks->enqueueMessage(serverID, m); return true; }
    void requestShutdown() { disconnect(); }
    const TimeVal& getCurrWallClockTime() { return pCallbacks->getCurrWallClockTime(); }
    FDSet* getReadSet() { return pCallbacks->getReadSet(); }
    ServerID getServerID() const { return serverID; }
    AgentTypeDB* getAgentTypeDB() { return pCallbacks->getAgentTypeDB(); }
    bool amRunningUnderIntegrated() { return true; }
    
  private:
    void sendLoad();
    
    AgentManager agent_manager;
    TimeVal tv_last_load_send;
    MachineLoad mload;
  };
  

} //namespace spades
 
#endif
