/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* This file defines message that go between the engine and communication server */

#ifndef CSEMESSAGE_H_
#define CSEMESSAGE_H_

#include <iostream>
#include <netinet/in.h>
#include "sharedtypes.hpp"
#include "MessageParser.hpp"
#include "DataArray.hpp"
#include "Writable.hpp"
#include "Load.hpp"

namespace spades
{
  class AgentTypeDB; //defined in AgentTypeDB.hpp
  
  enum CSEMessageType
    {
      CSEM_AgentInitialized,
      CSEM_AgentInitError,
      CSEM_Exit,
      CSEM_Act,
      CSEM_RequestTimeNotify,
      CSEM_DoneThinking,
      CSEM_TimeUpdate,
      CSEM_AgentLost,
      CSEM_MigrationData,
      CSEM_MachineLoad,
      CSEM_AgentTypes
    };

  /******************************************************************************************/

  class CSEMessage;


  inline CSEMessageType csem_ntoh (CSEMessageType t) { return static_cast<CSEMessageType>(ntohl (t)); }
  inline CSEMessageType csem_hton (CSEMessageType t) { return static_cast<CSEMessageType>(htonl (t)); }

  /******************************************************************************************/
  /******************************************************************************************/
  /******************************************************************************************/

  class CSEMessage_AgentInitialized;
  class CSEMessage_AgentInitError;
  class CSEMessage_Exit;
  class CSEMessage_Act;
  class CSEMessage_RequestTimeNotify;
  class CSEMessage_DoneThinking;
  class CSEMessage_TimeUpdate;
  class CSEMessage_AgentLost;
  class CSEMessage_MigrationData;
  class CSEMessage_MachineLoad;
  class CSEMessage_AgentTypes;

  class CSEMessageVisitor
  {
  public:
    CSEMessageVisitor() {}
    virtual ~CSEMessageVisitor() {}

    virtual void visitAgentInitialized(CSEMessage_AgentInitialized* p) = 0;
    virtual void visitAgentInitError(CSEMessage_AgentInitError* p) = 0;
    virtual void visitExit(CSEMessage_Exit* p) = 0;
    virtual void visitAct(CSEMessage_Act* p) = 0;
    virtual void visitRequestTimeNotify(CSEMessage_RequestTimeNotify* p) = 0;
    virtual void visitDoneThinking(CSEMessage_DoneThinking* p) = 0;
    virtual void visitTimeUpdate(CSEMessage_TimeUpdate* p) = 0;
    virtual void visitAgentLost(CSEMessage_AgentLost* p) = 0;
    virtual void visitMigrationData(CSEMessage_MigrationData* p) = 0;
    virtual void visitMachineLoad(CSEMessage_MachineLoad* p) = 0;
    virtual void visitAgentTypes(CSEMessage_AgentTypes* p) = 0;
  };
  
  
  /******************************************************************************************/
  /******************************************************************************************/
  /******************************************************************************************/

  class CSEMessage
  {
  public:
    CSEMessage (CSEMessageType t) : mtype (t) {}
    virtual ~ CSEMessage () {}

    CSEMessageType getType () const { return mtype; }

    int writeTo (Writable* out) const;

    /* These functions need to convert to network byte order */
    virtual int writeData (Writable* out) const = 0;
    virtual int readData (char *buffer, unsigned max_len) = 0;
    //if this returns -1, then this message has a variable data length
    virtual int getDataLength() const = 0;
    virtual bool isEnoughData (char *buffer, unsigned max_len) const;

    virtual void print (std::ostream & o) const = 0;
    friend std::ostream & operator << (std::ostream & o, const CSEMessage & m)
    { m.print (o); return o; }

    virtual void accept(CSEMessageVisitor* p) = 0;

    static CSEMessage *getNew (CSEMessageType t);
    static CSEMessage *newFromBuffer (char *buffer, unsigned dimension,
				      int *pbytes_consumed, MessageParser::MessageError * perr);
  protected:
    CSEMessageType mtype;

  };

  /******************************************************************************************/

  //The connection request is the initialize
  class CSEMessage_AgentInitialized
    : public CSEMessage
  {
  public:
    CSEMessage_AgentInitialized (AgentID a = AGENTID_INVALID)
      : CSEMessage (CSEM_AgentInitialized), agentid (a) {}
    ~CSEMessage_AgentInitialized () {}

    int writeData (Writable* out) const;
    int readData (char *buffer, unsigned max_len);
    int getDataLength() const;

    void print (std::ostream & o) const
    { o << "CSEM:AgentInitialized(" << "a:" << agentid << ")"; }

    void accept(CSEMessageVisitor* p) { p->visitAgentInitialized(this); }

    AgentID getAgentID () const { return agentid; }
    void setAgentID (AgentID a) { agentid = a; }

  private:
    AgentID agentid;
  };

  /******************************************************************************************/

  class CSEMessage_AgentInitError
    : public CSEMessage
  {
  public:
    CSEMessage_AgentInitError (AgentID a = AGENTID_INVALID)
      : CSEMessage (CSEM_AgentInitError), agentid (a) {}
    ~CSEMessage_AgentInitError () {}

    int writeData (Writable* out) const;
    int readData (char *buffer, unsigned max_len);
    int getDataLength() const;

    void print (std::ostream & o) const
    { o << "CSEM:AgentInitError(" << "a:" << agentid << ")"; }

    void accept(CSEMessageVisitor* p) { p->visitAgentInitError(this); }

    AgentID getAgentID () const { return agentid; }
    void setAgentID (AgentID a) { agentid = a; }

  private:
    AgentID agentid;
  };

  /******************************************************************************************/
  class CSEMessage_Exit
    : public CSEMessage
  {
  public:
    CSEMessage_Exit () : CSEMessage (CSEM_Exit) {}
    ~CSEMessage_Exit () {}

    int writeData (Writable* out) const { return 0; }
    int readData (char *buffer, unsigned max_len) { return 0; }
    int getDataLength() const { return 0; }

    void print (std::ostream & o) const { o << "CSEM:Exit"; }

    void accept(CSEMessageVisitor* p) { p->visitExit(this); }

  private:
  };

  /******************************************************************************************/

  class CSEMessage_Act
    : public CSEMessage
  {
  public:
    CSEMessage_Act ()
      : CSEMessage (CSEM_Act),
	simtime (SIMTIME_INVALID), agentid (AGENTID_INVALID),
	data ()
    {}
    CSEMessage_Act (SimTime t, AgentID a, DataArray& d)
      : CSEMessage (CSEM_Act), simtime (t), agentid (a), data(d) {}
    ~CSEMessage_Act () {}

    int writeData (Writable* out) const;
    int readData (char *buffer, unsigned max_len);
    int getDataLength() const { return -1; }
    bool isEnoughData (char *buffer, unsigned max_len) const;

    void print (std::ostream & o) const
    { o << "CSEM:Act("
	<< "t:" << simtime << ", "
	<< "a:" << agentid << ", "
	<< "dlen:" << data.getSize() << ")"; }

    void accept(CSEMessageVisitor* p) { p->visitAct(this); }

    AgentID getAgentID () const { return agentid; }
    void setAgentID (AgentID a) { agentid = a; }

    SimTime getTime () const { return simtime; }
    void setTime (SimTime t) { simtime = t; }

    const DataArray& getActData() const { return data; }
    
  private:
    SimTime simtime;
    AgentID agentid;
    DataArray data;
  };

  /******************************************************************************************/

  class CSEMessage_RequestTimeNotify
    : public CSEMessage
  {
  public:
    CSEMessage_RequestTimeNotify ()
      : CSEMessage (CSEM_RequestTimeNotify),
	simtime (SIMTIME_INVALID), agentid (AGENTID_INVALID) { }
    CSEMessage_RequestTimeNotify (SimTime t, AgentID a)
      : CSEMessage (CSEM_RequestTimeNotify), simtime (t), agentid (a) { }
    ~CSEMessage_RequestTimeNotify () {}

    int writeData (Writable* out) const;
    int readData (char *buffer, unsigned max_len);
    int getDataLength() const;

    void print (std::ostream & o) const
    { o << "CSEM:RequestTimeNotify("
	<< "t:" << simtime << ", "
	<< "a:" << agentid << ")"; }

    void accept(CSEMessageVisitor* p) { p->visitRequestTimeNotify(this); }

    AgentID getAgentID () const { return agentid; }
    void setAgentID (AgentID a) { agentid = a; }

    SimTime getTime () const { return simtime; }
    void setTime (SimTime t) { simtime = t; }

  private:
    SimTime simtime;
    AgentID agentid;
  };

  /******************************************************************************************/

  class CSEMessage_DoneThinking
    : public CSEMessage
  {
  public:
    CSEMessage_DoneThinking ()
      : CSEMessage (CSEM_DoneThinking),
	simtime (SIMTIME_INVALID), agentid (AGENTID_INVALID) {}
    CSEMessage_DoneThinking (SimTime t, AgentID a)
      : CSEMessage (CSEM_DoneThinking), simtime (t), agentid (a) {}
    ~CSEMessage_DoneThinking () {}

    int writeData (Writable* out) const;
    int readData (char *buffer, unsigned max_len);
    int getDataLength() const;

    void print (std::ostream & o) const
    { o << "CSEM:DoneThinking("
	<< "t:" << simtime << ", "
	<< "a:" << agentid << ")"; }

    void accept(CSEMessageVisitor* p) { p->visitDoneThinking(this); }

    AgentID getAgentID () const { return agentid; }
    void setAgentID (AgentID a) { agentid = a; }

    SimTime getTime () const { return simtime; }
    void setTime (SimTime t) { simtime = t; }

  private:
    SimTime simtime;
    AgentID agentid;
  };

  /******************************************************************************************/

  class CSEMessage_TimeUpdate
    : public CSEMessage
  {
  public:
    CSEMessage_TimeUpdate ()
      : CSEMessage (CSEM_TimeUpdate),
	simtime (SIMTIME_INVALID), agentid (AGENTID_INVALID)
    {}
    CSEMessage_TimeUpdate (SimTime t, AgentID a)
      : CSEMessage (CSEM_TimeUpdate), simtime (t), agentid (a)
    {}
    ~CSEMessage_TimeUpdate () {}

    int writeData (Writable* out) const;
    int readData (char *buffer, unsigned max_len);
    int getDataLength() const;

    void print (std::ostream & o) const
    { o << "CSEM:TimeUpdate("
	<< "t:" << simtime << ", "
	<< "a:" << agentid << ")";
    }

    void accept(CSEMessageVisitor* p) { p->visitTimeUpdate(this); }

    AgentID getAgentID () const { return agentid; }
    void setAgentID (AgentID a) { agentid = a; }

    SimTime getTime () const { return simtime; }
    void setTime (SimTime t) { simtime = t; }

  private:
    SimTime simtime;
    AgentID agentid;
  };

  /******************************************************************************************/
  class CSEMessage_AgentLost
    : public CSEMessage
  {
  public:
    CSEMessage_AgentLost (AgentID a = AGENTID_INVALID,
			  AgentLostReason reason = ALR_None)
      : CSEMessage (CSEM_AgentLost), agentid (a), reason(reason)  {}
    ~CSEMessage_AgentLost () {}

    int writeData (Writable* out) const;
    int readData (char *buffer, unsigned max_len);
    int getDataLength() const;

    void print (std::ostream & o) const
    { o << "CSEM:AgentLost(" << "a:" << agentid << ")"; }

    void accept(CSEMessageVisitor* p) { p->visitAgentLost(this); }

    AgentID getAgentID () const { return agentid; }
    void setAgentID (AgentID a) { agentid = a; }

    AgentLostReason getReason () const { return reason; }
    void setReason (AgentLostReason c) { reason = c; }

  private:
    AgentID agentid;
    AgentLostReason reason;
  };

  /******************************************************************************************/

  class CSEMessage_MigrationData
    : public CSEMessage
  {
  public:
    CSEMessage_MigrationData ()
      : CSEMessage (CSEM_MigrationData),
	simtime (SIMTIME_INVALID), agentid (AGENTID_INVALID),
	data () {}
    CSEMessage_MigrationData (SimTime t, AgentID a, DataArray d)
      : CSEMessage (CSEM_MigrationData), simtime (t), agentid (a), data(d) {}
    ~CSEMessage_MigrationData () {}

    int writeData (Writable* out) const;
    int readData (char *buffer, unsigned max_len);
    int getDataLength() const { return -1; }
    bool isEnoughData (char *buffer, unsigned max_len) const;

    void print (std::ostream & o) const
    { o << "CSEM:MigrationData("
	<< "t:" << simtime << ", "
	<< "a:" << agentid << ", "
	<< "dlen:" << data.getSize() << ")"; }

    void accept(CSEMessageVisitor* p) { p->visitMigrationData(this); }

    AgentID getAgentID () const { return agentid; }
    void setAgentID (AgentID a) { agentid = a; }

    SimTime getTime () const { return simtime; }
    void setTime (SimTime t) { simtime = t; }

    const DataArray& getMigData() const { return data; }
    
  private:
    SimTime simtime;
    AgentID agentid;
    DataArray data;
  };

  /******************************************************************************************/

  class CSEMessage_MachineLoad
    : public CSEMessage
  {
  public:
    CSEMessage_MachineLoad () : CSEMessage (CSEM_MachineLoad), load_info() {}
    CSEMessage_MachineLoad (const LoadInfo& l) : CSEMessage (CSEM_MachineLoad), load_info(l) {}
    ~CSEMessage_MachineLoad () {}

    int writeData (Writable* out) const;
    int readData (char *buffer, unsigned max_len);
    int getDataLength() const;

    void print (std::ostream & o) const
    { o << "CSEM:MachineLoad(" << load_info << ")"; }

    void accept(CSEMessageVisitor* p) { p->visitMachineLoad(this); }

    const LoadInfo& getLoadInfo() const { return load_info; }
  private:
    LoadInfo load_info;
  };

  /******************************************************************************************/

  class CSEMessage_AgentTypes
    : public CSEMessage
  {
  public:
    CSEMessage_AgentTypes () : CSEMessage (CSEM_AgentTypes), at_names() {}
    CSEMessage_AgentTypes (const AgentTypeDB& atdb)
      : CSEMessage (CSEM_AgentTypes), at_names() { setFromATDB(atdb); }
    ~CSEMessage_AgentTypes () {}

    int writeData (Writable* out) const;
    // a 0 here will mean that we don't have enough data
    int readData (char *buffer, unsigned max_len);
    int getDataLength() const { return -1; } //variable length
    bool isEnoughData (char *buffer, unsigned max_len) const;

    void print (std::ostream & o) const;

    void accept(CSEMessageVisitor* p) { p->visitAgentTypes(this); }

    typedef std::set<std::string> ATNameSet;

    void setFromATDB(const AgentTypeDB& atdb);
    const ATNameSet& getATNameSet() const { return at_names; }
    void clear() { at_names.clear(); }
    
  private:
    ATNameSet at_names;
    
  };

} //spades namespace

#endif
