/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* A class to record and manage all available agent types */

#ifndef AGENT_TYPE_DB_H_
#define AGENT_TYPE_DB_H_

#include <map>
#include "AgentType.hpp"
#include "FileReader.hpp"
#include "utility.hpp"
#include "XMLParser.hpp"

namespace spades
{
  class AgentTypeDBParser;
  
  class AgentTypeDB
  {
  public:
    AgentTypeDB();
    ~AgentTypeDB();

    friend std::ostream& operator <<(std::ostream & os, const AgentTypeDB& atdb);

    void clear();
    
  protected:
    friend class AgentTypeDBParser;
    //takes over the memory
    void addAgentType(AgentType* at, const char* warning_tag = NULL);

    // if non-null, warning_tag will be included with the warnings for redefinitions
    void takeTypesFrom(AgentTypeDB* pdb, const char* warning_tag = NULL);
    
    void resolveReferences();
    /* The algorithm here is not very smart. In particular, it just starts following alias
       links until a non-alias type is found or max_depth steps are taken. If max_steps are taken,
       then it is assumed that this type is in a loop */
    void checkForReferenceLoops(int max_depth, bool delete_bad);

    // there should be no reference loops when this is called
    // Removes all types which point to undefined types
    // returns the number of items pruned
    int pruneNullReferents();
    
  private:
    typedef std::map<std::string, AgentType*, std::less<std::string> > AgentTypeContainer;
    AgentTypeContainer agent_type_cont;

  public:

    typedef AgentTypeContainer::iterator AgentTypeIterator;
    typedef AgentTypeContainer::const_iterator AgentTypeConstIterator;

    AgentTypeIterator getAgentType(std::string id) { return  agent_type_cont.find(id); }
    AgentTypeConstIterator getAgentType(std::string id) const { return  agent_type_cont.find(id); }

    bool isIteratorValid(AgentTypeIterator i) const { return i != agent_type_cont.end(); }
    bool isIteratorValid(AgentTypeConstIterator i) const { return i != agent_type_cont.end(); }

    AgentTypeIterator nullIterator() { return agent_type_cont.end(); }
    AgentTypeConstIterator nullIterator() const { return agent_type_cont.end(); }

    AgentTypeIterator getBeginIterator() { return agent_type_cont.begin(); }
    AgentTypeConstIterator getBeginIterator() const { return agent_type_cont.begin(); }

    /* this is not really the "right" way to do this, but I'm being lazy.
       Really we should make the iterators into new classes with well defined operators
       I'm not sure that it's worth it or that it is the right thing to do
       I wasnt to hide what the actual storage type of the agent type container is */
    AgentType* deref(AgentTypeIterator i) const { return i->second; }
    const AgentType* deref(AgentTypeConstIterator i) const { return i->second; }

  private:
    //not defined, should never be called
    AgentTypeDB(AgentTypeDB&);
    //not defined, should never be called
    AgentTypeDB operator=(const AgentTypeDB& db);
  };

  class AgentTypeDBParser
    : public XMLParser
  {
  public:
    //Does NOT take over the memory
    AgentTypeDBParser(AgentTypeDB* pdb);
    ~AgentTypeDBParser();
    
  protected:
    void notifyStartParse();
    void notifyEndParse();
    void startElementHandler(const XML_Char *name, const XML_Char **atts);
    void endElementHandler(const XML_Char *name);

    void virtualReturnControl(XMLParser* child);
    
  private:
    
    enum ParseMode {
      PM_Invalid,
      PM_BeforeFirst,
      PM_Middle,
      PM_AfterEnd
    };
    ParseMode mode;
    AgentTypeDB* pdb;
    AgentTypeParser* child_parser;
    
  };
  
  
} //spades namespace

#endif
