/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* A class containing information about agent types */

#ifndef AGENT_TYPE_H_
#define AGENT_TYPE_H_

#include <string>
#include <iostream>
#include "utility.hpp"
#include "FDSet.hpp"
#include "sharedtypes.hpp"
#include "StringReplacer.hpp"
#include "XMLParser.hpp"

namespace spades
{
  class ProcessTimer; // defined in "ProcessTimer.hpp"
  class AgentInterface; // defined in AgentInterface.hpp
  class AgentManager; // defined in AgentManager.hpp
  class AgentTypeDB; // defined in AgentTypeDB.hpp
  
  /*****************************************************************************/

  class AgentType
  {
  public:
    AgentType() : id() {}
    AgentType(const std::string& id) : id(id) {}
    virtual ~AgentType() {}

    const std::string& getID() const { return id; }
    void setID(const std::string& id) { this->id = id; }

    // See the note at AgentTypeAlias::print
    virtual void print(std::ostream& os) const = 0;
    friend std::ostream& operator <<(std::ostream & os, const AgentType &at)
    { at.print(os); return os; }

    // starts an agent of this type and gives back an AgentInterface
    // Returns NULL on error
    virtual AgentInterface* startAgent(AgentID agent_id,
				       SimTime start_time,
				       AgentManager* pManager) const = 0;

    // returns false if this is an undefined referent
    virtual bool resolveReferences(AgentTypeDB& db) { return true; }
    virtual bool checkForReferenceLoop(int curr_depth, int max_depth) const
    { return false; }
    // returns false if any referent in the chain is undefined
    virtual bool isReferentLive() const { return true; }
    
  private:
    std::string id;
  };

  class AgentTypeParser
    : public XMLParser
  {
  public:
    AgentTypeParser() {}
    virtual ~AgentTypeParser() {}

    virtual AgentType* getAgentType() = 0;
    virtual void forgetAgentType() = 0;
  };
  /*****************************************************************************/
  
  /* The input format is in XML, but the current format for the command line
     is parsed plain text. This will be fixed in a future release
     (space) spearates arguments
     " can be used to bound an argument, i.e. get a space in an argument name
     The " will not be passed to the executable
     If you want to include a " in a string, you must include that argument in " "
     and use "" to get a single "
     No other characters are special
     Note in particular that this means that \ does not mean anything special, nor
     can any shell style redirection or anything like that be done
     Example:
     foo -a b --code \\
     gives 'foo', '-a', 'b', '--code', '\\'
     bar -a "Some Name"
     gives 'bar', '-a', 'Some Name'
     baz -a "Some""Name"
     gives 'bar', '-a', 'Some"Name'
     bif -a """SomeName"""
     gives 'bif', '-a', '"Some Name"'
  */
  class AgentTypeExternal
    : public AgentType
  {
  public:
    AgentTypeExternal();
    ~AgentTypeExternal();

    int  getInputFD() const { return input_fd; }
    void setInputFD(int fd) { input_fd = fd; }
    int  getOutputFD() const { return output_fd; }
    void setOutputFD(int fd) { output_fd = fd; }

    const ProcessTimer* getProcessTimer() const { return process_timer; }
    //takes over the memory
    void setProcessTimer(ProcessTimer* p)
    { if (process_timer) process_timer; process_timer = p; }
    
    const std::string& getWorkingDir() const { return working_dir; }
    void setWorkingDir(const std::string& wd) { working_dir = trimWhiteSpace(wd); }
    
    const char* getExecutable() const;
    char* const* getArguments() const { return argv; }
    bool setCommandLine(const std::string& cmdline);
    
    /* using the directory of the path specified, resolves the working dir into a fully
       qualified path name.
       Returns false if it can not be resolved */
    bool resolveWorkingDirGivenStartingFile(const char* fn);

    // starts an agent of this type and gives back an AgentInterface
    // Returns NULL on error
    // Any descriptors which should be selected on should go to into p_read_set 
    AgentInterface* startAgent(AgentID agent_id, SimTime start_time, AgentManager* pManager) const;

    void print(std::ostream& os) const;

  private:

    void clearArgv();
    void copyArgv(char** a);

    // sets environment variables for the agent
    void setupEnvForAgent() const;

    int input_fd;
    int output_fd;

    ProcessTimer* process_timer;
    
    std::string working_dir;
    //it's required that the last allocated element is a NULL
    char** argv;

  };

  class AgentTypeExternalParser
    : public AgentTypeParser
  {
  public:
    //Does NOT take over the memory
    AgentTypeExternalParser(AgentTypeExternal* patype);
    ~AgentTypeExternalParser();

    AgentType* getAgentType() { return patype; }
    void forgetAgentType() { patype = NULL; }
    
  protected:
    void startElementHandler(const XML_Char *name, const XML_Char **atts);
    void endElementHandler(const XML_Char *name);

  private:
    AgentTypeExternal* patype;
  };
  
  /*****************************************************************************/

  // This class provides a "name-only" AgentType
  // It can be used by the engine and/or world model to have a list
  // of valid agent types without actually knowing how to start an agent
  class AgentTypePlaceholder
    : public AgentType
  {
  public:
    AgentTypePlaceholder() : AgentType() {}
    ~AgentTypePlaceholder() {}

    //always returns NULL
    AgentInterface* startAgent(AgentID agent_id, SimTime start_time, AgentManager* pManager) const;

    void print(std::ostream& os) const;
  };
  
  class AgentTypePlaceholderParser
    : public AgentTypeParser
  {
  public:
    //Does NOT take over the memory
    AgentTypePlaceholderParser(AgentTypePlaceholder* patype);
    ~AgentTypePlaceholderParser();

    AgentType* getAgentType() { return patype; }
    void forgetAgentType() { patype = NULL; }
    
  protected:
    void startElementHandler(const XML_Char *name, const XML_Char **atts);
    void endElementHandler(const XML_Char *name);

  private:
    AgentTypePlaceholder* patype;
  };
  
  /*****************************************************************************/

  // This class provides a "name-only" AgentType
  // It can be used by the engine and/or world model to have a list
  // of valid agent types without actually knowing how to start an agent
  class AgentTypeAlias
    : public AgentType
  {
  public:
    AgentTypeAlias() : AgentType(), referent_name(), referent(NULL) {}
    ~AgentTypeAlias() {}

    //just calls the referenced type
    AgentInterface* startAgent(AgentID agent_id, SimTime start_time, AgentManager* pManager) const;

    /* Once resolveReferences is called, calling this function can lead to an infinite loop
       if this type is part of a reference loop. Therefore, you need to be careful
       to not print this type (even to an action log) between resolveReferences and
       checkForReferenceLoop */
    void print(std::ostream& os) const;

    const std::string& getReferentName() const { return referent_name; }
    // we have to reset the referent now
    void setReferentName(std::string r) { referent_name = r; referent = NULL; }

    AgentType* getReferent() { return referent; }
    const AgentType* getReferent() const { return referent; }

    bool resolveReferences(AgentTypeDB& db);
    /* The algorithm here is not very smart. In particular, it just starts following alias
       links until a non-alias type is found or max_depth steps are taken. If max_steps are taken,
       then it is assumed that this type is in a loop */
    bool checkForReferenceLoop(int curr_depth, int max_depth) const;
    
    // returns false if any referent in the chain is undefined
    bool isReferentLive() const;

  private:
    std::string referent_name;
    AgentType* referent;
  };
  
  class AgentTypeAliasParser
    : public AgentTypeParser
  {
  public:
    //Does NOT take over the memory
    AgentTypeAliasParser(AgentTypeAlias* patype);
    ~AgentTypeAliasParser();

    AgentType* getAgentType() { return patype; }
    void forgetAgentType() { patype = NULL; }
    
  protected:
    void startElementHandler(const XML_Char *name, const XML_Char **atts);
    void endElementHandler(const XML_Char *name);

  private:
    AgentTypeAlias* patype;
  };
  
  /*****************************************************************************/

  // This class provides an integrated AgentType
  // That is, the agent will live in the same process as the commserver
  // The agent can be specified as a dyanmically loaded library
  class AgentTypeIntegrated
    : public AgentType
  {
  public:
    AgentTypeIntegrated();
    ~AgentTypeIntegrated();
    
    const std::string& getLibPath() const { return lib_path; }
    void setLibPath(const std::string& lp) { lib_path = trimWhiteSpace(lp); }

    const ProcessTimer* getProcessTimer() const { return process_timer; }
    //takes over the memory
    void setProcessTimer(ProcessTimer* p)
    { if (process_timer) process_timer; process_timer = p; }
    
    std::string getInitArg() const { return init_arg; }
    void setInitArg(const std::string& wd) { init_arg = wd; }

    bool getRequireIntegratedCommserver() const { return require_integrated_commserver; }
    void setRequireIntegratedCommserver(bool b) { require_integrated_commserver = b; }

    /* using the directory of the path specified, resolves the lib_path into a fully
       qualified path name.
       Returns false if it can not be resolved */
    bool resolveLibPathGivenStartingFile(const char* fn);

    //always returns NULL
    AgentInterface* startAgent(AgentID agent_id, SimTime start_time, AgentManager* pManager) const;

    void print(std::ostream& os) const;

  private:
    std::string lib_path;
    ProcessTimer* process_timer;
    bool require_integrated_commserver;
    std::string init_arg;
  };
  
  class AgentTypeIntegratedParser
    : public AgentTypeParser
  {
  public:
    //Does NOT take over the memory
    AgentTypeIntegratedParser(AgentTypeIntegrated* patype);
    ~AgentTypeIntegratedParser();

    AgentType* getAgentType() { return patype; }
    void forgetAgentType() { patype = NULL; }
    
  protected:
    void startElementHandler(const XML_Char *name, const XML_Char **atts);
    void endElementHandler(const XML_Char *name);

  private:
    AgentTypeIntegrated* patype;
  };
  
} //spades namespace


#endif
