/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

#ifndef AGENT_MANAGER_H_
#define AGENT_MANAGER_H_

#include <map>
#include "sharedtypes.hpp"
#include "AgentInterface.hpp"
#include "CSEMessage.hpp"
#include "ECSMessage.hpp"
#include "MessageParser.hpp"
#include "TCPSocket.hpp"
#include "ProcessKillList.hpp"
#include "MessageServer.hpp"

namespace spades
{

  class AgentManager; //defined below
  class AgentTypeDB; //defined in AgentTypeDB.hpp
  
  /**************************************************************************************/

  class ECSMessageVisitor_AgentManager
    : public ECSMessageVisitor
  {
  public:
    ECSMessageVisitor_AgentManager(AgentManager* p)
      : pAM(p), message_err(MessageParser::ME_None), messages_processed(0) {}
    ~ECSMessageVisitor_AgentManager() {}

    void visitInitAck(ECSMessage_InitAck* p);
    void visitStartAgent(ECSMessage_StartAgent* p);
    void visitKillAgent(ECSMessage_KillAgent* p);
    void visitExit(ECSMessage_Exit* p);
    void visitSense(ECSMessage_Sense* p);
    void visitTimeNotify(ECSMessage_TimeNotify* p);
    void visitMigrationRequest(ECSMessage_MigrationRequest* p);

    /* every visit will set the message error */
    MessageParser::MessageError getMessageError() const { return message_err; }

    int getMessagesProcessed() const { return messages_processed; }
    
  private:
    AgentManager* pAM;
    MessageParser::MessageError message_err;
    int messages_processed;
  };
  

  /**************************************************************************************/

  // This is an interface that anyone who uses the AgentManager will have to implement
  
  class AgentManagerCallbacks
  {
  public:
    AgentManagerCallbacks() {}
    virtual ~AgentManagerCallbacks() {}

    // This class must take over the responsibility for the memory
    virtual bool sendToEngine (CSEMessage * m) = 0;

    virtual void requestShutdown() = 0;

    virtual const TimeVal& getCurrWallClockTime() = 0;

    virtual FDSet* getReadSet() = 0;

    virtual ServerID getServerID() const = 0;

    virtual AgentTypeDB* getAgentTypeDB() = 0;

    virtual bool amRunningUnderIntegrated() = 0;
  };
  
  /**************************************************************************************/

  class AgentManager
  {
  public:
    AgentManager (AgentManagerCallbacks* p);
    ~AgentManager ();

    bool initialize();
    
    bool startAgent (ECSMessage_StartAgent *m);

    bool removeAgent(AgentID a);
    bool removeAllAgents();
    
    // puts a message in the agent's queue
    bool sendAgentMessage (AgentID agent_id, ToAgentMessage *m);

    // this is used by the MessageVisitor
    void notifyFailedAgentStart();

    // checks for reads from agents from the read_set
    // returns the number of descriptors from which data was read
    // adds the numbers of messags processed to the value *pmessages_processed
    // If you make changes to this method, you'll probably want to change
    // quickQueuedMessageCheck
    int processReadFD(const fd_set* read_set, int* pmessages_processed);

    // Does a quick scan of the agents to see if anyone has messages queued up
    // Does not attempt to read from file descriptors or any other processing,
    // just checks the out queues
    // Returns number of messages processed (generally means sent to the engine)
    // If you make changes to this method, you'll probably want to change
    // processReadFD
    int quickQueuedMessageCheck();
    
    //prints useful debugging information to stderr
    void errorTrace();

    void checkupOnKilledProcesses();
    void waitForAllKilledProcesses();

    SimTime getLatestSimtime() const { return latest_simtime; }

    MessageParser::MessageError processMessage(ECSMessage* m);

    int getNumLostAgents() const { return num_unnatural_lost_agents; }

    //checks that the processes for the agents are still going
    // if only_thinking, then only those agents in thinking mode are checked
    // returns the number of agnets lost
    int checkupOnAgents(bool only_thinking);

    const TimeVal& getCurrWallClockTime()
    { return pCallbacks->getCurrWallClockTime(); }

    MessageServer*   getIPCMessageServer() { return &ipc_message_server; }
    ProcessKillList* getProcessKillList() { return &kill_list; }
    
    bool amRunningUnderIntegrated() { return pCallbacks->amRunningUnderIntegrated(); }
  protected:

    friend class ECSMessageVisitor_AgentManager;
    friend class AgentType;
    friend class AgentTypeExternal;
    
    void updateLatestTime(SimTime t) { if (t>latest_simtime) latest_simtime = t; }

    AgentManagerCallbacks* pCallbacks;
    
  private:
    typedef std::map<AgentID, AgentInterface*> AgentMap;

    bool removeAgent(AgentMap::iterator iter);

    void handleAbnormalLostAgent(AgentID a, AgentLostReason reason);

    bool initialized;
    
    AgentMap mapAgent;

    ProcessKillList kill_list;

    int num_unnatural_lost_agents;

    //This is the latest simulation time of messages to the comm server
    SimTime latest_simtime;

    ECSMessageVisitor_AgentManager message_visitor;

    // Handles IPC stuff from the agent processes
    MessageServer ipc_message_server;
  };


  
} //spades namespace

#endif
