/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

#ifndef AGENTINTERFACE_H_
#define AGENTINTERFACE_H_

#include <list>
#include <set>
#include <fstream>
#include "ECSMessage.hpp"
#include "CSEMessage.hpp"
#include "FromAgentMessage.hpp"
#include "ToAgentMessage.hpp"
#include "ThinkTimeFile.hpp"
#include "AgentType.hpp"
#include "ProcessTimer.hpp"
#include "TimeVal.hpp"

namespace spades
{
  class AgentManager; //defined in AgentManager.hpp
  class ProcessKillList; // defined in ProcessKillList.hpp
  class MessageServer; //defined in MessageServer.hpp
  
  // Class AgentInterface - keeps communication with an agent and queues all the requests for it
  // Note: you have to be careful with the ProcessTimer here. It registers itself
  // as a watcher for pids. You need to tell it to unregister before destroying
  // this object, otherwise the MessageServer could have a stale ForkWatcher pointer
  
  class AgentInterface
  {
  public:
    enum ShutdownStatus
      {
	SS_None,
	SS_Migration, //exit after migration data
	SS_LostProcess,
	SS_BadFD,
	SS_AgentRequest,
	SS_TimerFailure,
	SS_ThinkTooLongWallClock,
	SS_ThinkTooLongSim,
	SS_MessageProcessError
      };

  public:
    AgentInterface(AgentManager* pmanager,
		   AgentID id,
		   SimTime t);
    virtual ~AgentInterface();

    AgentID getID() const { return agent_id; }

    SimTime getSimTime() const { return last_comm_time; }
  
    virtual void addReadToFDSet(FDSet* pfdset) const = 0;
    virtual void removeReadFromFDSet(FDSet* pfdset) const = 0;
    virtual bool isReadInFDSet(const fd_set* pfdset) const = 0;
    
    //Add messages to the queue (or send them to the agent if it is waiting)
    bool enqueueMessage(ToAgentMessage *m);

    //Add message to the beginning of the queue
    bool prequeueMessage(ToAgentMessage* m);
    
    //returns whether there are messages in the queue now
    virtual bool readFromAgent() = 0;

    bool isEngineOutQueueEmpty() const { return out_queue.empty(); }
    CSEMessage *getNextOutMessage();

    AgentStatus getStatus() const { return status; }
    
    ShutdownStatus getShutdownStatus() const { return shutdown_status; }
    void initiateShutdown(ShutdownStatus stat);

    // Gives up the responsibility for watching over processes to the MessageServer and to
    // the ProcessKillList
    virtual void handOffProcessWatching(MessageServer* pmserver,
					ProcessKillList* ppkl,
					const TimeVal& tv_curr) = 0;

    virtual std::string getProcessStatus() = 0;

    // returns false iff shutdown status is not none
    // need to check the out queue after this
    virtual bool checkupOnAgent(bool only_thinking) = 0;
    
  protected:
    //takes over the memory
    // This must be called before anything is done with this class
    void initializeProcessTimer(ProcessTimer* ppt);
      
    friend class ToAgentMessage_TimeNotify;
    friend class ToAgentMessage_Sense;
    friend class ToAgentMessage_MigrationRequest;
    friend class ToAgentMessage_Exit;
    friend class ToAgentMessage_InitData;
    friend class ToAgentMessage_Error;
    friend class ToAgentMessage_ThinkTime;

    virtual bool sendMessageTo(ToAgentMessage_TimeNotify* m) = 0;
    virtual bool sendMessageTo(ToAgentMessage_Sense* m) = 0;
    virtual bool sendMessageTo(ToAgentMessage_MigrationRequest* m) = 0;
    virtual bool sendMessageTo(ToAgentMessage_Exit* m) = 0;
    virtual bool sendMessageTo(ToAgentMessage_InitData* m) = 0;
    virtual bool sendMessageTo(ToAgentMessage_Error* m) = 0;
    virtual bool sendMessageTo(ToAgentMessage_ThinkTime* m) = 0;
    
    void increaseSimTime(SimTime delta)
    { last_comm_time += delta; }
    //if the sim time is already above this, nothing happens
    void updateSimTimeTo(SimTime t)
    { if (last_comm_time < t) last_comm_time = t; }

    void startThink(AgentStatus status);

    //returns whether messages have been added to the output queue
    bool handleAgentMessageAct(const DataArray& d);
    bool handleAgentMessageRequestTimeNotify(SimTime t);
    bool handleAgentMessageDoneThinking();
    bool handleAgentMessageExit();
    bool handleAgentMessageInitDone();
    bool handleAgentMessageMigrationData(const DataArray& d);
    bool handleAgentMessageRequestCurrentThinkTime();
    
    ProcessTimer* getProcessTimer() { return process_timer; }
    const TimeVal& getLastThinkStart() const { return tv_last_think_start; }
    
    // gets the next major timepoints, update tp_last_major and returns the diff
    SimTime getNextMajorTimepoint();
    // gets the next minor timepoint, update tp_last_minor, sends diff with last major
    SimTime getNextMinorTimepoint();

    void enqueueEngineMessage(CSEMessage* m) { out_queue.push_back(m); }
    
    //This is information needed to track the status of the agents
    AgentStatus status;
    
    AgentManager* pmanager;
    
    // a record of how much think time we have notified the engine of
    SimTime curr_sent_think_time;
    
  private:

    int movePendingMessages();
    bool finishThink();

    // looks at the message queue to determine if a message should be sent to the agent
    void checkMessageQueue();

    /* This can have two meanings
       When the agent is waiting, this is the time of the last done_thinking (accounting for the latency
       of the think)
       When the agent is thinking, it is the time of the last message sent to the agent */
    SimTime last_comm_time;
    /* the last wall clock time of the agent start think */
    TimeVal tv_last_think_start;

    AgentID agent_id;               // the global id of a client
    ShutdownStatus shutdown_status;
  
    ProcessTimer* process_timer;
    //this should never be NULL outside the constructor
    ProcessTimer::Timepoint* tp_last_major;
    // getting a new major tp deletes this
    ProcessTimer::Timepoint* tp_last_minor;

    //messages to be sent to the agent
    typedef std::list<ToAgentMessage*> InQueue;
    //messages from the agent to be sent to the engine
    typedef std::list<CSEMessage*> OutQueue;
    //this has to be a multi-set because lt_pFromAgentMessage is only a partial order!
    typedef std::multiset<FromAgentMessage*, lt_pFromAgentMessage> PendingMessageQueue;
  
    InQueue in_queue;
    OutQueue out_queue;
    PendingMessageQueue pending_message_queue;

    ThinkTimeFile think_time_file;

  };

} //spades namespace

#endif
