/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* This class maintains the information specific to an agent that the simulation
   engine needs in order to run. It encapsulates a queue of fixed agent events as
   well */

#ifndef AGENT_INFO_H_
#define AGENT_INFO_H_

namespace spades 
{

  class SimEngine; //defined in SimEngine.hpp
  
  class AgentInfo
  {
  public:
    AgentInfo ();
    AgentInfo (ServerID serv, AgentStatus s, SimTime t,
	       AgentTypeDB::AgentTypeConstIterator at)
      : server(serv), status (s), curr_time (t), agent_type(at) {}
    ~AgentInfo ();

    ServerID getServer() const { return server; }
  
    AgentStatus getStatus () const { return status; }

    SimTime getCurrTime () const { return curr_time; }
    void updateCurrTime (SimTime t);
    
    AgentTypeDB::AgentTypeConstIterator getAgentType() const { return agent_type; }

    void setInitialized(SimTime curr_time);

    bool startThink(SimTime t, ThinkingType think_type);
    bool finishThink(SimTime done_time);

    bool startMigration(ServerID s, SimTime t);
    bool finishMigration();
    
    /* this could result in the event immediately being realized! */
    /* returns whether it was queued succesfully */
    bool enqueueAgentEvent(FixedAgentEvent* e);
    int sizeOfAgentEventQueue() { return pending_agent_events.size(); }
  
    SimTime getMinAgentTime(SimTime action_latency);

    static void setSimEngine(SimEngine* p) { pSimEngine = p; }
  
  private:

    /* checks to see if any events in the event queue are ready to be realized;
       returns number realized */
    int checkForReadyEvents();
  
    typedef std::multiset<FixedAgentEvent *, event_lt> PendingAgentEvents;
    /* these are events which are ready to be realized as far as the SimEngine in concerned.
       The reason we have an additional queue is that we need to insure that events are sent to the
       agents in the correct time order. In order to achieve more parallelism, events can be
       moved from SimEngine::pending_events out of order, since the only important constraint is
       that each agent receives its events in time order */
    PendingAgentEvents pending_agent_events;

    ServerID server;
    AgentStatus status;
    SimTime curr_time;		// the time to which this agent has advanced

    SimTime think_start_time;
    
    AgentTypeDB::AgentTypeConstIterator agent_type;
  
    static SimEngine* pSimEngine;
  };
  
} //namespace spades

#endif
