/* -*- Mode: c++ -*- */

/*
 *Copyright:

 Copyright (C) 2002, 2003 Patrick Riley
 Copyright (C) 2001 Patrick Riley and Emil Talpes

 This file is part of the SPADES simulation system.

 The SPADES simulation system is free software; you can
 redistribute it and/or modify it under the terms of the GNU Lesser
 General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option)
 any later version.

 The SPADES simulation system is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with the SPADES simulation system; if not, write to
 the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* these classes manage migrations of agents.
   It tracks the neccessary information, records migrations in progress, and calls back
   to the SimEngine for information and notification of migrations */

#ifndef MIGRATION_MANAGER_H_
#define MIGRATION_MANAGER_H_

#include <map>
#include <list>
#include <set>
#include <functional>
#include "sharedtypes.hpp"
#include "CSEMessage.hpp"
#include "ECSMessage.hpp"
#include "utility.hpp"

namespace spades
{

  class AgentCSAllocManager; //defined below
  
  /********************************************************************************/

  /* This is the interface to the SimEngine that the AgentCSAllocManager needs */
  class AgentCSAllocCallbacks
  {
  public:
    AgentCSAllocCallbacks() {}
    virtual ~AgentCSAllocCallbacks() {}

    virtual void notifyCommserverReady(ServerID s) = 0;
    
    //returns whether agent a can migrate right now
    virtual bool canAgentMigrate(AgentID a) const = 0;
    virtual void notifyAgentMigrating(AgentID a, ServerID s) = 0;
    virtual void notifyAgentDoneMigrating(AgentID a) = 0;
    // sends a message to a communication server. returns true on success
    virtual bool sendCommMessage (ServerID comm_server, ECSMessage * m) = 0;
    virtual bool sendAgentMessage (AgentID agent, ECSMessage * m) = 0;

    virtual const char* getAgentTypeName(AgentID agent) = 0;

    virtual const TimeVal& getCurrWallClockTime() const = 0;

    virtual void disconnectCommServer(ServerID s) = 0;
  };

  /********************************************************************************/
  class MigrationMessageProcessor
    : public CSEMessageVisitor
  {
  public:
    MigrationMessageProcessor(AgentCSAllocManager* p)
      : CSEMessageVisitor(), pMan(p), used_message(false), server(SERVERID_INVALID) {}
    ~MigrationMessageProcessor() {}
    
    void visitAgentInitialized(CSEMessage_AgentInitialized* p);
    void visitAgentInitError(CSEMessage_AgentInitError* p);
    void visitExit(CSEMessage_Exit* p);
    void visitAct(CSEMessage_Act* p);
    void visitRequestTimeNotify(CSEMessage_RequestTimeNotify* p);
    void visitDoneThinking(CSEMessage_DoneThinking* p);
    void visitTimeUpdate(CSEMessage_TimeUpdate* p);
    void visitAgentLost(CSEMessage_AgentLost* p);
    void visitMigrationData(CSEMessage_MigrationData* p);
    void visitMachineLoad(CSEMessage_MachineLoad* p);
    void visitAgentTypes(CSEMessage_AgentTypes* p);

    bool didUseMessage() const { return used_message; }
    void setServer(ServerID s) { server = s; }
    ServerID getServer() const { return server; }
    
  private:
    AgentCSAllocManager* pMan;
    bool used_message;
    ServerID server;
  };
  
  /********************************************************************************/

  class AgentCSAllocManager
  {
  public:
    AgentCSAllocManager(AgentCSAllocCallbacks* p);
    ~AgentCSAllocManager();

    /* the SimEngine is the primary user of this class */

    //returns whether this did anything with the message
    bool processMessage(ServerID s, CSEMessage* m, const TimeVal& tv_curr);

    /* these functions are used to manage average agent response time */
    void agentStartThink(AgentID a, ThinkingType think, const TimeVal& tv_curr);
    void agentEndThink(AgentID a, ThinkingType think, const TimeVal& tv_curr);

    //This will be called after all messages are processed.
    void update();
    
    int getNumReadyServers() const;
    
    bool isServerReady(ServerID s) const;
    
    ServerID getServerForNewAgent(const char* at_name);

    const std::set<ServerID>& getServersForAgentType(const char* at_name) const;

    /* These functions are called early in the connect/disconnect process so that
       new requests and such are handled correctly */
    void agentConnect(ServerID s, AgentID a);
    void agentDisconnect(AgentID a);

    void commserverConnect(ServerID s);
    void commserverDisconnect(ServerID s);

  protected:
    friend class MigrationMessageProcessor;

    AgentCSAllocCallbacks* pCallbacks;

    void setMachineLoad(ServerID s, const LoadInfo& l);

    // if this agent is not migrating, returns false
    bool finishAgentMigration(AgentID a);

    void handleAgentTypes(ServerID s, const std::set<std::string>& set_types);
    
  private:

    /* private classes for maintaining information */
    class MigCommServerInfo 
    {
    public:
      MigCommServerInfo()
	: load_info(), tv_last_update(), num_agents(0), tv_connect(), ready_for_agents(false) {}
      MigCommServerInfo(const TimeVal& t)
	: load_info(), tv_last_update(), num_agents(0), tv_connect(t), ready_for_agents(false) {}
      MigCommServerInfo(const LoadInfo& l, const TimeVal& t)
	: load_info(l), tv_last_update(t), num_agents(0), tv_connect(t), ready_for_agents(false) {}
      ~MigCommServerInfo() {}

      bool isReady() const { return ready_for_agents; }
      
      LoadInfo load_info;
      TimeVal tv_last_update;
      int num_agents; //number of agents on this comm server
      TimeVal tv_connect;
      bool ready_for_agents;
    };

    class MigAgentInfo
    {
    public:
      MigAgentInfo(ServerID s = SERVERID_INVALID) : server(s) { }
      ~MigAgentInfo() {}

      MeanSummary response_delay; /* in seconds */
      ServerID server;
      std::list<TimeVal> list_start_think_times;
    };

    void recalculateTotalResponseDelay();
    ServerID getLeastLoadedServer(const std::set<ServerID>& set_servers);

    //This is where the AgentCSAllocManager can start migration requests
    //returns the number of migrations started
    int checkForMigrations();

    // This checks whether there are any commservers which connected, but have not moved
    // to ready
    void checkForStalledCommServers();

    TimeVal tv_curr;
    
    typedef std::map<ServerID, MigCommServerInfo, std::less<ServerID> > CommServerMap;
    typedef std::map<AgentID, MigAgentInfo, std::less<AgentID> > AgentMap;
    typedef std::map<std::string, std::set<ServerID> > AgentTypeServersMap;
    
    CommServerMap mapCSInfo;
    AgentMap      mapAgentInfo;
    AgentTypeServersMap mapATServers;

    int num_ready_servers;
    
    /* the set of agents that are currently migrating */
    typedef std::set<AgentID, std::less<AgentID> > MigAgentSet;
    MigAgentSet setMigAgents;
  
    MeanSummary total_response_delay;

  private:
    // never defined, should never be used
    AgentCSAllocManager(const AgentCSAllocManager&);
    // never defined, should never be used
    AgentCSAllocManager& operator=(const AgentCSAllocManager&);
    
  };

} //spades namespace

#endif
