// -*-c++-*-

/***************************************************************************
                                loader.hpp
                             -------------------
                         dynamically open a library
    begin                : 2002-10-10
    copyright            : (C) 2002-2005 by The RoboCup Soccer Simulator 
                           Maintenance Group.
    email                : sserver-admin@lists.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU LGPL as published by the Free Software  *
 *   Foundation; either version 2 of the License, or (at your option) any  *
 *   later version.                                                        *
 *                                                                         *
 ***************************************************************************/


#ifndef RCSS_LIB_LOADER_HPP
#define RCSS_LIB_LOADER_HPP


#if defined(_WIN32) || defined(__WIN32__) || defined(WIN32) || defined(CYGWIN)
  #ifdef RCSSLIB_EXPORTS
    #define RCSSLIB_API __declspec(dllexport)
    #define RCSSLIB_EXTERN
  #else
    #define RCSSLIB_API __declspec(dllimport)
    #define RCSSLIB_EXTERN extern
  #endif
#elif defined(macintosh) || defined(__APPLE__) || defined(__APPLE_CC__)
  #define RCSSLIB_API
  #define RCSSLIB_EXTERN extern
#else
  #define RCSSLIB_API
  #define RCSSLIB_EXTERN
#endif


#include <map>
#include <stack>
#include <list>
#include <boost/weak_ptr.hpp>
#include <iostream>
#include "loaderimpl.hpp"

/*
RCSSLIB_EXTERN template class RCSSLIB_API boost::shared_ptr< rcss::lib::LoaderImpl >;
class RCSSLIB_API boost::detail::weak_count;
RCSSLIB_EXTERN template class RCSSLIB_API boost::weak_ptr< rcss::lib::LoaderImpl >;
RCSSLIB_EXTERN template class RCSSLIB_API std::allocator< std::pair< boost::shared_ptr< rcss::lib::LoaderImpl >, boost::weak_ptr< rcss::lib::LoaderImpl > > >;
RCSSLIB_EXTERN template class RCSSLIB_API std::vector< std::pair< boost::shared_ptr< rcss::lib::LoaderImpl >, boost::weak_ptr< rcss::lib::LoaderImpl > > >;
RCSSLIB_EXTERN template class RCSSLIB_API std::allocator< std::pair< boost::shared_ptr< char >, boost::weak_ptr< rcss::lib::LoaderImpl > > >;
RCSSLIB_EXTERN template class RCSSLIB_API std::vector< std::pair< boost::shared_ptr< char >, boost::weak_ptr< rcss::lib::LoaderImpl > > >;
*/

namespace rcss
{
    namespace lib
    {
        class RCSSLIB_API WeakLoader;

        class RCSSLIB_API Loader
        {
        public:
            enum AutoExt { NO_AUTO_EXT = LoaderImpl::NO_AUTO_EXT,
                           AUTO_EXT = LoaderImpl::AUTO_EXT };
            enum Error { LIB_OK = LoaderImpl::LIB_OK, 
                         NOT_FOUND = LoaderImpl::NOT_FOUND,
                         INIT_ERROR = LoaderImpl::INIT_ERROR,
                         SYSTEM_ERROR = LoaderImpl::SYSTEM_ERROR };
	    enum ForceRecalc { TRY_CACHE, FORCE_RECALC };

            typedef boost::shared_ptr< LoaderImpl > Impl;
            typedef boost::weak_ptr< LoaderImpl > WeakImpl; 
            
        private:
	    static const char S_OK_ERR_STR[];
	    static const char S_NOT_FOUND_ERR_STR[];
	    static const char S_INIT_ERR_STR[];
	  
	    typedef std::vector< std::pair< boost::shared_ptr< char >, WeakImpl > > CacheMap;
            typedef std::vector< std::pair< Impl, WeakImpl > > DepMap;

        public:
            static
            void
            addPath( const char* path );

            static
            void
            setPath( const char* path );
            
            static
            const char*
            getPath();
            
	    /// returns the list of available modules that can
	    /// currently be loaded.  Getting the list can be time
	    /// consuming, so we cache the list.  The list is
	    /// invalidated whenever the path is changed or a library
	    /// is loaded (because loading a library might make a new
	    /// loader available).  A new list will be fetched
	    /// whenever the cached value is invalidated or /param
	    /// force is specified
	    static
	    std::vector< boost::shared_ptr< char > >
	    listAvailableModules( ForceRecalc = TRY_CACHE );

            static
            Impl
            loadFromCache( const char* lib );

            static
            size_t
            libsLoaded();

            static
	    boost::shared_ptr< char >
            strip( const boost::shared_ptr< char >& filename );
            
            static
	    boost::shared_ptr< char >
            stripDirName( const boost::shared_ptr< char >& filename );
            
            static
            boost::shared_ptr< char >
            stripExt( const boost::shared_ptr< char >& filename );
    

            Loader();           
            Loader( WeakLoader& loader );

            bool
            open( const char* lib, AutoExt autoext = AUTO_EXT );

            bool
            isOpen() const;

            void
            close();

            const char*
            name() const;

            const char*
            strippedName() const;

	    Error
	    error() const;

	    const char*
	    errorStr() const;
        private:
            static
            void
            priv_setPath( const boost::shared_ptr< char >& path );
            
            static
            boost::shared_ptr< char >
            priv_getPath();

            static
            void
            addToCache( const boost::shared_ptr< char >& lib_name, const Impl& lib );
            
            static
            void
            addDep( const Impl& lib );
            
            friend class LoaderImpl;
            friend class StaticLoader;
            friend class WeakLoader;
            
        private:
            static CacheMap s_cached_libs;
            static DepMap s_deps;
            
            Impl m_impl;
	    Error m_error;
	    boost::shared_ptr< char > m_system_err_str;

            static boost::shared_ptr< char > s_path;
	    static std::vector< boost::shared_ptr< char > > s_available;
	    static bool s_available_valid;

        };
        
        class RCSSLIB_API WeakLoader
        {
        public:
            WeakLoader( Loader& loader )
                : m_impl( loader.m_impl )
            {}
            
        private:
            friend class Loader;
            Loader::WeakImpl m_impl;
        };
	}
}



#if defined(_WIN32) || defined(__WIN32__) || defined WIN32
#  if defined _WINDLL
#    define RCSSLIBCLIENT_API __declspec(dllexport)
#  else
#    define RCSSLIBCLIENT_API
#  endif
#else
#  define RCSSLIBCLIENT_API
#endif

#define RCSSLIB_INIT( modulename ) \
namespace { const char* RCSS_MODULE_NAME = #modulename; } \
extern "C" RCSSLIBCLIENT_API \
bool \
modulename##_initialize()

#define RCSSLIB_FIN( modulename ) \
extern "C" RCSSLIBCLIENT_API \
void \
modulename##_finalize() 

#endif
