// -*-c++-*-

/***************************************************************************
                                factory.hpp 
                             -------------------
    Template singleton for creating polymorphic objects based on some idx
    begin                : 2002-10-08
    copyright            : (C) 2002 by The RoboCup Soccer Simulator 
                           Maintenance Group.
    email                : sserver-admin@lists.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU LGPL as published by the Free Software  *
 *   Foundation; either version 2 of the License, or (at your option) any  *
 *   later version.                                                        *
 *                                                                         *
 ***************************************************************************/


#ifndef RCSSFACTORY_H
#define RCSSFACTORY_H

#include <vector>
#include <list>
#include <iostream>
#include <memory>

namespace rcss
{
    namespace lib
    {
	template< typename X >
	class less
	{
        public:
            bool operator()( X a, X b )
            {
                return a < b;
            }
        };       
  
        template<>
	class less< const char* >
        {
        public:
            bool operator()( const char* a, const char* b )
            {
                return strcmp( a, b ) < 0;
            }
        };
     
        template<> 
	class less< char* >
        {
        public:
            bool operator()( char* a, char* b )
		{
		    return strcmp( a, b ) < 0;
		}
        };
    
      
        /*!
        //===========================================================
        //
        //  CLASS: RegHolderImpl
        //
        //  DESC: Base type for AutoReger
        //
        //===========================================================
        */

        class RegHolderImpl
        {
        public:
            RegHolderImpl()
            {}
        
            virtual 
            ~RegHolderImpl()
            {}
        };
    
        typedef std::auto_ptr< RegHolderImpl > RegHolder;


        /*!
        //===========================================================
        //
        //  CLASS: AutoReger
        //
        //  DESC: Used for automatic registration. 
        //  NOTE: Auto registration Cannot be used in dynamic libraries
        //
        //===========================================================
        */

        template< typename OF >
        class AutoReger
            : public RegHolderImpl
        {
        public:
            typedef OF Factory;
            typedef typename Factory::Creator Creator;
            typedef typename Factory::Index Index;
        
            AutoReger( Factory& fact, Creator creator, const Index& idx )
                : m_fact( fact ), 
                  m_idx( idx )
            {
                m_fact.reg( creator, idx ); 
            }
        
            virtual
            ~AutoReger()
            { m_fact.dereg( m_idx ); }
        private:
            template< class OF2 >
            AutoReger( const AutoReger< OF2 >& ); // not used

            template< class OF2 >
            AutoReger&
            operator=( const AutoReger< OF2 >& ); // not used

        private:
            Factory& m_fact;
            const Index& m_idx;
        };

        
/*!
//===================================================================
//
//  CLASS: Factory
//
//  DESC: An Generic Object Factory (aka Class Store)
//
//===================================================================
*/
 
        template< class Cre, 
                  class I = const char*,
                  class Com = less< I > >
        class Factory
        {
        public:
            typedef Cre Creator;
            typedef I Index;
            typedef Com Compare;
        private:
			typedef std::vector< std::pair< Index, std::vector< Creator > > > Map;
           
        public:       
            Factory()
            {}

            ~Factory()
            {}
            
            void
            reg( Creator c, const Index& idx )
            {
				for( typename Map::iterator i = m_creators.begin();
					 i != m_creators.end(); ++i )
				{
					if( !Compare()( idx, i->first ) && !Compare()( i->first, idx ) )
						{
                            i->second.push_back( c );
                            return;
                        }
				}
                std::vector< Creator > creators;
                creators.push_back( c );
                m_creators.push_back( std::make_pair( idx, creators ) ); 
            }

            void
            dereg( const Index& idx )
            {
				for( typename Map::iterator i = m_creators.begin();
					 i != m_creators.end(); ++i )
				{
					if( !Compare()( idx, i->first ) 
                        && !Compare()( i->first, idx ) )
					{
                        i->second.pop_back();
                        if( i->second.empty() )
                            m_creators.erase( i );
						return;
					}
				}
            }
            
            bool
            getCreator( Creator& c, const Index& idx )
            {
                for( typename Map::iterator i = m_creators.begin();
					 i != m_creators.end(); ++i )
				{
					if( !Compare()( idx, i->first ) 
                        && !Compare()( i->first, idx ) )
                    {
						c = i->second.back();
                        return true;
                    }
				}
                return false;
            }

            std::list< Index >
            list()
            {
                std::list< Index > rval;
                for( typename Map::iterator i = m_creators.begin();
                     i != m_creators.end(); ++i )
                    rval.push_back( i->first );
                return rval;
            }
            
            std::ostream&
            printList( std::ostream& o = std::cout )
            {
                for( typename Map::iterator i = m_creators.begin();
                     i != m_creators.end(); ++i )
                    o << "\t" << i->first 
                      << "(" << i->second.size() << ")" 
                      << std::endl;
                return o;
            }

            size_t
            size() const
            { return m_creators.size(); }

            size_t
            size( const Index& idx ) const
            {
				for( typename Map::const_iterator i = m_creators.begin();
					 i != m_creators.end(); ++i )
				{
					if( !Compare()( idx, i->first ) 
                        && !Compare()( i->first, idx ) )
					{
						return i->second.size();
					}
				}
                return 0;
            }

            RegHolder
            autoReg( Creator c, const Index& i )
            {
                return RegHolder( new AutoReger< Factory< Creator, Index, Compare > >( *this, c, i ) ); 
            }

        private:
            Map m_creators;
        };
    }
}

#endif
