// -*-c++-*-

/***************************************************************************
                                parser.hpp  
                          Parser for Config options
                             -------------------
    begin                : 14-MAY-2003
    copyright            : (C) 2003 by The RoboCup Soccer Server 
                           Maintenance Group.
    email                : sserver-admin@lists.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU LGPL as published by the Free Software  *
 *   Foundation; either version 2 of the License, or (at your option) any  *
 *   later version.                                                        *
 *                                                                         *
 ***************************************************************************/

#ifndef PARSER_HPP
#define PARSER_HPP

#include <string>
#include "rcssbase/parser.h"
#include <boost/shared_ptr.hpp>
#include <list>
#include "../lib/shared_ptr.hpp"

extern int RCSS_CONF_parse( void * );

class RCSS_CONF_Lexer;

namespace rcss
{
  namespace conf
  {
      class Lexer;
      
      class GenericBuilder;
      
      class Holder
      {
      public:
          enum Type { S_INT, S_DOUB, S_BOOL, S_STR };

      private:
          
          union 
          {
              int m_int;
              double m_double;
              bool m_bool;
          };
          std::string m_str;
          Type m_type;

      public:
          Holder&
          operator=( int x )
          {
              m_int = x;
              m_type = S_INT;
              return *this;
          }
          
          Holder&
          operator=( double x )
          {
              m_double = x;
              m_type = S_DOUB;
              return *this;
          }
          
          Holder&
          operator=( bool x )
          {
              m_bool = x;
              m_type = S_BOOL;
              return *this;
          }
          
          Holder&
          operator=( const std::string& x )
          {
              m_str = x;
              m_type = S_STR;
              return *this;
          }
          
          Type
          getType() const
          { return m_type; }

          int
          getInt() const
          { return m_int; }
          
          double
          getDouble() const
          { return m_double; }
          
          bool
          getBool() const
          { return m_bool; }
          
          std::string&
          getStr()
          { return m_str; }
          
          const std::string&
          getStr() const
          { return m_str; }
      };


      
      class Parser
          : public rcss::Parser
      {
      public:
          typedef conf::Lexer Lexer;
          
      private:	  
          class Param
          {
          public:
              Lexer* m_lexer;
              GenericBuilder& m_builder;
	    std::string m_str;

              class StreamState
              {
              public:
                  std::istream* m_strm;
                  std::string m_name;
                  int m_lineno;
                  
                  StreamState( std::istream& strm, const std::string& name, int lineno )
		    : m_strm( &strm ),
		      m_name( name ),
		      m_lineno( lineno )
                  {}
              };

              typedef std::list< StreamState > StreamStack;
            
              StreamStack m_stack;

	      std::map< const char*, lib::shared_ptr<GenericBuilder> > m_modules;

          public:
              void
              setStream( std::istream& strm, const std::string& name = "" );
            
              void
              setStream( std::istream& strm, int lineno );

              void
              restoreStream();
            
              void
              setStream( std::istream& strm, const std::string& name, int lineno );
              
            
              Param( GenericBuilder& builder );
              
              ~Param();

	    void
	    setCurrentStr( const std::string& str )
	    {
	      m_str = str;
	    }

          private:
	    
              Param( const Param& param ); // not used
              
              Parser::Param&
              Parser::Param::operator=( const Parser::Param& param ); // not used

        
          public:
              bool
              parse( std::istream& strm, const std::string& streamname );

              Lexer&
              getLexer()
              { return *m_lexer; }
              
              GenericBuilder&
              getBuilder()
              { return m_builder; }
              
	    const std::string&
	    name() const
	    { return m_stack.front().m_name; }

	    int
	    lineno() const
	    { return m_stack.front().m_lineno; }

	    const std::string&
	    currToken()
	    { return m_str; }

              void
              newLine( int count = 1 );
              
              bool
              include( const std::string& filename );

	      void
	      load( const std::string& libname );

	      void
	      setpath( const std::string& path );

	      void
	      addpath( const std::string& path );
	  };

      private:
          Param m_param;

          friend int RCSS_CONF_lex( Holder*,
                                    Param& param );
          friend Param& getParam( void* );
          friend class Param;
          friend class RCSS_CONF_Lexer;

          template< class V >
          friend
          bool
          buildParam( Parser::Param& param,
                      const std::string& module_name, 
                      const std::string& param_name,
                      V value );
          
          bool
          doParse( std::istream& strm );
                    
      public:
          Parser( GenericBuilder& builder );

          ~Parser();

          bool
          parse( int argc, const char * const * argv );

          bool
          parse( std::istream& strm );

          bool
          parse( std::istream& strm, const std::string& name );

          bool
          parse( const std::string& file );

          bool
          parseCreateConf( const std::string& conf_name );

      };
  }
}


#endif
