// -*-c++-*-

/***************************************************************************
                               genericbuilder.hpp  
                Interface for building config parameter
                             -------------------
    begin                : 16-JUN-2003
    copyright            : (C) 2003 by The RoboCup Soccer Server 
                           Maintenance Group.
    email                : sserver-admin@lists.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU LGPL as published by the Free Software  *
 *   Foundation; either version 2 of the License, or (at your option) any  *
 *   later version.                                                        *
 *                                                                         *
 ***************************************************************************/

#ifndef GENERICBUILDER_HPP
#define GENERICBUILDER_HPP

#include <string>
#include <iostream>
#include <list>
#include <boost/shared_ptr.hpp>

#include "../lib/factory.hpp"
#include "../lib/shared_ptr.hpp"

namespace rcss
{
    namespace conf
    {
      class Parser;
      class GenericBuilder;

	class StatusHandler
	{
	public:
	    StatusHandler();

	    virtual
	    ~StatusHandler();

	  /// called after a hander has been subscribed
	    virtual
	    void
	    handlerAdded( GenericBuilder& publisher );


	  /// called after a handler has been unsubscribed
	    virtual
	    void
	    handlerRemoved( GenericBuilder& publisher );

	    virtual
	    void
	    parseError( const std::string& curr,
			const std::string& err,
			const std::string& name,
			int lineno );

	    virtual
	    void
	    buildError( const std::string& module,
			const std::string& param,
			const std::string& err,
			const std::string& name,
			int lineno );

	    virtual
	    void
	    buildWarning( const std::string& module,
			  const std::string& param,
			  const std::string& warn,
			  const std::string& name,
			  int lineno );

	    virtual
	    void
	    creatingConfFile( const std::string& conf_name );

	    virtual
	    void
	    createdConfFile( const std::string& conf_name );

	    virtual
	    void
	    confCreationFailed( const std::string& conf_name,
				int error );
	  virtual
	  void
	  includeFailed( const std::string& filename,
			 int error,
			 const std::string& name,
			 int lineno );

	  virtual
	  void
	  loadFailed( const std::string& libname,
		      const std::string& error,
		      const std::list< std::string >& avail,
		      const std::string& name,
		      int lineno );


	};       

        class GenericBuilder
        {
        public:
	    typedef rcss::lib::shared_ptr< GenericBuilder > Ptr;
	    typedef Ptr(*Creator)( GenericBuilder* parent );
	    typedef rcss::lib::Factory< Creator > Factory;

	    static
	    Factory&
	    factory();

	    
	    GenericBuilder( const std::string& progname );

	    GenericBuilder( GenericBuilder* parent );
            
            virtual
            ~GenericBuilder();

            template< typename V >
            bool
            buildParam( const std::string& module_name,
                        const std::string& param_name,
                        V value,
			const std::string& name,
			int lineno );

            void
            createConfFile( std::ostream& conf );

	    void
	    displayHelp();

	  virtual
	  void
	  displayUsage( const std::string& ) {}

	  virtual
	  void
	  displayGenericHelp() {}

	  virtual
	  void
	  displayDetailedHelp() {}

	  /// removeHandler must be called before the handler is destroyed
	    void
	    addHandler( StatusHandler& handler );

	  
	    void
	    removeHandler( StatusHandler& handler );

	    void
	    manageModule( const rcss::lib::Loader& module );

	    void
	    manageChild( const rcss::lib::shared_ptr< GenericBuilder >& child );

	    void
	    clearModules();
	private:
	    std::list< rcss::lib::Loader > m_modules;
	    std::list< rcss::lib::shared_ptr< GenericBuilder > > m_managed_children;
       protected:
	    void
	    addChild( GenericBuilder& child );

	    void
	    removeChild( GenericBuilder& child );



            template< typename V >
            void
            createConfFileEntry( std::ostream& conf, 
                                 const std::string& module_name,
                                 const std::string& param_name,
                                 V value,
                                 const std::string& desc );

	    template< typename V >
	    void
	    displayHelpEntry( std::ostream& out,
			      const std::string& module_name,
			      const std::string& param_name,
			      V value,
			      const std::string& desc );

	    void
	    displayHelpEntry( std::ostream& out,
			      const std::string& module_name,
			      const std::string& param_name,
			      const std::string& desc );

	    void
	    displayHelpEntry( std::ostream& out,
			      const std::string& param_name,
			      const std::string& desc );

            template< typename V >
            void
            writeConfValue( std::ostream& conf, V value );


            template< typename V >
            void
            writeConfType( std::ostream& conf, V value );

          private:
            virtual
            bool
            doBuildParam( const std::string& module_name,
                          const std::string& param_name,
                          int value,
			  const std::string& name,
			  int lineno ) = 0;

            virtual
            bool
            doBuildParam( const std::string& module_name,
                          const std::string& param_name,
                          bool value,
			  const std::string& name,
			  int lineno ) = 0;

            virtual
            bool
            doBuildParam( const std::string& module_name,
                          const std::string& param_name,
                          double value,
			  const std::string& name,
			  int lineno ) = 0;

            virtual
            bool
            doBuildParam( const std::string& module_name,
                          const std::string& param_name,
                          const std::string& value,
			  const std::string& name,
			  int lineno ) = 0;

            virtual
            void
            doCreateConfFile( std::ostream& conf ) = 0;

	public:

	  void
	  parseError( const std::string& curr,
		      const std::string& err,
		      const std::string& name,
		      int lineno );

	protected:
	    void
	    buildError( const std::string& module,
		     const std::string& param,
		     const std::string& err,
		     const std::string& name,
		     int lineno );

	    void
	    buildWarning( const std::string& module,
		       const std::string& param,
		       const std::string& warn,
		       const std::string& name,
		       int lineno );
	public:
	    void
	    creatingConfFile( const std::string& conf_name );

	    void
	    createdConfFile( const std::string& conf_name );

	    void
	    confCreationFailed( const std::string& conf_name,
				  int error );

	  void
	  includeFailed( const std::string& filename,
			 int error,
			 const std::string& name,
			 int lineno );

	  void
	  loadFailed( const std::string& libname,
		      const std::string& error,
		      const std::list< std::string >& avail,
		      const std::string& name,
		      int lineno );

	    bool
	    success() const;

	  void
	  reset();

	  const std::string&
	  progName() const;
	      
	  bool
	  genericHelpRequested() const;

	  bool
	  detailedHelpRequested() const;

	  void
	  requestGenericHelp();

	  void
	  requestDetailedHelp( const std::string& module_name );

	  virtual
	  void
	  doRequestDetailedHelp( const std::string& );

	  void
	  addedToParser( Parser& p );

	  void
	  removedFromParser();

	  Parser*
	  parser();
	protected:
	  void
	  requestDetailedHelp();
	private:

	  // does not take ownership
	  Parser* m_parser;
	    std::list< StatusHandler* > m_handlers;
	    std::list< GenericBuilder* > m_children;



	  bool m_err;

	  std::string m_progname;
	  GenericBuilder* m_parent;
	  bool m_generic_help_requested;
	  bool m_detailed_help_requested;

        };  

	template< typename V >
	inline
	bool
	GenericBuilder::buildParam( const std::string& module_name,
				    const std::string& param_name,
				    V value,
				    const std::string& name,
				    int lineno )
	{
	  bool rval = doBuildParam( module_name, param_name, value, name, lineno );
	  for( std::list< GenericBuilder* >::iterator i = m_children.begin();
	       i != m_children.end(); ++i )
	    rval = (*i)->buildParam( module_name, param_name, value, name, lineno ) && rval;
	  return rval;
	}


	template< typename V >
	inline
	void
	GenericBuilder::createConfFileEntry( std::ostream& conf, 
					     const std::string& module_name,
					     const std::string& param_name,
					     V value,
					     const std::string& desc )
	{
	    conf << "# " << module_name << "::" 
		 << param_name << std::endl;
	    if( !desc.empty() )
	    {
		conf << "/* ";
		int count = 3;
		std::string::const_iterator start = desc.begin();
		std::string::const_iterator end = desc.begin();
		for( std::string::const_iterator i = desc.begin();
		     i != desc.end(); ++i, ++count )
		{
		    switch( *i )
		    {
			case '\n':
			    end = i;
			    conf << std::string( start, end ) << std::endl;
			    count = 0;
			    start = end = i+1;
			    break;
			case ' ':
			case '\t':
			    end = i;
			    break;
			default:
			    if( count > 70 )
			    {
				conf << std::string( start, end ) << std::endl;
				for( std::string::const_iterator j = end;
				     j != i; ++j )
				{
				    if( *j == ' ' || *j == '\t' )
					++end;
				    else
					break;
				}
				count = std::distance( end, i );
				start = end;
			    }
			    break;
		    }
		}
		conf << std::string( start, desc.end() ) << " */\n";
	    }
	    conf << module_name << "::" << param_name 
		 << " = ";
	    writeConfValue( conf, value );
	    conf << std::endl << std::endl;
	}


	template< typename V >
	inline
	void
	GenericBuilder::displayHelpEntry( std::ostream& conf, 
					  const std::string& module_name,
					  const std::string& param_name,
					  V value,
					  const std::string& desc )
	{
	  conf << "\t" << module_name << "::" 
	       << param_name << "=";
	      writeConfType( conf, value );
	      conf << std::endl;
	    if( !desc.empty() )
	    {
		conf << "\t\t";
		int count = 3;
		std::string::const_iterator start = desc.begin();
		std::string::const_iterator end = desc.begin();
		for( std::string::const_iterator i = desc.begin();
		     i != desc.end(); ++i, ++count )
		{
		    switch( *i )
		    {
			case '\n':
			    end = i;
			    conf << std::string( start, end ) << std::endl << "\t\t";
			    count = 0;
			    start = end = i+1;
			    break;
			case ' ':
			case '\t':
			    end = i;
			    break;
			default:
			    if( count > 62 )
			    {
			        conf << std::string( start, end ) << std::endl << "\t\t";
				for( std::string::const_iterator j = end;
				     j != i; ++j )
				{
				    if( *j == ' ' || *j == '\t' )
					++end;
				    else
					break;
				}
				count = std::distance( end, i );
				start = end;
			    }
			    break;
		    }
		}
		conf << std::string( start, desc.end() ) << "\n\n";
	    }
	    conf << "\t\tcurrent value: ";
	    writeConfValue( conf, value );
	    conf << std::endl << std::endl;
	}

	inline
	void
	GenericBuilder::displayHelpEntry( std::ostream& conf, 
					  const std::string& module_name,
					  const std::string& param_name,
					  const std::string& desc )
	{
	  conf << "\t" << module_name << "::" << param_name;
	  conf << std::endl;
	    if( !desc.empty() )
	    {
		conf << "\t\t";
		int count = 3;
		std::string::const_iterator start = desc.begin();
		std::string::const_iterator end = desc.begin();
		for( std::string::const_iterator i = desc.begin();
		     i != desc.end(); ++i, ++count )
		{
		    switch( *i )
		    {
			case '\n':
			    end = i;
			    conf << std::string( start, end ) << std::endl << "\t\t";
			    count = 0;
			    start = end = i+1;
			    break;
			case ' ':
			case '\t':
			    end = i;
			    break;
			default:
			    if( count > 62 )
			    {
			        conf << std::string( start, end ) << std::endl << "\t\t";
				for( std::string::const_iterator j = end;
				     j != i; ++j )
				{
				    if( *j == ' ' || *j == '\t' )
					++end;
				    else
					break;
				}
				count = std::distance( end, i );
				start = end;
			    }
			    break;
		    }
		}
		conf << std::string( start, desc.end() ) << "\n\n";
	    }
	}


	inline
	void
	GenericBuilder::displayHelpEntry( std::ostream& conf, 
					  const std::string& param_name,
					  const std::string& desc )
	{
	  conf << "\t" << param_name;
	  conf << std::endl;;
	    if( !desc.empty() )
	    {
		conf << "\t\t";
		int count = 3;
		std::string::const_iterator start = desc.begin();
		std::string::const_iterator end = desc.begin();
		for( std::string::const_iterator i = desc.begin();
		     i != desc.end(); ++i, ++count )
		{
		    switch( *i )
		    {
			case '\n':
			    end = i;
			    conf << std::string( start, end ) << std::endl << "\t\t";
			    count = 0;
			    start = end = i+1;
			    break;
			case ' ':
			case '\t':
			    end = i;
			    break;
			default:
			    if( count > 62 )
			    {
			        conf << std::string( start, end ) << std::endl << "\t\t";
				for( std::string::const_iterator j = end;
				     j != i; ++j )
				{
				    if( *j == ' ' || *j == '\t' )
					++end;
				    else
					break;
				}
				count = std::distance( end, i );
				start = end;
			    }
			    break;
		    }
		}
		conf << std::string( start, desc.end() ) << "\n\n";
	    }
	}

	
        template< typename V >
        inline
        void
        GenericBuilder::writeConfValue( std::ostream& conf, V value )
        {
            conf << value;
        }

        template<>
        inline
        void
        GenericBuilder::writeConfValue< bool >( std::ostream& conf, 
						bool value )
        {
	    if( value )
		conf << "true";
	    else
		conf << "false";
        }

        template<>
        inline
        void
        GenericBuilder::writeConfValue< const bool& >( std::ostream& conf, 
						       const bool& value )
        {
	    if( value )
		conf << "true";
	    else
		conf << "false";
        }

        template<>
        inline
        void
        GenericBuilder::writeConfValue< const std::string& >( std::ostream& conf, 
                                                              const std::string& value )
        {
            conf << '\'' << value << '\'';
        }

        template<>
        inline
        void
        GenericBuilder::writeConfValue< std::string >( std::ostream& conf, 
                                                       std::string value )
        {
            conf << '\'' << value << '\'';
        }



        template< typename V >
        inline
        void
        GenericBuilder::writeConfType( std::ostream& conf, V value )
        {
            conf << "<VALUE>";
        }

        template<>
        inline
        void
        GenericBuilder::writeConfType< bool >( std::ostream& conf, bool )
        {
            conf << "<on|off|true|false|1|0|>";
        }

        template<>
        inline
        void
        GenericBuilder::writeConfType< int >( std::ostream& conf, int )
        {
            conf << "<INTEGER>";
        }


        template<>
        inline
        void
        GenericBuilder::writeConfType< double >( std::ostream& conf, double )
        {
            conf << "<REAL>";
        }

        template<>
        inline
        void
        GenericBuilder::writeConfType< const std::string& >( std::ostream& conf, 
							     const std::string& )
        {
            conf << "'<STRING>'";
        }

        template<>
        inline
        void
        GenericBuilder::writeConfType< std::string >( std::ostream& conf, 
						      std::string )
        {
            conf << "'<STRING>'";
        }
    }
}

#endif
