// -*-c++-*-

/***************************************************************************
                               builder.hpp  
                Interface for building config parameter
                             -------------------
    begin                : 14-MAY-2003
    copyright            : (C) 2003 by The RoboCup Soccer Server 
                           Maintenance Group.
    email                : sserver-admin@lists.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU LGPL as published by the Free Software  *
 *   Foundation; either version 2 of the License, or (at your option) any  *
 *   later version.                                                        *
 *                                                                         *
 ***************************************************************************/

#ifndef BUILDER_HPP
#define BUILDER_HPP

#include "genericbuilder.hpp"
#include <map>
#include <boost/shared_ptr.hpp>
#include <iosfwd>

namespace rcss
{
  namespace conf
  {
      	class StreamStatusHandler
	  : public StatusHandler
	{
	public:
	  StreamStatusHandler();
	  StreamStatusHandler( std::ostream& errstrm );

	  virtual
	  ~StreamStatusHandler();

	    virtual
	    void
	    parseError( const std::string& curr,
			const std::string& err,
			const std::string& name,
			int lineno );

	    virtual
	    void
	    buildError( const std::string& module,
			const std::string& param,
			const std::string& err,
			const std::string& name,
			int lineno );

	    virtual
	    void
	    buildWarning( const std::string& module,
			  const std::string& param,
			  const std::string& warn,
			  const std::string& name,
			  int lineno );

	    virtual
	    void
	    creatingConfFile( const std::string& conf_name );

	    virtual
	    void
	    createdConfFile( const std::string& conf_name );

	    virtual
	    void
	    confCreationFailed( const std::string& conf_name,
				int error );

	  virtual
	  void
	  includeFailed( const std::string& filename,
			 int error,
			 const std::string& name,
			 int lineno );

	  virtual
	  void
	  loadFailed( const std::string& libname,
		      const std::string& error,
		      const std::list< std::string >& avail,
		      const std::string& name,
		      int lineno );
	private:
	  std::ostream& m_errstrm;
	};       


      class Builder
          : public GenericBuilder
      {
      private:
          template< typename Arg >
          class SetterBase
          {
          public:
              virtual
              ~SetterBase()
              {}
          
              virtual
              void
              operator()( Arg arg ) = 0;

          };
      
          template< typename Arg, typename C >
          class SetterMFun
              : public SetterBase< Arg >
          {
          public:
              SetterMFun( C* c, void(C::*fun)( Arg ) )
                  : m_class( c ),
                    m_fun( fun )
              {}
          
              void
              operator()( Arg arg )
              { (m_class->*m_fun)( arg ); }

          private:
              C* m_class;
              void(C::*m_fun)( Arg );
          };
      
          template< typename Arg >
          class SetterPFun
              : public SetterBase< Arg >
          {
          public:
              SetterPFun( void(*fun)( Arg ) )
                  : m_fun( fun )
              {}
          
              void
              operator()( Arg arg )
              { (*m_fun)( arg ); }
          
          private:
              void(*m_fun)( Arg );
          };
      
          template< typename Arg >
          class SetterArg
              : public SetterBase< Arg >
          {
          public:
              SetterArg( Arg& arg )
                  : m_arg( arg )
              {}
          
              void
              operator()( Arg arg )
              { m_arg = arg; }
          
          private:
              Arg& m_arg;
          };

      public:
          template< typename Arg >
          class Setter
          {
          public:
              Setter( const Setter< Arg >& s )
                  : m_setter( s.m_setter )
              {}

              template< typename C >
              Setter( C* c, void(C::*fun)( Arg ) )
                  : m_setter( new rcss::conf::Builder::SetterMFun< Arg, C >( c, fun ) )
              {}  

              Setter( void(*fun)( Arg ))
                  : m_setter( new rcss::conf::Builder::SetterPFun< Arg >( fun ) )
              {}

              Setter( Arg& arg )
                  : m_setter( new rcss::conf::Builder::SetterArg< Arg >( arg ) )
              {}  

              void
              operator()( Arg arg )
              { m_setter->operator()( arg ); }
          
          private:
              boost::shared_ptr< rcss::conf::Builder::SetterBase< Arg > > m_setter;
          };

          template< typename Arg >
	  static
          inline
          Setter< Arg >
          makeSetter( Arg& arg )
          { return Setter< Arg >( arg ); }
      
          template< typename Arg >
	  static
          inline
          Setter< Arg >
          makeSetter( void(*&pfun)( Arg ) )
          { return Setter< Arg >( pfun ); }
      
          template< typename Arg, typename C >
          static
	  inline
          Setter< Arg >
          makeSetter( C* c, void(C::*mfun)( Arg ) )
          { return Setter< Arg >( c, mfun ); }
      
          template< typename Arg, typename C >
          static
	  inline
          Setter< Arg >
          makeSetter( C& c, void(C::*mfun)( Arg ) )
          { return Setter< Arg >( &c, mfun ); }

      private:

          template< typename RVal >
          class GetterBase
          {
          public:
              virtual
              ~GetterBase()
              {}
          
              virtual
              RVal
              operator()() const = 0;

          };
      
          template< typename RVal, typename C >
          class GetterMFun
              : public GetterBase< RVal >
          {
          public:
              GetterMFun( const C* c, RVal(C::*fun)() const )
                  : m_class( c ),
                    m_fun( fun )
              {}
	      
              RVal
              operator()() const
              {
		  return (m_class->*m_fun)(); 
	      }
          

          private:
              const C* m_class;
              RVal(C::*m_fun)() const;
          };

          template< typename RVal >
          class GetterPFun
              : public GetterBase< RVal >
          {
          public:
              GetterPFun( RVal(*fun)() )
                  : m_fun( fun )
              {}
          
              RVal
              operator()() const
              {
		  return (*m_fun)(); 
	      }
          

          private:
              RVal(*m_fun)();
          };
      
          template< typename RVal >
          class GetterRVal
              : public GetterBase< RVal >
          {
          public:
              GetterRVal( const RVal& rval )
                  : m_rval( rval )
              {}
          
              RVal
              operator()() const
              {
                  return m_rval; 
              }
          
          private:
              const RVal& m_rval;
          };

      public:
          template< typename RVal >
          class Getter
          {
          public:
              template< typename C >
              Getter( const C* c, RVal(C::*fun)() const )
                  : m_getter( new rcss::conf::Builder::GetterMFun< RVal, C >( c, fun ) )
              {}  
          
              Getter( RVal(*fun)() )
                  : m_getter( new rcss::conf::Builder::GetterPFun< RVal >( fun ) )
              {}  
          
              Getter( const RVal& rval )
                  : m_getter( new rcss::conf::Builder::GetterRVal< RVal >( rval ) )
              {}  
          
              RVal
              operator()() const
              {
		  return (*m_getter)(); 
	      }
          
          private:
              boost::shared_ptr< rcss::conf::Builder::GetterBase< RVal > > m_getter;
          };

          template< typename RVal >
          static
	  inline
          Getter< RVal >
          makeGetter( const RVal& rval )
          {
              return Getter< RVal >( rval ); 
          }
      
          template< typename RVal >
          static
	  inline
          Getter< RVal >
          makeGetter( RVal(*&pfun)() )
          { return Getter< RVal >( pfun ); }
      
      
          template< typename RVal, typename C >
          static
	  inline
          Getter< RVal >
          makeGetter( C* c, RVal(C::*mfun)() const )
          { return Getter< RVal >( c, mfun ); }
      
          template< typename RVal, typename C >
          static
	  inline
          Getter< RVal >
          makeGetter( const C& c, RVal(C::*mfun)() const )
          { return Getter< RVal >( &c, mfun ); }

      private:
          template< typename V >
          class ParamInfo
          {
          public:
              ParamInfo( const rcss::conf::Builder::Setter< V >& setter,
                         const rcss::conf::Builder::Getter< V >& getter,
                         const std::string& desc )
                  : m_setter( setter ),
                    m_getter( getter ),
                    m_desc( desc )
              {}
              
              void
              set( V value )
              { m_setter( value ); }

              V
              get() const
              { 
                  return m_getter(); 
              }

              const std::string&
              desc() const
              { return m_desc; }
          private:
              rcss::conf::Builder::Setter< V > m_setter;
              rcss::conf::Builder::Getter< V > m_getter;
              const std::string m_desc;
          };

          typedef std::map< std::string, ParamInfo< int > > IntMap;
          typedef std::map< std::string, ParamInfo< bool > > BoolMap;
          typedef std::map< std::string, ParamInfo< double > > DoubMap;
          typedef std::map< std::string, ParamInfo< std::string > > StrMap;


      public:
	Builder( const std::string& progname,
		 const std::string& module_name );
	Builder( GenericBuilder* parent,
		 const std::string& module_name );

          virtual
          ~Builder();
          
          const std::string&
          getModuleName() const;      

	  virtual
	  void
	  doRequestDetailedHelp( const std::string& module_name );


      public:
          template< typename P >
          bool
          set( const std::string& param, P value )
          {
	    return doBuildParam( param, value, "none", 0 );
          }
 
          template< typename P >
          bool
          get( const std::string& param,
	       P& value ) const;
 
      protected:          
	  virtual
	  void
	  displayUsage( const std::string& progname );

	  virtual
	  void
	  displayGenericHelp();

	  virtual
	  void
	  displayDetailedHelp();

      private:
          virtual
          bool
          doBuildParam( const std::string& module_name,
                        const std::string& param_name,
                        int value,
			const std::string& name,
			int lineno );

          virtual
          bool
          doBuildParam( const std::string& module_name,
                        const std::string& param_name,
                        bool value,
			const std::string& name,
			int lineno );

          virtual
          bool
          doBuildParam( const std::string& module_name,
                        const std::string& param_name,
                        double value,
			const std::string& name,
			int lineno );

          virtual
          bool
          doBuildParam( const std::string& module_name,
                        const std::string& param_name,
                        const std::string& value,
			const std::string& name,
			int lineno );

          virtual
          void
          doCreateConfFile( std::ostream& conf );

          template< typename V >
          void
          addParam( const std::string& name, 
                    const ParamInfo< V >& info );

      public:
          template< typename V >
          void
          addParam( const std::string& name, 
                    const Setter< V >& setter,
                    const Getter< V >& getter, 
                    const std::string& desc )
          { addParam< V >( name, ParamInfo< V >( setter, getter, desc ) ); }

          template< typename V >
          void
          addParam( const std::string& name, 
                    V& param, 
                    const std::string& desc )
          { 
              addParam( name, 
                        makeSetter( param ),
                        makeGetter( param ),
                        desc );
          }

      private:
          bool
          doBuildParam( const std::string& param_name, int value,
			const std::string& name,
			int lineno );

          bool
          doBuildParam( const std::string& param_name, bool value,
			const std::string& name,
			int lineno );

          bool
          doBuildParam( const std::string& param_name, double value,
			const std::string& name,
			int lineno );

          bool
          doBuildParam( const std::string& param_name, const std::string& value,
			const std::string& name,
			int lineno );

          bool
          setParam( const std::string& param_name,
                    ParamInfo< int > param,
                    int value,
		    const std::string& name,
		    int lineno );

          bool
          setParam( const std::string& param_name,
                    ParamInfo< int > param,
                    bool value,
		    const std::string& name,
		    int lineno );

          bool
          setParam( const std::string& param_name,
                    ParamInfo< int > param,
                    double value,
		    const std::string& name,
		    int lineno );

          bool
          setParam( const std::string& param_name,
                    ParamInfo< int > param,
                    const std::string& value,
		    const std::string& name,
		    int lineno );

          bool
          setParam( const std::string& param_name,
                    ParamInfo< bool > param,
                    int value,
		    const std::string& name,
		    int lineno );

          bool
          setParam( const std::string& param_name,
                    ParamInfo< bool > param,
                    bool value,
		    const std::string& name,
		    int lineno );

          bool
          setParam( const std::string& param_name,
                    ParamInfo< bool > param,
                    double value,
		    const std::string& name,
		    int lineno );

          bool
          setParam( const std::string& param_name,
                    ParamInfo< bool > param,
                    const std::string& value,
		    const std::string& name,
		    int lineno );

          bool
          setParam( const std::string& param_name,
                    ParamInfo< double > param,
                    int value,
		    const std::string& name,
		    int lineno );

          bool
          setParam( const std::string& param_name,
                    ParamInfo< double > param,
                    bool value,
		    const std::string& name,
		    int lineno );

          bool
          setParam( const std::string& param_name,
                    ParamInfo< double > param,
                    double value,
		    const std::string& name,
		    int lineno );

          bool
          setParam( const std::string& param_name,
                    ParamInfo< double > param,
                    const std::string& value,
		    const std::string& name,
		    int lineno );

          bool
          setParam( const std::string& param_name,
                    ParamInfo< std::string > param,
                    int value,
		    const std::string& name,
		    int lineno );

          bool
          setParam( const std::string& param_name,
                    ParamInfo< std::string > param,
                    bool value,
		    const std::string& name,
		    int lineno );

          bool
          setParam( const std::string& param_name,
                    ParamInfo< std::string > param,
                    double value,
		    const std::string& name,
		    int lineno );

          bool
          setParam( const std::string& param_name,
                    ParamInfo< std::string > param,
                    const std::string& value,
		    const std::string& name,
		    int lineno );




      private:
          const std::string m_module_name;
          IntMap m_ints;
          BoolMap m_bools;
          DoubMap m_doubs;
          StrMap m_strs;
      };   

      template<>
//       inline
      bool
      Builder::get< int >( const std::string& param,
			   int& value ) const;
//       {
//           IntMap::const_iterator i = m_ints.find( param );
//           if( i == m_ints.end() )
// 	      return false;
	  
// 	  value = i->second.get();
// 	  return true;
//       }
      
      template<>
//       inline
      bool
      Builder::get< bool >( const std::string& param,
			    bool& value ) const;
//       {
//           BoolMap::const_iterator i = m_bools.find( param );
//           if( i == m_bools.end() )
// 	      return false;
// 	  value = i->second.get();
//           return true;
//       }
      
      template<>
//       inline
      bool
      Builder::get< double >( const std::string& param,
			      double& value ) const;
//       {
//           DoubMap::const_iterator i = m_doubs.find( param );
//           if( i == m_doubs.end() )
// 	      return false;
// 	  value = i->second.get();
//           return true;
//       }
      
      template<>
//       inline
      bool
      Builder::get< std::string >( const std::string& param,
				   std::string& value ) const;
//       {
//           StrMap::const_iterator i = m_strs.find( param );
//           if( i == m_strs.end() )
//               return false;
// 	  value = i->second.get();
//           return true;
//       }


      template<>
//       inline
      void
      Builder::addParam< int >( const std::string& name, 
                                const ParamInfo< int >& info );
//       {
//           m_ints.insert( std::make_pair( name, info ) );
//       }
      
      template<>
//       inline
      void
      Builder::addParam< bool >( const std::string& name, 
                                 const ParamInfo< bool >& info );
//       {
//           m_bools.insert( std::make_pair( name, info ) );
//       }
      
      template<>
//       inline
      void
      Builder::addParam< double >( const std::string& name, 
                                   const ParamInfo< double >& info );
//       {
//           m_doubs.insert( std::make_pair( name, info ) );
//       }
      
      template<>
//       inline
      void
      Builder::addParam< std::string >( const std::string& name, 
                                        const ParamInfo< std::string >& info );
//       {
//           m_strs.insert( std::make_pair( name, info ) );
//       }
      
      
  }
}

#endif
