package razgriz.news.ui;

import java.util.MissingResourceException;
import java.util.Properties;
import java.util.ResourceBundle;

import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.FontRegistry;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

import razgriz.news.NewsPlugin;
import razgriz.news.favlist.FavoriteListService;
import razgriz.news.model.FavoriteList;

/**
 * The main plugin class to be used in the desktop.
 */
public class NewsUIPlugin extends AbstractUIPlugin {
    public static final String ID = NewsUIPlugin.class.getName();

    //The shared instance.
    private static NewsUIPlugin plugin;
    //Resource bundle.
    private ResourceBundle resourceBundle;

    private FavoriteList favoriteList;

    /**
     * The constructor.
     */
    public NewsUIPlugin() {
        super();
        plugin = this;
        try {
            resourceBundle = ResourceBundle.getBundle(
                "razgriz.news.ui.NewsUIPluginResources"
            );
        } catch (MissingResourceException x) {
            resourceBundle = null;
        }
    }

    /**
     * This method is called upon plug-in activation
     */
    public void start(BundleContext context) throws Exception {
        super.start(context);
        initFontRegistry();

        FavoriteListService service =
            NewsPlugin.getDefault().getFavoriteListService();
        favoriteList = service.load(NewsUI.FAVORITE_LIST_NAME);

        refreshProxySet();
    }

    /**
     * This method is called when the plug-in is stopped
     */
    public void stop(BundleContext context) throws Exception {
        FavoriteListService service =
            NewsPlugin.getDefault().getFavoriteListService();
        service.save(NewsUI.FAVORITE_LIST_NAME);

        super.stop(context);
    }

    /**
     * Returns the shared instance.
     */
    public static NewsUIPlugin getDefault() {
        return plugin;
    }

    /**
     * Returns the string from the plugin's resource bundle,
     * or 'key' if not found.
     */
    public static String getResourceString(String key) {
        ResourceBundle bundle = NewsUIPlugin.getDefault().getResourceBundle();
        try {
            return (bundle != null) ? bundle.getString(key) : key;
        } catch (MissingResourceException e) {
            return key;
        }
    }

    /**
     * Returns the plugin's resource bundle,
     */
    public ResourceBundle getResourceBundle() {
        return resourceBundle;
    }

    private void refreshProxySet() {
        IPreferenceStore store = getPreferenceStore();
        Properties prop = System.getProperties();
        if (store.getBoolean(NewsUI.PREF_PROXY_USE)) {
            prop.setProperty(NewsUI.SYSPROP_PROXY_SET, "true");
            prop.setProperty(
                NewsUI.SYSPROP_PROXY_HOST, store.getString(NewsUI.PREF_PROXY_HOST)
            );
            prop.setProperty(
                NewsUI.SYSPROP_PROXY_PORT, store.getString(NewsUI.PREF_PROXY_PORT)
            );
        } else {
            prop.remove(NewsUI.SYSPROP_PROXY_SET);
            prop.remove(NewsUI.SYSPROP_PROXY_HOST);
            prop.remove(NewsUI.SYSPROP_PROXY_PORT);
        }
    }

    protected void initializeDefaultPreferences(IPreferenceStore store) {
        store.setDefault(NewsUI.PREF_PROXY_USE, false);
        store.setDefault(NewsUI.PREF_PROXY_HOST, "");
        store.setDefault(NewsUI.PREF_PROXY_PORT, "");

        store.addPropertyChangeListener(
            new IPropertyChangeListener() {
                public void propertyChange(PropertyChangeEvent event) {
                    refreshProxySet();
                }
            }
        );
    }

    protected FontRegistry initFontRegistry() {
        FontRegistry ret = JFaceResources.getFontRegistry();

        Display display = Display.getCurrent();
        Font sysfont = display.getSystemFont();
        FontData[] sysfontdata = sysfont.getFontData();
        ret.put(NewsUI.FONT_SYSTEM_NORMAL, sysfontdata);
        
        FontData[] sysfontbolddata = sysfont.getFontData();
        for (int i = 0; i < sysfontbolddata.length; i++) {
            sysfontbolddata[i].setStyle(SWT.BOLD);
        }
        ret.put(NewsUI.FONT_SYSTEM_BOLD, sysfontbolddata);

        return ret;
    }
    public static FontRegistry getFontRegistry() {
        return JFaceResources.getFontRegistry();
    }
    public static Font getFont(String symbolicName) {
        return JFaceResources.getFont(symbolicName);
    }

    public static FavoriteList getFavoriteList() {
        return NewsUIPlugin.getDefault().favoriteList;
    }

    public static IEditorPart openEditor(IEditorInput input, String editorId) {
        try {
            IWorkbenchPage page =
                PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
            return page.openEditor(input, editorId);
        } catch (PartInitException e) {
            throw new IllegalStateException(e.getMessage());
        }
    }
}
