/*
 * Created on 2004/09/25
 *
 * @author KAMO Masahiko (mkamo@users.sourceforge.jp)
 */
package razgriz.news.ui.views;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.ViewPart;

import razgriz.news.model.Article;
import razgriz.news.model.News;
import razgriz.news.model.NewsPackage;
import razgriz.news.model.util.NewsSwitch;
import razgriz.news.ui.NewsUI;
import razgriz.news.ui.NewsUIPlugin;
import razgriz.news.ui.actions.ClipArticleAction;
import razgriz.news.ui.browser.NewsBrowser;
import razgriz.news.ui.browser.NewsBrowserInput;

/**
 * 
 */
public class ArticleListView extends ViewPart {
    public static final String ID = ArticleListView.class.getName();

    private static final String KEY_NEWS_URI = "newsUri";

    private static final DateFormat DF_ARTICLE_DATE =
        new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
    
    private News news;
    
    private TableViewer viewer;

    private Action clipArticleAction;

    public ArticleListView() {
        super();
    }

    public ArticleListView getThis() {
        return this;
    }

    public static String createSecondaryId(News news) {
        return news.getLink().replace(':', '!');
    }

    public void init(IViewSite site, IMemento memento) throws PartInitException {
        super.init(site, memento);
        if (memento == null) {
            return;
        }

        Resource res = NewsUIPlugin.getFavoriteList().eResource();
        String uri = memento.getString(KEY_NEWS_URI);
        if (!StringUtils.isEmpty(uri)) {
            news = (News) res.getEObject(uri);
        }
    }

    public void saveState(IMemento memento) {
        super.saveState(memento);

        if (news != null) {
            Resource res = NewsUIPlugin.getFavoriteList().eResource();
            memento.putString(KEY_NEWS_URI, res.getURIFragment(news));
        }
    }

    public void createPartControl(Composite parent) {
        viewer = createArticleTable(parent);

        makeActions();
        contributeToActionBars();
        hookDoubleClickAction();

        setInput(news);
    }

    private TableViewer createArticleTable(Composite parent) {
        TableViewer ret = new TableViewer(
            parent, SWT.MULTI | SWT.FULL_SELECTION
        );
        ret.setContentProvider(new ArticleListContentProvider());
        ret.setLabelProvider(new ArticleListLabelProvider());

        Table table = ret.getTable();
        table.setFont(
            JFaceResources.getFont(NewsUI.FONT_SYSTEM_BOLD)
        );
        
        TableColumn nameCol = new TableColumn(table, SWT.LEFT);
        nameCol.setText(NewsUIPlugin.getResourceString("title.title"));

        TableColumn dateCol = new TableColumn(table, SWT.LEFT);
        dateCol.setText(NewsUIPlugin.getResourceString("title.date"));

        TableColumn descCol = new TableColumn(table, SWT.LEFT);
        descCol.setText(NewsUIPlugin.getResourceString("title.description"));

        table.setHeaderVisible(true);

        ret.setSorter(new ArticleSorter());
        return ret;
    }

    private void makeActions() {
        clipArticleAction = new ClipArticleAction(
            getSite(),
            new ClipArticleAction.NewsInput() {
                public News getNews() {
                    return news;
                }
            }
        );
    }

    private void contributeToActionBars() {
        IActionBars bars = getViewSite().getActionBars();
        fillLocalPullDown(bars.getMenuManager());
        fillLocalToolBar(bars.getToolBarManager());
    }

    private void fillLocalPullDown(IMenuManager manager) {
        manager.add(clipArticleAction);
    }
    private void fillLocalToolBar(IToolBarManager manager) {
        manager.add(clipArticleAction);
    }

    private void hookDoubleClickAction() {
        viewer.addDoubleClickListener(new IDoubleClickListener() {
            public void doubleClick(DoubleClickEvent event) {
                IStructuredSelection sel = 
                    (IStructuredSelection) event.getSelection();
                for (Iterator ite = sel.iterator(); ite.hasNext(); ) {
                    Object obj = ite.next();
                    if (!(obj instanceof Article)) {
                        continue;
                    }
                    Article article = (Article) obj;
                    try {
                        article.setRead(true);
                        NewsBrowser editor =
                            (NewsBrowser) NewsUIPlugin.openEditor(
                                new NewsBrowserInput(article),
                                NewsBrowser.ID
                            );
                        editor.activateBrowserPage();
                    } catch(Exception e) {
                        throw new IllegalArgumentException(e.getMessage()); // todo:
                    }
                }
            }
        });
    }

    public void setFocus() {
        viewer.getControl().setFocus();
    }

    public void updateTableAppearance() {
        Table table = viewer.getTable();
        if (news != null) {
            TableItem[] items = table.getItems();
            for (int i = 0; i < items.length; i++) {
                TableItem item = items[i];
                Article article = (Article) item.getData();
                String fontsym = article.isRead()?
                    NewsUI.FONT_SYSTEM_NORMAL:
                        NewsUI.FONT_SYSTEM_BOLD;
                item.setFont(NewsUIPlugin.getFont(fontsym));
            }

            TableColumn[] cols = table.getColumns();
            for (int i = 0; i < cols.length; i++) {
                cols[i].pack();
            }
        }
    }

    public void setInput(News news) {
        this.news = news;
        viewer.setInput(news);
        setPartName(
            news == null? "": news.getTitle()
        );
        updateTableAppearance();
    }

    private class ArticleListContentProvider
        implements IStructuredContentProvider, Adapter {

        TableViewer viewer;
        News news;

        Set adaptedSet;
        public ArticleListContentProvider() {
            adaptedSet = new HashSet();
        }

        public void addAdaption(EObject adapted) {
            adapted.eAdapters().add(this);
            adaptedSet.add(adapted);
        }
        public void removeAdaption(EObject adapted) {
            adaptedSet.remove(adapted);
            adapted.eAdapters().remove(this);
        }
        public void clearAdaption() {
            for (Iterator ite = adaptedSet.iterator(); ite.hasNext(); ) {
                EObject adapted = (EObject) ite.next();
                adapted.eAdapters().remove(this);
                ite.remove();
            }
        }

        public void inputChanged(
            Viewer viewer, Object oldInput, Object newInput
        ) {
            this.viewer = (TableViewer) viewer;
            News newNews = (News) newInput;

            clearAdaption();

            if (newNews != null) {
                addAdaption(newNews);
                List newartlist = newNews.getArticles();
                for (int i = 0; i < newartlist.size(); i++) {
                    Article article = (Article) newartlist.get(i);
                    addAdaption(article);
                }
            }

            news = newNews;
        }
        public void dispose() {
            clearAdaption();
        }

        public Object[] getElements(Object parent) {
            if (parent == null) {
                return new Object[0];
            }
            News n = (News) parent;
            List articlelist = n.getArticles();
            return articlelist.toArray(new Article[articlelist.size()]);
        }
        public void notifyChanged(Notification notification) {
            if (notification.isTouch()) {
                return;
            }

            final Object obj = notification.getNotifier();
            final NewsSwitch sw = new ArticleListUpdateSwitch(this, notification);
//            if (viewer != null && !viewer.getControl().isDisposed()) {
            viewer.getControl().getDisplay().asyncExec(
                new Runnable() {
                    public void run() {
                        sw.doSwitch((EObject) obj);
                    }
                }
            );
//            }
        }
        public Notifier getTarget() {
            return null;
        }
        public void setTarget(Notifier newTarget) {
        }
        public boolean isAdapterForType(Object type) {
            return false;
        }
    }

    private class ArticleListUpdateSwitch extends NewsSwitch {
        private ArticleListContentProvider provider;
        private Notification notification;
        public ArticleListUpdateSwitch(
            ArticleListContentProvider provider, Notification notification
        ) {
            this.provider = provider;
            this.notification = notification;
        }
        public Object caseNews(News news) {
            switch (notification.getEventType()) {
            case Notification.ADD: {
                EObject eobj = (EObject) notification.getNewValue();
                provider.addAdaption(eobj);
                viewer.refresh();
                updateTableAppearance();
                break;
            }
            case Notification.ADD_MANY: {
                List eobjlist = (List) notification.getNewValue();
                for (int i = 0; i < eobjlist.size(); i++) {
                    EObject eobj = (EObject) eobjlist.get(i);
                    provider.addAdaption(eobj);
                }
                viewer.refresh();
                updateTableAppearance();
                break;
            }
            case Notification.REMOVE: {
                EObject eobj = (EObject) notification.getOldValue();
                provider.removeAdaption(eobj);
                viewer.refresh();
                updateTableAppearance();
                break;
            }
            case Notification.REMOVE_MANY: {
                List eobjlist = (List) notification.getOldValue();
                for (int i = 0; i < eobjlist.size(); i++) {
                    EObject eobj = (EObject) eobjlist.get(i);
                    provider.removeAdaption(eobj);
                }
                viewer.refresh();
                updateTableAppearance();
                break;
            }
            case Notification.SET: {
                int fid = notification.getFeatureID(News.class);
                if (fid == NewsPackage.NEWS__TITLE)  {
                    setPartName(notification.getNewStringValue());
                } else if (
                    fid == NewsPackage.NEWS__GROUP ||
                    fid == NewsPackage.NEWS__FAVORITE_LIST
                ) {
                    if (notification.getNewValue() == null) {
                        provider.clearAdaption();
                        IWorkbenchPage page = PlatformUI.getWorkbench().
                            getActiveWorkbenchWindow().getActivePage();
                        page.hideView(getThis());
                    }
                }
                break;
            }
            }
            return news;
        }
        public Object caseArticle(Article article) {
            switch (notification.getEventType()) {
            case Notification.SET:
                updateTableAppearance();
            }
            return article;
        }
    }

    private class ArticleListLabelProvider extends LabelProvider
        implements ITableLabelProvider {

        public String getColumnText(Object obj, int index) {
            Article article = (Article) obj;
            switch (index) {
            case 0:
                return StringUtils.defaultString(article.getTitle());
            case 1:
                Date d = article.getDate();
                return d == null? "": StringUtils.defaultString(
                    DF_ARTICLE_DATE.format(d)
                );
            case 2:
                return StringUtils.defaultString(article.getDescription());
            }
            return null;
        }

        public Image getColumnImage(Object obj, int index) {
            return null;
        }
    }

    private class ArticleSorter extends ViewerSorter {
        public int compare(Viewer viewer, Object e1, Object e2) {
            Article a1 = (Article) e1;
            Article a2 = (Article) e2;
            if (a1.getDate() == null || a2.getDate() == null) {
                return 0;
            }
            return a2.getDate().compareTo(a1.getDate());
        }
}
}
