/*
 * Created on 2005/02/20
 *
 * @author KAMO Masahiko (mkamo@users.sourceforge.jp)
 */
package razgriz.news.ui.dialogs;

import java.beans.PropertyDescriptor;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;
import java.util.Map.Entry;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

/**
 * 
 */
public class BeanEditorDialog extends TitleAreaDialog {
    public static int OK = TitleAreaDialog.OK;
    public static int CANCEL = TitleAreaDialog.CANCEL;

    private String title;
    private String message;

    private Object bean;
    private PropertyValidator validator;
    private PropertyDecorator decorator;
    private PropertySorter sorter;

    private Map propDescToFieldMap;

    public BeanEditorDialog(
        Shell parentShell,
        String title,
        String message,
        Object bean,
        PropertyValidator validator,
        PropertyDecorator decorator,
        PropertySorter sorter
    ) {
        super(parentShell);

        this.title = title;
        this.message = message;

        this.bean = bean;
        this.validator = validator;
        this.decorator = decorator;
        this.sorter = sorter;
        
        propDescToFieldMap = new HashMap();
    }

    private void validate(Map propMap) {
        String errmsg = null;
        if (validator != null) {
            errmsg = validator.validate(propMap);
        }
        setErrorMessage(errmsg);
        getButton(IDialogConstants.OK_ID).setEnabled(errmsg == null);
    }
    
    private Map createPropertyNameToValueMap() {
        Map ret = new TreeMap();

        for (
            Iterator ite = propDescToFieldMap.entrySet().iterator();
            ite.hasNext();
        ) {
            try {
                Entry entry = (Entry) ite.next();
                PropertyDescriptor propdesc = (PropertyDescriptor) entry.getKey();
                Field field = (Field) entry.getValue();
                ret.put(propdesc.getName(), field.getValue());
            } catch (Exception e) {
                throw new IllegalArgumentException(e.getMessage()); // todo:
            }
        }
        
        return ret;
    }

    private void updateBeanProperties() {
        for (
            Iterator ite = propDescToFieldMap.entrySet().iterator();
            ite.hasNext();
        ) {
            try {
                Entry entry = (Entry) ite.next();
                PropertyDescriptor propdesc = (PropertyDescriptor) entry.getKey();
                Field field = (Field) entry.getValue();
                PropertyUtils.setSimpleProperty(
                    bean, propdesc.getName(), field.getValue()
                );
            } catch (Exception e) {
                throw new IllegalArgumentException(e.getMessage()); // todo:
            }
        }
    }

    protected Control createContents(Composite parent) {
        Control ret = super.createContents(parent);
        validate(createPropertyNameToValueMap());
        return ret;
    }

    protected void configureShell(Shell newShell) {
        super.configureShell(newShell);
        newShell.setText(title);
    }

    protected Control createDialogArea(Composite parent) {
        setTitle(title);
        setMessage(message);
        
        Composite component = (Composite) super.createDialogArea(parent);
        Composite container = new Composite(component, SWT.NONE);
        container.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        GridLayout layout = new GridLayout();
        layout.numColumns = 2;
        layout.verticalSpacing = 9;
        container.setLayout(layout);

        PropertyDescriptor[] propdescs =
            PropertyUtils.getPropertyDescriptors(bean);

        if (sorter != null) {
            Arrays.sort(
                propdescs,
                new Comparator() {
                    public int compare(Object o1, Object o2) {
                        PropertyDescriptor pd1 = (PropertyDescriptor) o1;
                        PropertyDescriptor pd2 = (PropertyDescriptor) o2;
                        return sorter.compare(pd1.getName(), pd2.getName());
                    }
                }
            );
        }

        for (int i = 0; i < propdescs.length; i++) {
            PropertyDescriptor propdesc = propdescs[i];
            if (PropertyUtils.isWriteable(bean, propdesc.getName())) {
                if (propdesc.getPropertyType().equals(String.class)) {
                    Field field = addStringField(container, propdesc);
                    propDescToFieldMap.put(propdesc, field);
                }
            }
        }
        
        return container;
    }

    protected Field addStringField(
        Composite container, PropertyDescriptor propDesc
    ) {
        try {
            Label label= new Label(container, SWT.NONE);
            String labelText = null;
            if (decorator != null) {
                labelText = decorator.decorateLabelText(propDesc.getName());
            }
            labelText = StringUtils.isBlank(labelText)?
                "&" + StringUtils.capitalize(propDesc.getName()) + ":":
                    labelText;
            label.setText(labelText);

            final Text text = new Text(container, SWT.BORDER | SWT.SINGLE);
            text.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
            String initValue = StringUtils.defaultString(
                (String) PropertyUtils.getProperty(bean, propDesc.getName())
            );
            text.setText(initValue);

            text.addModifyListener(new ModifyListener() {
                public void modifyText(ModifyEvent e) {
                    validate(createPropertyNameToValueMap());
                }
            });

            return new Field() {
                public Object getValue() {
                    return text.getText();
                }
            };
        } catch(Exception e) {
            throw new IllegalArgumentException(e.getMessage()); // todo:
        }
    }

    protected void okPressed() {
        updateBeanProperties();
        super.okPressed();
    }

    public static interface PropertyValidator {
        public String validate(Map propNameToValueMap);
    }
    public static interface PropertyDecorator {
        public String decorateLabelText(String propName);
    }
    public static interface PropertySorter {
        public int compare(String propName1, String propName2);
    }

    protected static interface Field {
        public Object getValue();
    }
}
