<?php
/**
 * Research Artisan Lite: Website Access Analyzer
 * Copyright (C) 2009 Research Artisan Project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * @copyright Copyright (C) 2009 Research Artisan Project
 * @license GNU General Public License (see license.txt)
 * @author ossi
 */

/**
 * RA View
 */
class RaView {
/* ------------------------------------------------------------------------ */
 
/* -- Define -- */
  /**
   * LAYOUT_DIR
   * @var string
   */
  const LAYOUT_DIR = 'layout';
/* ------------------------------------------------------------------------ */

/* -- Private Property -- */
  /**
   * request
   * @var RaRequest
   */
  private $_request = null;
  /**
   * session
   * @var RaSession
   */
  private $_session = null;
  /**
   * message
   * @var RaMessage
   */
  private $_message = null;
  /**
   * result
   * @var RaResult
   */
  private $_result = null;
  /**
   * charset
   * @var string
   */
  private $_charset = null;
/* ------------------------------------------------------------------------ */

/* -- Public Method -- */
  /**
   * Constructer
   * @param RaRequest $request Request
   * @param RaSession $session Session
   * @param RaMessage $message Message
   * @param RaResult $result Result
   */
  public function __construct(RaRequest $request, RaSession $session, RaMessage $message, RaResult $result=null) {
    $this->_request = $request;
    $this->_session = $session;
    $this->_message = $message;
    $this->_result = $result;
    if (!is_null($result))
      $this->_charset = !is_null($result->getCharset()) ? $result->getCharset() : RaConfig::CHARSET;
  }
  
  /**
   * showView
   * @param string controller
   * @param string action
   */
  public function showView($controller, $action) {
    $redirect = $this->_result->getRedirect();
    if (!is_null($redirect)) {
      $this->_redirect($redirect);
    } else {
      $helper = $this->_loadHelper($this->_request, $this->_session, $this->_message, $this->_result, $controller, $action);
      $this->_loadView($controller, $action, $helper, $this->_charset);
    }
  }

  /**
   * showError
   * @param RaException exception
   */
  public function showError($exception) {
    header('Content-Type: text/html; charset='. RaConfig::CHARSET);
    require RA_CORE_DIR. RA_SYSTEM_DIR. 'RaError.php';
  }
/* ------------------------------------------------------------------------ */

/* -- Private Method -- */
  /**
   * loadHelper
   * @param RaRequest $request Request
   * @param RaSession $session Session
   * @param RaMessage $message Message
   * @param RaResult $result Result
   * @param string controller
   * @param string action
   * @return RaHelper $helper
   */
  private function _loadHelper(RaRequest $request, RaSession $session, RaMessage $message, RaResult $result=null, $controller, $action) {
    $this->_checkHelper($controller);
    $helperName = ucfirst($controller). 'Helper';
    return new $helperName($request, $session, $message, $result, $controller, $action);
  }

  /**
   * checkHelper
   * @param string controller
   */
  private function _checkHelper($controller) {
    $helperFile = RA_CORE_DIR. RA_HELPER_DIR. ucfirst($controller). 'Helper.php';
    if (file_exists($helperFile)) {
      require $helperFile;
    } else {
      throw new RaException(RaConfig::URL_NOTFOUND_ERR_MSG, RaConfig::URL_NOTFOUND_ERR_CODE);
    }
  }

  /**
   * loadView
   * @param string controller
   * @param string action
   * @param string helper
   * @param string charset
   */
  private function _loadView($controller, $action, $helper, $charset) {
    $viewDir = RA_CORE_DIR. RA_VIEW_DIR;
    $viewFile = $this->_checkView($viewDir, $controller, $action, array('.html'));
    if (!is_null($viewFile)) {
      $this->_loadHtml($viewDir, $viewFile, $controller, $action, $helper, $charset);
      return;
    }
    $viewFile = $this->_checkView($viewDir, $controller, $action, array('.txt', '.csv'));
    if (!is_null($viewFile)) {
      $this->_loadText($viewDir, $viewFile, $controller, $action, $helper, $charset);
      return;
    }
    throw new RaException(RaConfig::URL_NOTFOUND_ERR_MSG, RaConfig::URL_NOTFOUND_ERR_CODE);
  }

  /**
   * loadHtml
   * @param string viewDir
   * @param string viewFile
   * @param string controller
   * @param string action
   * @param string helper
   * @param string charset
   */
  private function _loadHtml($viewDir, $viewFile, $controller, $action, $helper, $charset) {
    ob_start();
    header('Content-Type: text/html; charset='. $charset);
    $layoutFiles = glob($viewDir. self::LAYOUT_DIR. DIRECTORY_SEPARATOR. '*.html');
    if (count($layoutFiles) > 0) {
      foreach($layoutFiles as $layoutFile) {
        require $layoutFile;
        break;
      }
    } else {
      require $viewFile;
    }
    ob_end_flush();
  }

  /**
   * loadText
   * @param string viewDir
   * @param string viewFile
   * @param string controller
   * @param string action
   * @param string helper
   * @param string charset
   */
  private function _loadText($viewDir, $viewFile, $controller, $action, $helper, $charset) {
    ob_start();
    $convertCharsets = array('Shift_JIS' => 'SJIS', 'EUC-JP' => 'EUC-JP', 'UTF-8' => 'UTF-8');
    $convertCharset = isset($convertCharsets[$charset]) ? $convertCharsets[$charset] : 'auto';
    $text = file_get_contents($viewFile);
    if ($text === false) $text = '';
    header('Content-Type: text/plain; charset='. $charset);
    header('Content-Disposition: attachment; filename='. basename($viewFile));
    $text .= $helper->getTextData();
    print mb_convert_encoding($text, $convertCharset, RaConfig::CHARSET);
    ob_end_flush();
  }

  /**
   * checkView
   * @param string viewDir
   * @param string controller
   * @param string action
   * @param string exts
   */
  private function _checkView($viewDir, $controller, $action, $exts) {
    $rtn = null;
    $viewFile = $viewDir. $controller. DIRECTORY_SEPARATOR. $action;
    foreach($exts as $ext) {
      if (file_exists($viewFile. $ext)) {
        $rtn = $viewFile. $ext;
        break;
      }
    }
    return $rtn;
  }

  /**
   * redirect
   * @param string redirect
   */
  private function _redirect($redirect) {
    header('location: '. $redirect);
  }
/* ------------------------------------------------------------------------ */

}
?>
