<?php
/**
 * Research Artisan Lite: Website Access Analyzer
 * Copyright (C) 2009 Research Artisan Project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * @copyright Copyright (C) 2009 Research Artisan Project
 * @license GNU General Public License (see license.txt)
 * @author ossi
 */
class InstallController extends BaseController {

  const DEFAULT_DATABASE_HOSTNAME = 'localhost';
  const DEFAULT_TABLE_PREFIX = 'ra_';

  public function __construct(RaRequest $request, RaSession $session, RaMessage $message, RaResult $result, $controller, $action) {
    parent::__construct($request, $session, $message, $result, $controller, $action);
  }

  public function step0() {
    $result = $this->result;
    $this->session->destroy();
    if ($this->_isInstalled()) {
      $result->setNextController('login');
      $result->setNextAction('login');
    } else {
      try {
        $this->checkPhpVersion();
        $this->checkPhpExtensions();
      } catch (Exception $exception) {
        $this->session->set('nextStop', true);
        throw $exception;
      }
    }
    return $result;
  }

  public function step1() {
    $result = $this->result;

    if (!$this->checkPost()) {
      $result->setNextController('install');
      $result->setNextAction('step0');
      return $result;
    }
    if (!$this->checkOwnPost('hostname')) {
      if (!File::deleteFile(SETTING_DATABASE_FILE)) $this->message->setMessage('データベース接続設定の削除に失敗しました。<br />'. '「setting」ディレクトリのアクセス権（パーミッション）を確認してください。<br />'. SETTING_DIR);
      return $result;
    }

    $hostname = $this->request->get('hostname');
    $username = $this->request->get('username');
    $pswd = $this->request->get('pswd');
    $dbname = $this->request->get('dbname');
    $tableprefix = $this->request->get('tableprefix');
    if (trim($hostname) == '') {
      $this->message->setMessage('ホスト名を入力してください。');
      return $result;
    } 
    if (trim($username) == '') {
      $this->message->setMessage('ユーザー名を入力してください。');
      return $result;
    } 
    if (trim($dbname) == '') {
      $this->message->setMessage('データベース名を入力してください。');
      return $result;
    } 
    if (trim($tableprefix) == '') {
      $this->message->setMessage('テーブルプレフィックスを入力してください。');
      return $result;
    } 

    if ($this->_defineDatabase($hostname, $username, $pswd, $dbname, $tableprefix)) {
      try {
        if (!RaModel::isDatabaseDefined()) require SETTING_DATABASE_FILE;
        $site = new Site();
        $site->checkMySQLVersion();
      } catch (Exception $exception) {
        switch ($exception->getCode()) {
          case RaConfig::MYSQL_NOTSUPPORT_VERSION_ERR_CODE:
            $message = 'PHPの環境設定に問題があります。<br />';
            $message .= '次のモジュールがインストールされていません。<br />';
            $message .= $exception->getMessage(). '<br />';
            $this->message->setMessage($message);
            break;
          default:
            $this->message->setMessage('入力内容に誤りがある為、データベースに接続できません。');
            break;
        }
        if (!File::deleteFile(SETTING_DATABASE_FILE)) $this->message->setMessage('データベース接続設定の削除に失敗しました。<br />'. '「setting」ディレクトリのアクセス権（パーミッション）を確認してください。<br />'. SETTING_DIR);
        return $result;
      }
      $result->setNextController('install');
      $result->setNextAction('step2');
    } else {
      $this->message->setMessage('データベース接続設定に失敗しました。<br />'. '「setting」ディレクトリのアクセス権（パーミッション）を確認してください。<br />'. SETTING_DIR);
      return $result;
    }

    return $result;
  }

  public function step2() {
    $result = $this->result;
    if ($this->_isInstalled()) {
      $result->setNextController('login');
      $result->setNextAction('login');
    }
    if (!$this->checkPost()) {
      $result->setNextController('install');
      $result->setNextAction('step0');
      return $result;
    }
    if ($this->checkRedirect()) {
      return $result;
    }

    $sitename = $this->request->get('sitename');
    $url = $this->request->get('url');
    $pswd = $this->request->get('pswd');
    $pswdConfirm = $this->request->get('pswd_confirm');
    if (trim($sitename) == '') {
      $this->message->setMessage('サイト名を入力してください。');
      return $result;
    } 
    if (mb_strlen($sitename) > 100) {
      $this->message->setMessage('サイト名は100文字以内で入力してください。');
      return $result;
    } 
    if (trim($url) == '') {
      $this->message->setMessage('URLを入力してください。');
      return $result;
    } 
    if (!Track::checkUrl($url)) {
      $this->message->setMessage('URLは「http://、https://〜」の形で正しく入力してください。');
      return $result;
    } 
    if (mb_strlen($url) > 100) {
      $this->message->setMessage('URLは100文字以内で入力してください。');
      return $result;
    } 
    if (trim($pswd) == '') {
      $this->message->setMessage('パスワードを入力してください。');
      return $result;
    } 
    if ($pswd != $pswdConfirm) {
      $this->message->setMessage('パスワードの入力を確認してください。');
      return $result;
    } 
    if (mb_strlen($pswd) > 100) {
      $this->message->setMessage('パスワードは100文字以内で入力してください。');
      return $result;
    } 

    if (!$this->defineSiteurl($url)) {
      $this->message->setMessage('サイト設定に失敗しました。<br />'. '「setting」ディレクトリのアクセス権（パーミッション）を確認してください。<br />'. SETTING_DIR);
      return $result;
    }

    $site = new Site();
    $site->setValue('sitename', $sitename);
    $site->setValue('url', $url);
    $site->setValue('pswd', md5($pswd));
    $site->save();

    $version = File::replaceCrlf(File::readFile(RA_CORE_DIR. 'version.txt'), '');
    $system = new System();
    $system->setValue('version', $version);
    $system->save();

    $alias = new Alias();
    $domain = new Domain();
    $domain->loadSqlData();
    $title = new Title();
    $log = new Log();

    $result->setNextController('install');
    $result->setNextAction('step3');

    return $result;
  }

  public function step3() {
    $result = $this->result;
    if (!$this->checkPost()) {
      $result->setNextController('install');
      $result->setNextAction('step0');
      return $result;
    }
    if (!$this->checkRedirect()) {
      $result->setNextController('install');
      $result->setNextAction('step0');
      return $result;
    }
    if (!File::writeFile(SETTING_INSTALL_COMPLETE_FILE, Config::ON)) {
      throw new RaException('インストール設定に失敗しました。<br />'. '「setting」ディレクトリのアクセス権（パーミッション）を確認してください。<br />'. SETTING_DIR);
    }
    return $result;
  }

  public function step4() {
    $result = $this->result;
    if (!$this->checkPost()) {
      $result->setNextController('install');
      $result->setNextAction('step0');
      return $result;
    }
    return $result;
  }

  private function _isInstalled() {
    try {
      if (!file_exists(SETTING_DATABASE_FILE)) return false;
      if (!RaModel::isDatabaseDefined()) 
        throw new RaException(RaConfig::DATABASE_CONFIG_UNDEFINED_ERR_MSG, RaConfig::DATABASE_CONFIG_UNDEFINED_ERR_CODE);
      if (!file_exists(SETTING_INSTALL_COMPLETE_FILE)) return false;
      $system = new System();
    } catch (Exception $exception) {
      if (!File::deleteFile(SETTING_DATABASE_FILE)) $this->message->setMessage('データベース接続設定の削除に失敗しました。<br />'. '「setting」ディレクトリのアクセス権（パーミッション）を確認してください。<br />'. SETTING_DIR);
      return false;
    }
    return true;
  }

  private function _defineDatabase($hostname, $username, $pswd, $dbname, $tableprefix) {
    $define = '';
    $define .= '<?php'. "\r\n";
    $define .= 'define(\''. RaConfig::DATABASE_DEFINE_HOST. '\',\''. $hostname. '\');'. "\r\n";
    $define .= 'define(\''. RaConfig::DATABASE_DEFINE_USER. '\',\''. $username. '\');'. "\r\n";
    $define .= 'define(\''. RaConfig::DATABASE_DEFINE_PASS. '\',\''. $pswd. '\');'. "\r\n";
    $define .= 'define(\''. RaConfig::DATABASE_DEFINE_DB_NAME. '\',\''. $dbname. '\');'. "\r\n";
    $define .= 'define(\''. RaConfig::DATABASE_DEFINE_TABLE_PREFIX. '\',\''. $tableprefix. '\');'. "\r\n";
    $define .= '?>';
    return File::writeFile(SETTING_DATABASE_FILE, $define);
  }

}
?>
