/* OggVorbisReader.c */
/* 2009/07/23        */

#include "StdAfx.h"

#include "OggVorbisReader.h"

/* */

BOOL QO_OggVorbisReader_Init(
	QO_OggVorbisReader_t* t,
	QO_MemoryPool_t*      pool,
	QO_Allocator_t*       allocator,
	QO_StreamReader_t*    reader)
{
	memset(t, 0, sizeof(QO_OggVorbisReader_t));

	if (!QO_OggReader_Init(&(t->Reader), pool, allocator, reader)) {
		return FALSE;
	}

	t->StreamSize = reader->Size(reader->Context);

	t->Position     = 0;
	t->SeekPosition = 0;
	t->Duration     = -1;

	t->Setup = QV_CreateDecoderSetup();
	if (t->Setup == NULL) {
		return FALSE;
	}

	t->Decoder = QV_CreateDecoder();
	if (t->Decoder == NULL) {
		return FALSE;
	}

	t->EndOfStream    = FALSE;
	t->PendingSamples = 0;
	t->SampleBuffer   = NULL;
	t->SampleUnit     = 0;
	t->PacketBuffer   = NULL;
	t->PacketIndex    = 0;

	return TRUE;
}

void QO_OggVorbisReader_Release(
	QO_OggVorbisReader_t* t)
{
	QV_ReleaseDecoder(t->Decoder);

	QV_ReleaseDecoderSetup(t->Setup);
}

/* */

#define HT_BOS 0x02
#define HT_EOS 0x04

/* */

#define ID_SIZE 0x20

BOOL QO_OggVorbisReader_Setup(
	QO_OggVorbisReader_t* t)
{
	INT32 packet = 0;

	UINT8 id[ID_SIZE];
	SIZE_T isize = 0;

	UINT8* setup = NULL;
	SIZE_T ssize = 0;

	QO_OggReader_t* reader = &(t->Reader);

	QO_OggHeader_t header;

	if (!QO_OggReader_ReadHeader(reader, &header)) {
		return FALSE;
	}

	if ((header.HeaderType & HT_BOS) == 0) {
		return FALSE;
	}

	t->StreamSerialNumber = header.StreamSerialNumber;
	t->PageSequenceNumber = header.PageSequenceNumber;

	if (!QO_OggReader_ReadPayload(reader)) {
		return FALSE;
	}

	if (reader->Packets != 1 || reader->Discontinue) {
		return FALSE;
	}

	if (reader->PacketsSize[0] > ID_SIZE) {
		return FALSE;
	}

	memcpy(id, reader->PacketsBuffer, reader->PacketsSize[0]);
	isize = reader->PacketsSize[0];

	packet++;

	while (setup == NULL) {
		if (!QO_OggReader_ReadHeader(reader, &header)) {
			return FALSE;
		}

		if (header.StreamSerialNumber != t->StreamSerialNumber) {
			if (!QO_OggReader_SkipPayload(reader)) {
				return FALSE;
			}

			continue;
		}

		if (header.PageSequenceNumber != t->PageSequenceNumber + 1) {
			return FALSE;
		}

		t->PageSequenceNumber = header.PageSequenceNumber;

		if (!QO_OggReader_ReadPayload(reader)) {
			return FALSE;
		}

		if (reader->Packets > 0) {
			UINT8* pb = reader->PacketsBuffer;

			INT32 i;
			for (i = 0; i < reader->Packets; i++) {
				if (packet == 2) {
					setup = pb;
					ssize = reader->PacketsSize[i];
				}
				packet++;
				pb += reader->PacketsSize[i];
			}
		}
	}

	if (setup == NULL || ssize == 0) {
		return FALSE;
	}

	if (!QV_SetupDecoderSetup(
		t->Setup,
		id,
		isize,
		setup,
		ssize)) {
		return FALSE;
	}

	if (!QV_SetupDecoder(
		t->Decoder,
		t->Setup,
		0x10000)) {
		return FALSE;
	}

	t->SampleUnit = QV_GetDecoderSetupChannels(t->Setup);

	return TRUE;
}

/* */

BOOL QO_OggVorbisReader_DetectSamples(
	QO_OggVorbisReader_t* t,
	INT64*                samples)
{
	QO_OggReader_t* reader = &(t->Reader);

	INT64 psz = reader->Size / 2;

	INT64 pos = t->StreamSize;
	if (pos >= psz) {
		pos -= psz;
	}
	pos &= ~(psz - 1);

	*samples = 0;

	if (!QO_OggReader_Seek(reader, pos)) {
		return FALSE;
	}

	for (; ; ) {
		QO_OggHeader_t header;

		if (!QO_OggReader_ReadHeader(reader, &header)) {
			return FALSE;
		}

		if (!QO_OggReader_SkipPayload(reader)) {
			return FALSE;
		}

		if (header.StreamSerialNumber == t->StreamSerialNumber &&
			(header.HeaderType & HT_EOS) != 0) {
			*samples = t->Duration = header.GranulePosition;
			break;
		}
	}

	return TRUE;
}

/* */

static BOOL ReadNextPage(
	QO_OggVorbisReader_t* t,
	QO_OggHeader_t*       header)
{
	QO_OggReader_t* reader = &(t->Reader);

	for (; ; ) {
		if (!QO_OggReader_ReadHeader(reader, header)) {
			return FALSE;
		}

		if (header->StreamSerialNumber == t->StreamSerialNumber) {
			if (!QO_OggReader_ReadPayload(reader)) {
				return FALSE;
			}
			break;
		}

		if (!QO_OggReader_SkipPayload(reader)) {
			return FALSE;
		}
	}

	return TRUE;
}

static BOOL GetPageSamples(
	QO_OggVorbisReader_t* t,
	INT32*                samples)
{
	QO_OggReader_t* reader = &(t->Reader);

	INT32 i, count = 0;

	UINT8* pb = reader->PacketsBuffer;

	*samples = 0;

	for (i = 0; i < reader->Packets; i++) {
		if (i > 0 || !reader->Discontinue) {
			INT32 s = 0;
			if (!QV_CheckDecoderChecker(t->Decoder, pb, reader->PacketsSize[i], &s)) {
				return FALSE;
			}

			count += s;
		}

		pb += reader->PacketsSize[i];
	}

	*samples = count;

	return TRUE;
}

/* */

BOOL QO_OggVorbisReader_QuerySampleRange(
	QO_OggVorbisReader_t* t,
	INT64                 pos,
	QO_OggVorbisRange_t*  range)
{
	QO_OggHeader_t header;

	QO_OggReader_t* reader = &(t->Reader);

	SIZE_T sz = reader->Size / 2;

	INT64 start = -1;
	INT64 end   = -1;

	range->Start = 0;
	range->End   = 0;

	QV_ResetDecoderChecker(t->Decoder);

	if (!QO_OggReader_Seek(reader, pos)) {
		return FALSE;
	}

	for (; ; ) {
		if (reader->Pointer >= sz) {
			break;
		}

		if (!ReadNextPage(t, &header)) {
			return FALSE;
		}

		if (reader->Packets > 0) {
			INT32 samples = 0;

			if (!reader->Discontinue && (reader->PacketsBuffer[0] & 1) != 0) { /* Skip Header Packet */
				continue;
			}

			if (!GetPageSamples(t, &samples)) {
				return FALSE;
			}

			if (samples > 0) {
				if (start < 0) {
					start = header.GranulePosition - samples;
				}

				end = header.GranulePosition;
			}
		}
	}

	range->Start = start;
	range->End   = end;

	return TRUE;
}

/* */

BOOL QO_OggVorbisReader_SeekPage(
	QO_OggVorbisReader_t* t,
	INT64                 pos)
{
	QO_OggHeader_t header;

	QO_OggReader_t* reader = &(t->Reader);

	QV_ResetDecoderChecker(t->Decoder);

	if (!QO_OggReader_Seek(reader, pos)) {
		return FALSE;
	}

	for (; ; ) {
		if (!ReadNextPage(t, &header)) {
			return FALSE;
		}

		if (reader->Packets > 0) {
			INT32 samples = 0;

			if (!reader->Discontinue && (reader->PacketsBuffer[0] & 1) != 0) { /* Skip Header Packet */
				continue;
			}

			if (!GetPageSamples(t, &samples)) {
				return FALSE;
			}

			if (samples > 0) {
				t->PageSequenceNumber = header.PageSequenceNumber;
				t->Position           = header.GranulePosition - samples;
				break;
			}
		}
	}

	return TRUE;
}

/* */

BOOL QO_OggVorbisReader_Seek(
	QO_OggVorbisReader_t* t,
	INT64                 sample)
{
	QO_OggReader_t* reader = &(t->Reader);

	INT64 pos = -1;

	INT64 page_size = reader->Size / 2;

	INT64 s = 0;
	INT64 e = t->StreamSize;

	INT64 last = -1;

	if (e >= page_size) {
		e -= page_size;
	}
	e &= ~(page_size - 1);

	while (s < e) {
		INT32 ss = (INT32)(s / page_size);
		INT32 ee = (INT32)(e / page_size);
		INT32 pp = ss + (ee - ss) / 2;

		INT64 p = pp * page_size;

		QO_OggVorbisRange_t r;
		if (!QO_OggVorbisReader_QuerySampleRange(t, p, &r)) {
			return FALSE;
		}

		if (sample >= r.Start && sample < r.End) {
			pos = p;
			break;
		}

		if (last == p) {
			pos = p;
			break;
		}

		if (sample < r.Start) {
			e = p;
		} else {
			s = p;
		}

		last = p;
	}

	if (pos < 0) {
		pos = s;
	}

	if (!QO_OggVorbisReader_SeekPage(t, pos)) {
		return FALSE;
	}

	QV_ResetDecoder(t->Decoder);

	t->EndOfStream = FALSE;

	t->SeekPosition = sample;

	t->PendingSamples = 0;
	t->SampleBuffer   = NULL;

	t->PacketBuffer = reader->PacketsBuffer;
	t->PacketIndex  = 0;

	return TRUE;
}

/* */

static BOOL Decode_Header(
	QO_OggVorbisReader_t* t);

static BOOL Decode_Packet(
	QO_OggVorbisReader_t* t);

/* */

BOOL QO_OggVorbisReader_Decode(
	QO_OggVorbisReader_t* t,
	VOID*                 buffer,
	INT32                 samples,
	INT32*                output)
{
	QO_OggReader_t* reader = &(t->Reader);

	INT32 count = 0;

	*output = 0;

	for (; ; ) {
		if (t->PendingSamples > 0) {
			count = samples;
			if (count > t->PendingSamples) {
				count = t->PendingSamples;
			}

			memcpy(
				buffer,
				t->SampleBuffer,
				count * t->SampleUnit * sizeof(INT16));

			t->SampleBuffer   += count * t->SampleUnit;
			t->PendingSamples -= count;

			t->Position += count;
			break;
		}

		if (t->PacketIndex >= reader->Packets) {
			if (t->EndOfStream) {
				break;
			}

			if (!Decode_Header(t)) {
				return FALSE;
			}
		}

		if (t->PacketIndex < reader->Packets) {
			if (!Decode_Packet(t)) {
				return FALSE;
			}
		}
	}

	*output = count;

	return TRUE;
}

/* */

static BOOL Decode_Header(
	QO_OggVorbisReader_t* t)
{
	QO_OggHeader_t header;

	QO_OggReader_t* reader = &(t->Reader);

	for (; ; ) {
		if (!ReadNextPage(t, &header)) {
			return FALSE;
		}

		if (header.PageSequenceNumber != t->PageSequenceNumber + 1) {
			return FALSE;
		}

		t->PageSequenceNumber = header.PageSequenceNumber;

		if (reader->Packets > 0) {
			break;
		}
	}

	if ((header.HeaderType & HT_EOS) != 0) {
		t->EndOfStream = TRUE;
	}

	t->PacketBuffer = reader->PacketsBuffer;
	t->PacketIndex  = 0;

	return TRUE;
}

static BOOL Decode_Packet(
	QO_OggVorbisReader_t* t)
{
	QO_OggReader_t* reader = &(t->Reader);

	if (t->PacketIndex > 0 || !reader->Discontinue) {
		QV_Output_t output = { 0 };

		if (!QV_DecodeFrame(
			t->Decoder,
			t->PacketBuffer,
			reader->PacketsSize[t->PacketIndex],
			&output)) {
			return FALSE;
		}

		if (output.Length > 0) {
			QV_Convert_t convert = { 0 };

			if (!QV_ConvertFrame(
				t->Decoder,
				&output,
				&convert)) {
				return FALSE;
			}

			t->PendingSamples = output.Length;
			t->SampleBuffer   = convert.Sample;

			if (t->Position < t->SeekPosition) {
				INT32 d = (INT32)(t->SeekPosition - t->Position);
				if (d > t->PendingSamples) {
					d = t->PendingSamples;
				}

				t->Position       += d;
				t->PendingSamples -= d;
				t->SampleBuffer   += d * t->SampleUnit;
			}

			if (t->Duration >= 0 &&
				t->Position + t->PendingSamples > t->Duration) {
				INT32 d = (INT32)(t->Position + t->PendingSamples - t->Duration);
				if (d > t->PendingSamples) {
					d = t->PendingSamples;
				}

				t->PendingSamples -= d;
			}
		}
	}

	t->PacketBuffer += reader->PacketsSize[t->PacketIndex];
	t->PacketIndex  += 1;

	return TRUE;
}

/* */

