// WaveWriter.h
// 2008/12/02

#pragma once

namespace QAX {

// MmError
class MmError : public RuntimeError {

	const char* m_Message;

public:

	MmError(const char* m) : m_Message(m)
	{
	}

	virtual ~MmError()
	{
	}

	virtual std::string ToString()
	{
		return std::string(m_Message);
	}

	static NO_RETURN void Throw(const char* m)
	{
		throw MmError(m);
	}

}; // MmError

/* */

// WaveWriter
class WaveWriter {

	HMMIO m_hMMIO;

	MMCKINFO m_ckRIFF;
	MMCKINFO m_ckData;

public:

	WaveWriter() : m_hMMIO(0)
	{
		ZeroMemory(&m_ckRIFF, sizeof(m_ckRIFF));
		ZeroMemory(&m_ckData, sizeof(m_ckData));
	}

	~WaveWriter()
	{
		Close();
	}

	void Create(
		LPCWSTR path,
		INT32   channels,
		INT32   samplingRate)
	{
		const FOURCC FCC_WAVE   = mmioFOURCC('W', 'A', 'V', 'E');
		const FOURCC FCC_FORAMT = mmioFOURCC('f', 'm', 't', ' ');
		const FOURCC FCC_DATA   = mmioFOURCC('d', 'a', 't', 'a');

		HMMIO hMMIO = 0;

		try {
			hMMIO = mmioOpenW(
				const_cast<LPWSTR>(path),
				0,
				MMIO_CREATE | MMIO_WRITE);
			if (hMMIO == 0) {
				MmError::Throw("mmioOpen");
			}

			/* */

			m_ckRIFF.fccType = FCC_WAVE;

			MMRESULT mRslt = mmioCreateChunk(
				hMMIO,
				&m_ckRIFF,
				MMIO_CREATERIFF);
			if (mRslt != MMSYSERR_NOERROR) {
				MmError::Throw("mmioCreateChunk.RIFF");
			}

			/* */

			MMCKINFO ckFormat;
			ZeroMemory(&ckFormat, sizeof(ckFormat));

			ckFormat.ckid = FCC_FORAMT;

			mRslt = mmioCreateChunk(
				hMMIO,
				&ckFormat,
				0);
			if (mRslt != MMSYSERR_NOERROR) {
				MmError::Throw("mmioCreateChunk.fmt_");
			}

			WAVEFORMATEX wfex;
			ZeroMemory(&wfex, sizeof(wfex));

			wfex.wFormatTag      = WAVE_FORMAT_PCM;
			wfex.nChannels       = channels;
			wfex.nSamplesPerSec  = samplingRate;
			wfex.nAvgBytesPerSec = 2 * channels * samplingRate;
			wfex.nBlockAlign     = 2 * channels;
			wfex.wBitsPerSample  = 16;

			LONG sz = mmioWrite(
				hMMIO,
				reinterpret_cast<LPCSTR>(&wfex),
				sizeof(PCMWAVEFORMAT));
			if (sz != sizeof(PCMWAVEFORMAT)) {
				MmError::Throw("mmioWrite.fmt_");
			}

			mRslt = mmioAscend(
				hMMIO,
				&ckFormat,
				0);
			if (mRslt != MMSYSERR_NOERROR) {
				MmError::Throw("mmioAscend.fmt_");
			}

			/* */

			m_ckData.ckid = FCC_DATA;

			mRslt = mmioCreateChunk(
				hMMIO,
				&m_ckData,
				0);
			if (mRslt != MMSYSERR_NOERROR) {
				MmError::Throw("mmioCreateChunk.data");
			}

		} catch (...) {
			if (hMMIO != 0) {
				mmioClose(hMMIO, 0);
			}
			throw;
		}

		m_hMMIO = hMMIO;
	}

	SIZE_T Write(
		const VOID* buffer,
		SIZE_T      size)
	{
		LONG sz = mmioWrite(
			m_hMMIO,
			static_cast<LPCSTR>(buffer),
			LONG(size));
		if (sz < 0) {
			MmError::Throw("mmioWrite");
		}
		return SIZE_T(sz);
	}

	void Close()
	{
		if (m_hMMIO != 0) {
			MMRESULT mRslt = mmioAscend(
				m_hMMIO,
				&m_ckData,
				0);
			if (mRslt != MMSYSERR_NOERROR) {
				MmError::Throw("mmioAscend.data");
			}

			mRslt = mmioAscend(
				m_hMMIO,
				&m_ckRIFF,
				0);
			if (mRslt != MMSYSERR_NOERROR) {
				MmError::Throw("mmioAscend.RIFF");
			}

			mRslt = mmioClose(
				m_hMMIO,
				0);
			if (mRslt != MMSYSERR_NOERROR) {
				m_hMMIO = 0;
				MmError::Throw("mmioClose");
			}

			m_hMMIO = 0;
		}
	}

}; // WaveWriter

} // namespace QAX

